package com.yizhi.logging.controller.manage;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.yizhi.logging.domain.RequestLog;
import com.yizhi.logging.servicve.AuthResourceService;
import com.yizhi.logging.servicve.RequestLogService;
import com.yizhi.logging.util.IpUtil;
import com.yizhi.logging.util.PageInfoUtil;
import com.yizhi.logging.vo.RequestLogPageVO;
import com.yizhi.logging.vo.RequestLogViewVO;
import com.yizhi.system.application.system.remote.AccountClient;
import com.yizhi.system.application.vo.AccountVO;
import com.yizhi.util.application.domain.BizResponse;
import com.yizhi.util.application.page.PageInfo;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;

/**
 * @ClassName RequestLogController
 * @Description TODO
 * @Author shengchenglong
 * @DATE 2021/1/13 17:23
 * @Version 1.0
 */
@RestController
@RequestMapping("/manage/requestLog")
@Api("请求日志API")
@Slf4j
public class RequestLogController {

    @Autowired
    private RequestLogService requestLogService;
    @Autowired
    private AuthResourceService authResourceService;
    @Autowired
    private AccountClient accountClient;

    @GetMapping("page")
    @ApiOperation("请求日志分页列表")
    public BizResponse<PageInfo<RequestLogPageVO>> page(
            @ApiParam(value = "搜索时间段 - 起始时间 时间戳") @RequestParam(name = "startTime", required = false) LocalDateTime startTime,
            @ApiParam(value = "搜索时间段 - 结束时间 时间戳") @RequestParam(name = "endTime", required = false) LocalDateTime endTime,
            @ApiParam(value = "分页参数 - 第几页，默认1") @RequestParam(name = "pageNo", required = false, defaultValue = "1") Integer pageNo,
            @ApiParam(value = "分页参数 - 每页条数，默认10") @RequestParam(name = "pageSize", required = false, defaultValue = "10") Integer pageSize) {
        log.info("=====page request");
    	Page<RequestLog> page = requestLogService.page(startTime, endTime, pageNo, pageSize);
        List<RequestLog> list = page.getContent();

        List<RequestLogPageVO> voList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(item -> {
                AccountVO accountVO = accountClient.findById(item.getAccountId());
                voList.add(
                        RequestLogPageVO
                                .builder()
                                .id(item.getId())
                                .requestTime(item.getStartTime().format(DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss")))
                                .description(authResourceService.getNameByUrl(item.getRequestUri()))
                                .ip(item.getRequestRemoteIp() + " | " + IpUtil.getHttpCityInfo(item.getRequestRemoteIp()))
                                .millis(Duration.between(item.getStartTime(), item.getEndTime()).toMillis() + "ms")
                                .terminalType(item.getTerminalType())
                                .username(accountVO == null ? "" : accountVO.getName())
                                .build()
                );
            });
        }
        return BizResponse.ok(PageInfoUtil.of(pageNo, pageSize, Math.toIntExact(page.getTotalElements()), voList));
    }

    @GetMapping("view")
    @ApiOperation("请求日志详情")
    public BizResponse<RequestLogViewVO> view(
            @ApiParam(value = "id") @RequestParam(name = "id") String id) {
        RequestLog requestLog = requestLogService.findById(id);
        if (requestLog == null) {
            return BizResponse.fail();
        }
        RequestLogViewVO vo = RequestLogViewVO.builder()
                .requestBody(requestLog.getRequestBody())
                .requestUri(requestLog.getRequestUri())
                .requestMethod(requestLog.getRequestMethod())
                .requestParams(requestLog.getRequestParams())
                .responseBody(requestLog.getResponseBody())
                .responseStatus(requestLog.getResponseStatus())
                .endTime(requestLog.getEndTime().format(DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss")))
                .startTime(requestLog.getStartTime().format(DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss")))
                .build()
        ;
        return BizResponse.ok(vo);
    }

}
