package com.yizhi.application.dashboard.controller;

import com.yizhi.application.vo.PromptItemVO;
import com.yizhi.application.vo.PromptStatisticsVO;
import com.yizhi.application.vo.ScrollPromptItemVO;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.exam.application.feign.MyExamClient;
import com.yizhi.exam.application.vo.MyExamParameterVO;
import com.yizhi.exam.application.vo.MyExamVO;
import com.yizhi.exam.application.vo.exam.MyExamPageVO;
import com.yizhi.research.application.feign.ResearchClient;
import com.yizhi.research.application.vo.BaseModel;
import com.yizhi.research.application.vo.api.PageVo;
import com.yizhi.site.application.feign.api.SystemMailboxClients;
import com.yizhi.site.application.vo.domain.SystemMailboxParamVo;
import com.yizhi.training.application.feign.LeaveWordClient;
import com.yizhi.training.application.feign.TrainingProjectClient;
import com.yizhi.training.application.vo.api.TrainingProjectMyParamVo;
import com.yizhi.util.application.domain.Response;
import com.yizhi.exam.application.feign.ExamApiClient;
import com.yizhi.research.application.vo.domain.ResearchVo;
import com.yizhi.training.application.vo.api.TrainingProjectListVo;
import com.yizhi.site.application.feign.PublicationManageFeignClients;
import com.yizhi.site.application.vo.site.PublicationParamVO;
import com.yizhi.site.application.vo.domain.PublicationVo;
import com.yizhi.system.application.system.remote.AccountClient;
import com.yizhi.system.application.vo.AccountVO;
import com.yizhi.system.application.vo.UserInfoVO;
import com.yizhi.system.application.vo.domain.Account;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.util.HashMap;

import com.baomidou.mybatisplus.plugins.Page;

@Api(tags = "提示统计接口")
@RestController
@RequestMapping("/manage/prompt")
public class PromptStatisticsController {

    private static final Logger LOGGER = LoggerFactory.getLogger(PromptStatisticsController.class);

    @Autowired
    private ResearchClient researchClient;

    @Autowired
    private TrainingProjectClient trainingProjectClient;

    @Autowired
    private ExamApiClient examApiClient;

    @Autowired
    private SystemMailboxClients systemMailboxClients;

    @Autowired
    private MyExamClient myExamClient;

    @Autowired
    private LeaveWordClient leaveWordClient;

    @Autowired
    private PublicationManageFeignClients publicationManageFeignClients;
    
    @Autowired
    private AccountClient accountClient;

    @ApiOperation(value = "获取提示统计信息", notes = "获取总提示数量及各类提示数量统计")
    @GetMapping("/statistics")
    public Response<PromptStatisticsVO> getPromptStatistics() {
        try {
            PromptStatisticsVO statistics = new PromptStatisticsVO();

            RequestContext context = ContextHolder.get();

            // 1. 获取工作任务数量（调研表biz_type=1，已上架且处于调研期间）
            Integer workTaskCount = getWorkTaskCount(context);
            statistics.setWorkTaskCount(workTaskCount);

            // 2. 获取主题活动数量（培训项目已上架未结束）
            Integer themeActivityCount = getThemeActivityCount(context);
            statistics.setThemeActivityCount(themeActivityCount);

            // 3. 获取我要投票数量（调研表biz_type=2，已上架）
            Integer voteCount = getVoteCount(context);
            statistics.setVoteCount(voteCount);

            // 4. 获取培训测试数量（考试相关的提示数量）
            Integer trainingTestCount = getTrainingTestCount(context);
            statistics.setTrainingTestCount(trainingTestCount);

            // 5. 获取信箱提示数量（未读消息数量）
            Integer mailboxCount = getMailboxCount(context);
            statistics.setMailboxCount(mailboxCount);

            // 6. 计算总提示数量
            int totalPromptCount = workTaskCount + themeActivityCount + voteCount + trainingTestCount + mailboxCount;
            statistics.setTotalPromptCount(totalPromptCount);

            return Response.ok(statistics);
        } catch (Exception e) {
            LOGGER.error("获取提示统计信息失败", e);
            return Response.fail("500", "获取提示统计信息失败");
        }
    }
    
    @ApiOperation(value = "获取提示统计信息列表", notes = "获取提示统计信息列表，返回类型和数量的映射")
    @GetMapping("/statistics/list")
    public Response<List<PromptItemVO>> getPromptStatisticsList() {
        try {
            List<PromptItemVO> promptList = new ArrayList<>();

            RequestContext context = ContextHolder.get();

            // 1. 获取工作任务数量（调研表biz_type=1，已上架且处于调研期间）
            Integer workTaskCount = getWorkTaskCount(context);
            promptList.add(new PromptItemVO("工作任务", workTaskCount));

            // 2. 获取主题活动数量（培训项目已上架未结束）
            Integer themeActivityCount = getThemeActivityCount(context);
            promptList.add(new PromptItemVO("主题活动", themeActivityCount));

            // 3. 获取我要投票数量（调研表biz_type=2，已上架）
            Integer voteCount = getVoteCount(context);
            promptList.add(new PromptItemVO("我要投票", voteCount));

            // 4. 获取培训测试数量（考试相关的提示数量）
            Integer trainingTestCount = getTrainingTestCount(context);
            promptList.add(new PromptItemVO("培训测试", trainingTestCount));

            // 5. 检查用户是否有权限查看信箱，如果有权限则添加信箱提示数量
            if (hasMailboxPermission()) {
                Integer mailboxCount = getMailboxCount(context);
                promptList.add(new PromptItemVO("信箱", mailboxCount));
            }

            return Response.ok(promptList);
        } catch (Exception e) {
            LOGGER.error("获取提示统计信息列表失败", e);
            return Response.fail("500", "获取提示统计信息列表失败");
        }
    }

    /**
     * 获取滚动提示数据
     * @return
     */
    @ApiOperation(value = "获取滚动提示数据", notes = "获取滚动提示数据，包括活动管理、考试列表、投票管理、投稿管理、信箱管理、留言管理等数据")
    @PostMapping("/scroll/data")
    public Response<Map<String, List<ScrollPromptItemVO>>> getScrollPromptData() {
        try {
            RequestContext context = ContextHolder.get();
            Map<String, List<ScrollPromptItemVO>> resultMap = new java.util.HashMap<>();

            // 1. 活动管理（4天内创建且已上架的数据）
            List<TrainingProjectListVo> activityList = getActivityManagementData(context);
            
            // 2. 考试列表（4天内创建且已上架的数据）
            List<MyExamVO> examList = getExamListData(context);
            
            // 3. 投票管理（4天内创建且已上架的数据）
            List<ResearchVo> voteList = getVoteManagementData(context);
            
            // 4. 投稿管理（4天内创建且已上架的数据）
            List<PublicationVo> articleList = getContributionManagementData(context);
            
            // 5. 信箱管理（4天内创建且已上架的数据）
            List<SystemMailboxParamVo> mailboxList = getMailboxManagementData(context);
            
            // 6. 留言管理（4天内创建且已上架的数据）
            List<com.yizhi.training.application.vo.domain.LeaveWordVo> messageList = getMessageManagementData(context);

            // 收集所有用户名
            List<Long> userIds = new ArrayList<>();
            
            // 收集活动管理的用户名
            if (activityList != null) {
                for (TrainingProjectListVo vo : activityList) {
                    if (vo.getCreatebyId() != null) {
                        userIds.add(vo.getCreatebyId());
                    }
                }
            }
            
            // 收集考试列表的用户名
            if (examList != null) {
                for (MyExamVO vo : examList) {
                    if (vo.getCreateById() != null ) {
                        userIds.add(vo.getCreateById());
                    }
                }
            }
            
            // 收集投票管理的用户名
            if (voteList != null) {
                for (ResearchVo vo : voteList) {
                    if (vo.getCreateById() != null ) {
                        userIds.add(vo.getCreateById());
                    }
                }
            }
            
            // 收集投稿管理的用户名
            if (articleList != null) {
                for (PublicationVo vo : articleList) {
                    if (vo.getCreateById() != null) {
                        userIds.add(vo.getCreateById());
                    }
                }
            }
            

            // 批量查询用户真实姓名
            Map<Long, String> realNameMap = new HashMap<>();
            if (!userIds.isEmpty()) {
                try {

                    List<Account> accountList = accountClient.accountList(context.getCompanyId(),userIds);
                    if (accountList != null) {
                        for (Account account : accountList) {

                            if (account.getName() != null && account.getFullName() != null && !account.getFullName().trim().isEmpty()) {
                                realNameMap.put(account.getId(), account.getFullName());

                            }
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error("批量查询用户真实姓名失败", e);
                }
            }
            
            // 1. 活动管理（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> activityItems = new ArrayList<>();
            if (activityList != null) {
                for (TrainingProjectListVo vo : activityList) {
                    String realName = realNameMap.getOrDefault(vo.getCreatebyId(), vo.getCreatebyName());

                    activityItems.add(new ScrollPromptItemVO(vo.getName(), realName, vo.getCreateTime()));
                }
            }
            resultMap.put("activityManagement", activityItems);

            // 2. 考试列表（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> examItems = new ArrayList<>();
            if (examList != null) {
                for (MyExamVO vo : examList) {
                    String realName = realNameMap.getOrDefault(vo.getCreateById(), vo.getCreateByName());
                    examItems.add(new ScrollPromptItemVO(vo.getName(), realName, vo.getCreateTime()));
                }
            }
            resultMap.put("examList", examItems);

            // 3. 投票管理（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> voteItems = new ArrayList<>();
            if (voteList != null) {
                for (ResearchVo vo : voteList) {
                    String realName = realNameMap.getOrDefault(vo.getCreateById(), vo.getCreateByName());
                    voteItems.add(new ScrollPromptItemVO(vo.getName(), realName, vo.getCreateTime()));
                }
            }
            resultMap.put("voteManagement", voteItems);

            // 4. 投稿管理（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> articleItems = new ArrayList<>();
            if (articleList != null) {
                for (PublicationVo vo : articleList) {
                    String realName = realNameMap.getOrDefault(vo.getCreateById(), vo.getAuthor());
                    articleItems.add(new ScrollPromptItemVO(vo.getFileName(), realName, vo.getCreateTime()));
                }
            }
            resultMap.put("submissionManagement", articleItems);

            // 5. 信箱管理（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> mailboxItems = new ArrayList<>();
            if (mailboxList != null) {
                for (SystemMailboxParamVo vo : mailboxList) {
                    mailboxItems.add(new ScrollPromptItemVO(vo.getFileName(), vo.getContactName(), vo.getSubmitTime()));
                }
            }
            resultMap.put("mailboxManagement", mailboxItems);

            // 6. 留言管理（4天内创建且已上架的数据）
            List<ScrollPromptItemVO> messageItems = new ArrayList<>();
            if (messageList != null) {
                for (com.yizhi.training.application.vo.domain.LeaveWordVo vo : messageList) {
                    messageItems.add(new ScrollPromptItemVO(vo.getName(), vo.getFullName(), vo.getCreateTime()));
                }
            }
            resultMap.put("messageManagement", messageItems);

            return Response.ok(resultMap);
        } catch (Exception e) {
            LOGGER.error("获取滚动提示数据失败", e);
            return Response.fail("500", "获取滚动提示数据失败");
        }
    }

    /**
     * 获取活动管理数据（4天内创建且已上架的数据）
     * @param context
     * @return
     */
    private List<TrainingProjectListVo> getActivityManagementData(RequestContext context) {
        try {
            TrainingProjectMyParamVo paramVo = new TrainingProjectMyParamVo();
            
            com.yizhi.training.application.model.BaseModel<TrainingProjectMyParamVo> model = 
                    new com.yizhi.training.application.model.BaseModel<>();
            model.setContext(context);
            model.setDate(new Date());
            model.setObj(paramVo);
            
            // 使用新的接口方法获取4天内创建的活动管理数据
            List<TrainingProjectListVo> response = trainingProjectClient.apiActivityManagementForScroll(model);
            
            if (response != null ) {
                return response;
            }
            
            return new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取活动管理数据失败", e);
            return new ArrayList<>();
        }
    }
    
    /**
     * 获取考试列表数据（4天内创建且已上架的数据）
     * @param context
     * @return
     */
    private List<MyExamVO> getExamListData(RequestContext context) {
        try {
            // 使用我的考试接口获取未完成的考试列表
            MyExamParameterVO param = new MyExamParameterVO();
            param.setSiteId(context.getSiteId());
            param.setAccountId(context.getAccountId());
            // 设置状态为未完成(进行中)的考试
            param.setState(1);
            List<Long> ids = context.getRelationIds();
            if (null != ids) {
                param.setIds(ids);
            } else {
                ids = new ArrayList<Long>();
                ids.add(context.getAccountId());
                param.setIds(ids);
            }

            List<MyExamVO> examData = myExamClient.getMyExamListFour(param);
            
            if (examData != null ) {
                return examData;
            }
            
            return new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取考试列表数据失败", e);
            return new ArrayList<>();
        }
    }

    /**
     * 获取投票管理数据（4天内创建且已上架的数据）
     * @param context
     * @return
     */
    private List<ResearchVo> getVoteManagementData(RequestContext context) {
        try {
            PageVo pageVo = new PageVo();

            pageVo.setAccountId(context.getAccountId());
            pageVo.setDate(new Date());
            pageVo.setBizType(2); // 投票类型
            pageVo.setState(1); // 已上架状态
            
            com.yizhi.research.application.vo.BaseModel<PageVo> model = new com.yizhi.research.application.vo.BaseModel<>();
            model.setContext(context);
            model.setDate(new Date());
            model.setObj(pageVo);
            
            // 使用新的接口方法获取4天内创建的投票管理数据
            List<ResearchVo> response = researchClient.apiVoteManagementForScroll(model);
            
            if (response != null) {
                return response;
            }
            
            return new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取投票管理数据失败", e);
            return new ArrayList<>();
        }
    }

    /**
     * 获取投稿管理数据（待审核的数据）
     * @param context
     * @return
     */
    private List<PublicationVo> getContributionManagementData(RequestContext context) {
        try {
            // 使用新的接口方法获取4天内创建的投稿管理数据
            List<PublicationVo> response = publicationManageFeignClients.selectContributionManagementForScroll();
            
            if (response != null) {
                return response;
            }
            
            return new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取投稿管理数据失败", e);
            return new ArrayList<>();
        }
    }

    /**
     * 获取信箱管理数据（未查看的信箱数据，需要考虑权限）
     * @param context
     * @return
     */
    private List<SystemMailboxParamVo> getMailboxManagementData(RequestContext context) {
        try {
            // 检查用户是否有权限查看信箱
            if (!hasMailboxPermission()) {
                return new ArrayList<>();
            }
            

            List<SystemMailboxParamVo> dataList = systemMailboxClients.selectMailboxManagementForScroll();
            
            return dataList != null ? dataList : new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取信箱管理数据失败", e);
            return new ArrayList<>();
        }
    }

    /**
     * 获取留言管理数据（待接收的留言数据）
     * @param context
     * @return
     */
    private List<com.yizhi.training.application.vo.domain.LeaveWordVo> getMessageManagementData(RequestContext context) {
        try {
            com.yizhi.training.application.vo.domain.LeaveWordParamVo paramVo = new com.yizhi.training.application.vo.domain.LeaveWordParamVo();
            paramVo.setPageNo(1);
            paramVo.setPageSize(Integer.MAX_VALUE);
            paramVo.setState(1);
            
            // 获取留言数据
            Page<com.yizhi.training.application.vo.domain.LeaveWordVo> response = leaveWordClient.page(paramVo);
            
            if (response != null && response.getRecords() != null) {
                return response.getRecords();
            }
            
            return new ArrayList<>();
        } catch (Exception e) {
            LOGGER.error("获取留言管理数据失败", e);
            return new ArrayList<>();
        }
    }

    /**
     * 获取工作任务数量
     * @param context
     * @return
     */
    private Integer getWorkTaskCount(RequestContext context) {
        try {
            PageVo pageVo = new PageVo();
            pageVo.setAccountId(context.getAccountId());
            pageVo.setDate(new Date());
            pageVo.setPageNo(1);
            pageVo.setPageSize(1);
            pageVo.setBizType(1); // 调研类型
            pageVo.setState(2); // 进行中的任务

            BaseModel<PageVo> model = new BaseModel<>();
            model.setDate(new Date());
            model.setObj(pageVo);
            model.setContext(context);

            // 获取调研列表，只取数量
            Page<ResearchVo> response = researchClient.apiListPage(model);
            if (response != null ) {

                return response.getTotal();
            }
            return 0;
        } catch (Exception e) {
            LOGGER.error("获取工作任务数量失败", e);
            return 0;
        }
    }

    /**
     * 获取主题活动数量
     * @param context
     * @return
     */
    private Integer getThemeActivityCount(RequestContext context) {
        try {
            TrainingProjectMyParamVo paramVo = new TrainingProjectMyParamVo();
            paramVo.setPageNo(1);
            paramVo.setPageSize(1);
            paramVo.setType(2); // 进行中的活动

            com.yizhi.training.application.model.BaseModel<TrainingProjectMyParamVo> model =
                    new com.yizhi.training.application.model.BaseModel<>();
            model.setContext(context);
            model.setDate(new Date());
            model.setObj(paramVo);

            // 获取培训项目列表，只取数量
            Page<TrainingProjectListVo> response = trainingProjectClient.apiMyPageList(model);
            if (response != null ) {
                return response.getTotal();
            }
            return 0;
        } catch (Exception e) {
            LOGGER.error("获取主题活动数量失败", e);
            return 0;
        }
    }

    /**
     * 获取投票数量
     * @param context
     * @return
     */
    private Integer getVoteCount(RequestContext context) {
        try {
            PageVo pageVo = new PageVo();
            pageVo.setAccountId(context.getAccountId());
            pageVo.setDate(new Date());
            pageVo.setPageNo(1);
            pageVo.setPageSize(1);
            pageVo.setBizType(2); // 投票类型
            pageVo.setState(2); // 进行中的投票

            BaseModel<PageVo> model = new BaseModel<>();
            model.setDate(new Date());
            model.setObj(pageVo);
            model.setContext(context);

            // 获取投票列表，只取数量
            Page<ResearchVo> response = researchClient.apiListPage(model);
            if (response != null ) {
                return response.getTotal();
            }
            return 0;
        } catch (Exception e) {
            LOGGER.error("获取投票数量失败", e);
            return 0;
        }
    }

    /**
     * 获取培训测试数量
     * @param context
     * @return
     */
    private Integer getTrainingTestCount(RequestContext context) {
        try {
            // 使用我的考试接口获取未完成的考试列表
            MyExamParameterVO param = new MyExamParameterVO();
            param.setSiteId(context.getSiteId());
            param.setAccountId(context.getAccountId());
            // 设置状态为未完成(进行中)的考试
            param.setState(1);
            param.setPageNo(1);
            param.setPageSize(1); // 设置一个较大的数值以获取所有未完成的考试
            List<Long> ids = context.getRelationIds();
            if (null != ids) {
                param.setIds(ids);
            } else {
                ids = new ArrayList<Long>();
                ids.add(context.getAccountId());
                param.setIds(ids);
            }
            MyExamPageVO examData = myExamClient.getMyExamListNew(param);
            // 返回未完成考试的数量
            return examData != null  ? examData.getPageTotal() : 0;
        } catch (Exception e) {
            LOGGER.error("获取培训测试数量失败", e);
            return 0;
        }
    }

    /**
     * 获取信箱提示数量（未读消息数量）
     * @param context
     * @return
     */
    private Integer getMailboxCount(RequestContext context) {
        try {
            // 获取未读消息数量
            Page<SystemMailboxParamVo> response = systemMailboxClients.selectMyPage(1,1, 2);
            if (response != null ) {
                return response.getTotal();
            }
            return 0;
        } catch (Exception e) {
            LOGGER.error("获取信箱提示数量失败", e);
            return 0;
        }
    }
    /**
     * 检查用户是否有权限查看信箱
     * @return
     */
    private boolean hasMailboxPermission() {
        try {
            // 调用SystemMailboxClients的hasPermission方法检查权限
            return systemMailboxClients.hasPermission();
        } catch (Exception e) {
            LOGGER.error("检查用户信箱权限失败", e);
            return false;
        }
    }
    
}