package com.yizhi.application.offlineCourse;

import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.core.application.enums.InternationalEnums;
import com.yizhi.course.application.feign.OfflineCourseReportClient;
import com.yizhi.course.application.vo.OfflineRecordeVO;
import com.yizhi.course.application.vo.ReportOfflineCourseDataStatisticsVo;
import com.yizhi.course.application.vo.ReportOfflineCourseStatisticsDetailVo;
import com.yizhi.util.application.domain.Response;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.text.ParseException;
import java.util.*;

/**
 * 线下课程报表
 *
 * @Author: shengchenglong
 * @Date: 2018/10/22 15:06
 */
@RestController
@RequestMapping("/manage/offlineCourseReport")
public class OfflineCourseReportController {

    private static final Logger LOGGER = LoggerFactory.getLogger(OfflineCourseReportController.class);

    @Autowired
    private OfflineCourseReportClient offlineCourseReportClient;

    /**
     * 数据统计
     *
     * @param param
     * @return
     */
    @ApiOperation(value = "数据统计", response = ReportOfflineCourseDataStatisticsVo.class)
    @PostMapping("/data/statistics")
    public Response<List<ReportOfflineCourseDataStatisticsVo>> dataStatistics(
            @ApiParam("参数形式如：{\"startTime\": \"2018-01-10\", \"endTime\": \"2018-01-10\"}")
            @RequestBody Map<String, Date> param
    ) {
        try {
            Date startTime = param.get("startTime");
            Date endTime = param.get("endTime");

            LOGGER.info("请求参数 :startTime={},endTime={}",startTime,endTime);
            List<ReportOfflineCourseDataStatisticsVo> vos = offlineCourseReportClient.dataStatistics(startTime.getTime(), endTime.getTime());
            return Response.ok(vos);
        } catch (Exception e) {
            LOGGER.error("", e);
            return Response.fail();
        }
    }

    /**
     * 统计详情
     *
     * @param param
     * @return
     */
    @ApiOperation(value = "统计详情", response = ReportOfflineCourseStatisticsDetailVo.class)
    @PostMapping("/statistics/detail")
    public Response<Page<ReportOfflineCourseStatisticsDetailVo>> statisticsDetail(
            @ApiParam("参数形式如：{\"startTime\": \"2018-01-10\", \"endTime\": \"2018-01-10\", \"name\" : \"course\"}")
            @RequestBody Map<String, String> param,
            @ApiParam("每页条数，默认10") @RequestParam(required = false, defaultValue = "10", name = "pageSize") Integer pageSize,
            @ApiParam("目标页码，默认1") @RequestParam(required = false, defaultValue = "1", name = "pageNo") Integer pageNo
    ) {
        try {
            String startTimeStr = param.get("startTime");
            String entTimeStr = param.get("endTime");

            LOGGER.info("请求参数={}",param);

            Date startTimeDate = null;
            Date endTimeDate = null;
            String[] pattern = new String[]{"yyyy-MM-dd"};
            try {
                startTimeDate = DateUtils.truncate(DateUtils.parseDate(startTimeStr, pattern), Calendar.DATE);
                endTimeDate = DateUtils.truncate(DateUtils.parseDate(entTimeStr, pattern), Calendar.DATE);
            } catch (ParseException e) {
                LOGGER.info("异常{}",e);
            }
            if (startTimeStr == null || entTimeStr == null) {
                return Response.fail("4001", InternationalEnums.OFFLINECOURSEREPORTCONTROLLER.getCode());
            }

            String name = param.get("name").trim();

            Page<ReportOfflineCourseStatisticsDetailVo> vos = offlineCourseReportClient.statisticsDetail(name, startTimeDate.getTime(), endTimeDate.getTime(), pageSize, pageNo);
            if (vos != null) {
                Map<String, Integer> pageMap = new HashMap<>();
                pageMap.put("pageNo", pageNo);
                pageMap.put("pageSize", pageSize);
                pageMap.put("pageTotal", vos.getTotal());
                Response response = Response.ok(vos.getRecords(), pageMap);
                return response;
            }
            return Response.ok();
        } catch (Exception e) {
            LOGGER.error("", e);
            return Response.fail();
        }
    }

    @GetMapping("/queryDetail/list")
    @ApiOperation(value = "根据线下课程id获取报表记录详情(分页查询)", notes = "", response = OfflineRecordeVO.class)
    public Response<Page<OfflineRecordeVO>> queryDetailList(
            @ApiParam(value = "offlineCourseId", required = true) @RequestParam(name = "offlineCourseId") Long offlineCourseId,
            @ApiParam(name = "pageNo", value = "跳转页数,默认第一页", required = true) @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
            @ApiParam(name = "pageSize", value = "每页条数,默认20条", required = true) @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize
    ) {

        Page<OfflineRecordeVO> page = offlineCourseReportClient.queryDetailList(offlineCourseId, pageNo, pageSize);
        return Response.ok(page);
    }

}
