package com.yizhi.application.album;

import java.util.Date;
import java.util.List;

import com.yizhi.album.application.feign.AlbumClient;
import com.yizhi.album.application.vo.AlbumVo;
import com.yizhi.album.application.vo.domain.Album;
import com.yizhi.album.application.vo.domain.AlbumVisibleRange;
import com.yizhi.core.application.enums.InternationalEnums;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.lecturer.application.enums.DeleteFlag;
import com.yizhi.site.application.feign.PortalManagePCFeignClients;
import com.yizhi.site.application.vo.domain.ProtalPlateVo;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.enums.i18n.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import com.baomidou.mybatisplus.plugins.Page;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

@Api(tags = "管理端-专辑", description = "专辑接口")
@RestController
@RequestMapping("/manage/album")
public class AlbumController {

	private Logger logger = LoggerFactory.getLogger(AlbumController.class);

	@Autowired
	private AlbumClient albumClient;

	@Autowired
	private PortalManagePCFeignClients portalManagePCFeignClient;

	@GetMapping(value = "/list")
	@ApiOperation(value = "专辑分页列表", notes = "返回带分页的专辑列表", response = Album.class)
	public Response<Page<Album>> list(
			@ApiParam(name = "name", value = "名称或者关键字 默认不传", required = false) @RequestParam(name = "name", required = false) String name,
			@ApiParam(name = "alClassifyId", value = "分类id，未分类传0 默认不传", required = false) @RequestParam(name = "alClassifyId", required = false) Long alClassifyId,
			@ApiParam(name = "status", value = "0 未上架 1 已上架 2 草稿 全部不传", required = false) @RequestParam(name = "status", required = false) Integer status,
			@ApiParam(name = "overed", value = "0 连载中 1 已完结 全部不传", required = false) @RequestParam(name = "overed", required = false) Integer overed,
			@ApiParam(name = "pageNo", value = "跳转页数,默认第一页", required = true) @RequestParam(name = "pageNo", defaultValue = "1") Integer pageNo,
			@ApiParam(name = "pageSize", value = "每页条数,默认10条", required = true) @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
		RequestContext context = ContextHolder.get();
		Page<Album> page = albumClient.list(name, alClassifyId, status, overed, context.getCompanyId(),
				context.getSiteId(), pageNo, pageSize);
		return Response.ok(page);
	}

	@GetMapping("/get")
	@ApiOperation(value = "专辑查看", notes = "专辑查看",response = Album.class)
	public Response<Album> get(
			@ApiParam(name = "id", value = "专辑id", required = true) @RequestParam(name = "id", required = true) Long id) {
		Album album = albumClient.get(id);
		return Response.ok(album);
	}

	@PostMapping("/save")
	@ApiOperation(value = "新增专辑第一步", notes = "点击下一步保存",response = AlbumVo.class)
	public Response<String> save(@ApiParam(name = "albumVo", value = "需要传入name，"
			+ "       classifyId，description，tags", required = true) @RequestBody AlbumVo albumVo) {
		RequestContext requestContext = ContextHolder.get();
		albumVo.setCreateById(requestContext.getAccountId());
		albumVo.setCreateByName(requestContext.getAccountName());
		albumVo.setCompanyId(requestContext.getCompanyId());
		albumVo.setSiteId(requestContext.getSiteId());
		albumVo.setCreateTime(new Date());
		AlbumVo vo = albumClient.save(albumVo);
		if (vo!=null) {
			return Response.ok(vo);
		}else {
			return Response.fail(InternationalEnums.ALBUMCONTROLLER1.getCode());
		}
		
	}

	@GetMapping("/up")
	@ApiOperation(value = "上架", notes = "上架操作")
	public Response<String> up(@ApiParam(value = "专辑id", required = true) @RequestParam(name = "id") Long id,@ApiParam(value = "保存类型", required = true)@RequestParam(name = "type")Integer type) {
		try {
			Integer f = albumClient.up2(id,type);
			if (f==1) {
				return Response.ok();
			} else if (f==2){
				return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode().toString(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
			}else {
				return Response.fail("4001", InternationalEnums.ALBUMCONTROLLER2.getCode());
			}
		} catch (Exception e) {
			return Response.fail("", e.getLocalizedMessage());
		}
	}

	@GetMapping("/down")
	@ApiOperation(value = "下架", notes = "下架操作")
	public Response<String> down(@ApiParam(value = "专辑id", required = true) @RequestParam(name = "id") Long id) {

		//新增首页模块关联查询
		ProtalPlateVo protalPlate = new ProtalPlateVo();
		protalPlate.setLinkContent(id);
		protalPlate.setState(DeleteFlag.YES.ordinal());
		ProtalPlateVo plate = portalManagePCFeignClient.getPlate(protalPlate);
		if (plate != null) {
			logger.info("专辑下架失败;albumId={},专辑已经关联到首页配置模块；不能下架",id);
			return Response.fail(InternationalEnums.OFFLINECOURSECONTROLLER5.getCode());
		}
		Boolean f = albumClient.down(id);
		if (f) {
			return Response.ok();
		} else {
			logger.info("专辑下架失败;albumId={},专辑下架失败.",id);
			return Response.fail(InternationalEnums.OFFLINECOURSECONTROLLER5.getCode(),InternationalEnums.OFFLINECOURSECONTROLLER5.getName());
		}
	}

	@PostMapping("/update")
	@ApiOperation(value = "新增专辑第三步", notes = "点击下一步保存")
	public Response<String> update(@ApiParam(name = "albumVo", value = "需要传入scope,"
			+ "commentEnable,documentEnable,overed,display", required = true) @RequestBody AlbumVo albumVo) {
		RequestContext requestContext = ContextHolder.get();
		albumVo.setUpdateById(requestContext.getAccountId());
		albumVo.setUpdateByName(requestContext.getAccountName());
		albumVo.setUpdateTime(new Date());
		Boolean b = albumClient.update(albumVo);
		if (b) {
			return Response.ok();
		}else {
            return Response.fail(InternationalEnums.ALBUMCONTROLLER1.getCode());
		}
	}
	
	@PostMapping("/authorize/insert")
	@ApiOperation(value = "专辑可见范围新增")
	public Response<String> insertReportVisibleRanges(
			@ApiParam(name = "专辑可见范围", value = "需传albumId、relationId、type") @RequestBody List<AlbumVisibleRange> albumVisableRanges) {
		try {
			boolean b = albumClient.insertAlbumVisibleRanges(albumVisableRanges);
			if (b) {
				return Response.ok();
			} else {
				return Response.fail(Constants.MSG_BIZ_FAIL);
			}
		} catch (Exception e) {
			e.printStackTrace();
			return Response.fail(Constants.MSG_BIZ_FAIL);
		}
	}
	
	@GetMapping("/VisibleRange")
	@ApiOperation(value = "指定专辑可见范围", response = AlbumVisibleRange.class)
	public Response<List<AlbumVisibleRange>> VisibleRange(
			@ApiParam(name = "albumId", value = "专辑id") @RequestParam("albumId") Long albumId) {
		try {
			List<AlbumVisibleRange> list = albumClient.VisibleRange(albumId);
			return Response.ok(list);
		} catch (Exception e) {
			e.printStackTrace();
			return Response.fail(Constants.MSG_BIZ_FAIL);
		}
	}
    @GetMapping("/list/export")
    @ApiOperation(value = "专辑课程清单导出")
    public Response<String> listExport(@ApiParam(name = "albumId", value = "专辑id ", required = false) @RequestParam(name = "albumId", required = false) Long albumId) {

        return Response.ok(albumClient.activityExport(albumId).get("result").toString());
    }
    
    @GetMapping("/list/search")
    @ApiOperation(value = "专辑列表查询",response = Album.class)
    public Response searchList(@ApiParam(name = "name", value = "专辑名称或关键字", required = false)@RequestParam(name = "name",required = false) String name,@ApiParam(name = "ids", value = "已关联的专辑id", required = false)@RequestParam(name = "ids",required = false) List<Long> ids) {
    	RequestContext context = ContextHolder.get();
        return Response.ok(albumClient.searchList(name,ids, context.getCompanyId(), context.getSiteId()));
    }
    
	@PostMapping("/delete")
	@ApiOperation(value = "专辑删除")
	public Response<String> delete(
			@ApiParam(name = "ids", value = "选中的专辑id") @RequestBody List<Long> ids) {
		try {
			Integer b = albumClient.delete(ids);
			if (b.equals(0)) {
				return Response.ok();
			} else  {
				return Response.fail(InternationalEnums.ALBUMCONTROLLER3.getCode());
			}
		} catch (Exception e) {
			e.printStackTrace();
			return Response.fail(Constants.MSG_BIZ_FAIL);
		}
	}
}
