/**
 *
 */
package com.yizhi.core.application.cache;

import org.apache.commons.collections.CollectionUtils;
import org.checkerframework.checker.units.qual.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author scotthu
 * @date 2018年3月8日
 */
@Component
public class RedisCache {

    @Autowired
    @Qualifier("RedisTemplate")
    private RedisTemplate redisTemplate;

    private Logger log = LoggerFactory.getLogger(RedisCache.class);

    /**
     * 判断key是否存在
     *
     * @param key
     * @return
     */
    public boolean exists(final String key) {
        return redisTemplate.hasKey(key);
    }

    /**
     * 条件缓存设置 在键不存在时才能设置
     *
     * @param key   键
     * @param value 缓存值
     * @return true成功 false失败
     */
    public boolean setIfAbsent(String key, Object value) {
        try {
            
            return redisTemplate.opsForValue().setIfAbsent(key, value);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }


    /**
     * 条件缓存设置 在键不存在时才能设置
     *
     * @param key   键
     * @param value 缓存值
     * @return true成功 false失败
     */
    public boolean setIfAbsent(String key, Object value, long timeout, TimeUnit unit) {
        try {

            return redisTemplate.opsForValue().setIfAbsent(key, value,timeout,unit);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }



    /**
     * 普通缓存设置
     *
     * @param key   键
     * @param value 缓存值
     * @return true成功 false失败
     */
    public boolean set(String key, Object value) {
        try {
            redisTemplate.opsForValue().set(key, value);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 普通缓存设置
     *
     * @param key   键
     * @param value 缓存值
     * @param time  过期时间
     * @return true成功 false失败
     */
    public boolean set(String key, Object value, long time) {
        try {
            if (time > 0)
                redisTemplate.opsForValue().set(key, value, time, TimeUnit.SECONDS);
            else
                set(key, value);

            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }


    /**
     * 设置hash条目
     *
     * @param key   键
     * @param item  条目hash键
     * @param value 条目值
     * @return true成功 false失败
     */
    public boolean hset(String key, String item, Object value) {
        try {
            redisTemplate.opsForHash().put(key, item, value);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }
    /**
     * 判断 key是否存在
     *
     * @param key 键
     * @return true 存在 false不存在
     */
    public Boolean hasKey(String key) {
        try {
            return redisTemplate.hasKey(key);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 查看 hash item 是否存在
     * @param key
     * @param item
     * @return
     */
    public boolean hExisted(String key, String item) {
        try {
            return redisTemplate.opsForHash().hasKey(key, item);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 获取 hash 的所有 item
     * @param key
     * @return
     */
    public Set<Object> hKeys(String key) {
        try {
            return redisTemplate.opsForHash().keys(key);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 查看 hash items 是否全部存在
     * @param key
     * @param items
     * @return
     */
    public boolean hAllExisted(String key, String[] items) {
        try {
            boolean flag = true;
            for (Object i : items) {
                if (!redisTemplate.opsForHash().hasKey(key, i)) {
                    flag = false;
                    break;
                }
            }
            return flag;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 设置hash条目  若条目已存在则返回失败
     *
     * @param key   键
     * @param item  条目hash键
     * @param value 条目值
     * @return true成功 false失败
     */
    public boolean hsetIfAbsent(String key, String item, Object value) {
        try {
            redisTemplate.opsForHash().putIfAbsent(key, item, value);
            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 设置hash条目
     *
     * @param key   键
     * @param item  条目hash键
     * @param value 条目值
     * @param time  过期时间
     * @return true成功 false失败
     */
    public boolean hset(String key, String item, Object value, long time) {
        try {
            redisTemplate.opsForHash().put(key, item, value);
            if (time > 0)
                expire(key, time);

            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 获得hashset条目值
     *
     * @param key  键
     * @param item 条目键
     * @return 条目值
     */
    public Object hget(String key, String item) {
        Object o = redisTemplate.opsForHash().get(key, item);
        if (o == null) {
            return null;
        }
        if ("{}".equals(o.toString())) {
            return null;
        }
        return o;
    }

    public Map<Object, Object> hmget(String key) {
        return redisTemplate.opsForHash().entries(key);
    }

    public void hdel(String key, Object[] items) {
        String[] itemStrs = new String[items.length];
        for (int i = 0; i < items.length; i++) {
            itemStrs[i] = String.valueOf(items[i]);
        }
        redisTemplate.opsForHash().delete(key, itemStrs);
    }

    /**
     * 设置缓存项的过期时间
     *
     * @param key  键
     * @param time 时间(秒)
     * @return true成功 false失败
     */
    public boolean expire(String key, long time) {
        try {
            if (time > 0)
                redisTemplate.expire(key, time, TimeUnit.SECONDS);

            return true;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return false;
        }
    }

    /**
     * 获取redis的值
     *
     * @param key
     * @return
     */
    public Object get(final String key) {
        return redisTemplate.opsForValue().get(key);
    }

    /**
     * 根据key精确匹配删除
     *
     * @param key
     */
    public void delete(String... key) {
        if (null == key) {
            return;
        } else if (key.length == 1) {
            redisTemplate.delete(key[0]);
        } else if (key.length > 1) {
            redisTemplate.delete(Arrays.asList(key));
        }
    }

    /**
     * 根据prex模糊匹配删除
     * @param prex
     */
    public void deleteByPrex(String prex) {
    	prex = prex+"**";
        Set<String> keys = redisTemplate.keys(prex);
        if (CollectionUtils.isNotEmpty(keys)) {
            redisTemplate.delete(keys);
        }
    }

}
