package com.yizhi.core.application.cache.distributedlock.impl;

import com.yizhi.core.application.cache.distributedlock.AbstractDistributedLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisCluster;
import redis.clients.jedis.commands.JedisCommands;
import redis.clients.jedis.params.SetParams;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * @Author: XieHaijun
 * @Description:
 * @Date: Created in 10:12 2019/4/16
 * @Modified By
 */
@Component
public class RedisDistributedLock extends AbstractDistributedLock {

    private final Logger logger = LoggerFactory.getLogger(RedisDistributedLock.class);

    private RedisTemplate<Object, Object> redisTemplate;

    private ThreadLocal<String> lockFlag = new ThreadLocal<String>();

    public static final String UNLOCK_LUA;

    static {
        StringBuilder sb = new StringBuilder();
        sb.append("if redis.call(\"get\",KEYS[1]) == ARGV[1] ");
        sb.append("then ");
        sb.append("    return redis.call(\"del\",KEYS[1]) ");
        sb.append("else ");
        sb.append("    return 0 ");
        sb.append("end ");
        UNLOCK_LUA = sb.toString();
    }

    public RedisDistributedLock(RedisTemplate<Object, Object> redisTemplate) {
        super();
        this.redisTemplate = redisTemplate;
    }

    /**
     * 加锁
     *
     * @param key         加锁的key
     * @param expire      过期时间(毫秒，对于大频发要控制好时间,预估业务处理的时间)
     * @param retry       重试次数（毫秒，结合sleepMillis随眠时间间隔，可以控制重试间隔最大时间)
     * @param sleepMillis 每次重试睡眠等待时间，毫秒
     * @return 返回false表示加锁失败，可以考虑结束业务（正常情况都是返回true)
     */
    @Override
    public boolean lock(String key, long expire, int retry, long sleepMillis) {
        boolean result = setRedis(key, expire);
        //logger.info("key={}首次加锁的状态={}",key,result);
        // 如果获取锁失败，按照传入的重试次数进行重试
        while ((!result) && retry-- > 0) {
            try {
                //logger.info("{}次进入循环key={}首次加锁的状态={}",retryTimes, key,result);
                Thread.sleep(sleepMillis);
            } catch (InterruptedException e) {
                //logger.info("加锁发生异常{}",e);
                return false;
            }
            result = setRedis(key, expire);
        }
        return result;
    }

    private boolean setRedis(String key, long expire) {
        try {
            String result = redisTemplate.execute(new RedisCallback<String>() {
                @Override
                public String doInRedis(RedisConnection connection) throws DataAccessException {
                    JedisCommands commands = (JedisCommands) connection.getNativeConnection();
                    String uuid = UUID.randomUUID().toString();
                    lockFlag.set(uuid);
                    // set(String key, String value, String nxxx, String expx, int time)，这个set()方法一共有五个形参：
                    //  第一个为key，我们使用key来当锁，因为key是唯一的。
                    //  第二个为value，我们传的是requestId,在解锁的时候就可以有依据。requestId可以使用UUID.randomUUID().toString()方法生成。
                    // 第三个为nxxx，这个参数我们填的是NX，意思是SET IF NOT EXIST，即当key不存在时，我们进行set操作；若key已经存在，则不做任何操作；
                    // 第四个为expx，这个参数我们传的是PX，意思是我们要给这个key加一个过期的设置，具体时间由第五个参数决定。expx的值只能取EX或者PX，代表数据过期时间的单位，EX代表秒，PX代表毫秒。
                    // 第五个为time，与第四个参数相呼应，代表key的过期时间。
                    SetParams setParams = SetParams.setParams().nx().px(expire);
//                    return commands.set(key, uuid, "NX", "PX", expire);
                    return commands.set(key, uuid, setParams);
                }
            });
            //logger.info("{}底层设置到redis的结果={}",key, result);
            return Objects.nonNull(result) && "OK".equalsIgnoreCase(result); // 否则返回结果=null
        } catch (Exception e) {
            logger.info(key + "底层设置到redis发生异常{}", e);
        }
        return false;
    }

    @Override
    public boolean releaseLock(String key) {
        // 释放锁的时候，有可能因为持锁之后方法执行时间大于锁的有效期，此时有可能已经被另外一个线程持有锁，所以不能直接删除
        try {
            List<String> keys = new ArrayList<>();
            keys.add(key);
            List<String> args = new ArrayList();
            args.add(lockFlag.get());

            // 使用lua脚本删除redis中匹配value的key，可以避免由于方法执行时间过长而redis锁自动过期失效的时候误删其他线程的锁
            // spring自带的执行脚本方法中，集群模式直接抛出不支持执行脚本的异常，所以只能拿到原redis的connection来执行脚本
            Long result = redisTemplate.execute(new RedisCallback<Long>() {
                @Override
                public Long doInRedis(RedisConnection connection) throws DataAccessException {
                    Object nativeConnection = connection.getNativeConnection();
                    // 集群模式和单机模式虽然执行脚本的方法一样，但是没有共同的接口，所以只能分开执行
                    // 集群模式
                    if (nativeConnection instanceof JedisCluster) {
                        return (Long) ((JedisCluster) nativeConnection).eval(UNLOCK_LUA, keys, args);
                    }
                    // 单机模式
                    else if (nativeConnection instanceof Jedis) {
                        return (Long) ((Jedis) nativeConnection).eval(UNLOCK_LUA, keys, args);
                    }
                    return 0L;
                }
            });
            //logger.info("{}底层redis解锁的结果={}",key,result);
            return result != null && result > 0; // 返回1表示删除成功
        } catch (Exception e) {
            logger.info(key + "底层解锁发生异常{}", e);
        }
        return false;
    }
}
