package com.yizhi.core.application.file.imp;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.URL;
import java.util.List;

import com.yizhi.core.application.file.constant.FileConstant;
import com.yizhi.core.application.file.domain.FileInfo;
import com.yizhi.core.application.log.TaskLogEvent;
import com.yizhi.core.application.task.AbstractTaskHandler;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.yizhi.core.application.file.domain.Template;
import com.yizhi.core.application.file.util.ProxyCreateUtil;
import com.yizhi.core.application.file.util.TemplateUtil;

/**
 * 抽象导入类
 * @author scotthu
 *
 * @date 2018年1月30日
 */
public abstract class AbstractImporter extends AbstractTaskHandler {
	
	protected String fileUrl;
	protected String savePath= FileConstant.SAVE_PATH;
	protected String fileName;
	protected String fileType;
	protected ImportCallBack callBack;
	protected Template template;
	protected Class domainClazz;
	
	private static Integer httpProxyPort;	
	private static String httpProxyUrl;
	
	private static final Logger logger = LoggerFactory.getLogger(AbstractImporter.class);
	
	public AbstractImporter(String fileUrl,String fileName,String savePath,ImportCallBack callBack, Template template,Class domainClazz) {
		this.fileUrl = fileUrl;
	//	this.fileName = DateUtil.getYMDHMSS()+fileName.substring(fileName.lastIndexOf("."), fileName.length());	
		this.fileName = fileName;
		this.callBack = callBack;
		this.template = template;
		this.domainClazz = domainClazz;
	}
	
	/**
	 * 解析文件类型
	 * @param fileName
	 * @return
	 */
	private static String getFileType(String fileName) {
		if(StringUtils.isEmpty(fileName))
			return "";
		
		int index = fileName.lastIndexOf(".");
		if(index <= 0)
			return "";
		
		return fileName.substring(index + 1);
	}
	/**
	 * 工厂方法  封装创建具体导入类的细节
	 * @param fileType
	 * @return
	 */
	public static <T> AbstractImporter build(String fileUrl,String savePath,String fileName,ImportCallBack callBack, Template template,Class domainClazz){
		AbstractImporter abstractImporter = null;
		String fileType = getFileType(fileName);
		if ("xls".equals(fileType) || "xlsx".equals(fileType)) {
			abstractImporter = new ExcelImporter(fileUrl,savePath,fileName,callBack, template,domainClazz);
		}else if("csv".equals(fileType)){
			abstractImporter = new CsvImporter(fileUrl,savePath,fileName,callBack, template,domainClazz);
		}
		return abstractImporter;
	}

	/**
	 * 工厂方法  封装创建具体导入类的细节
	 * @param fileType
	 * @return
	 */
	public static <T> AbstractImporter build(String fileUrl,String savePath,String fileName,ImportCallBack callBack, Template template,Class domainClazz,
			Integer proxyPort, String proxyUrl){
		AbstractImporter abstractImporter = null;
		String fileType = getFileType(fileName);
		if ("xls".equals(fileType) || "xlsx".equals(fileType)) {
			abstractImporter = new ExcelImporter(fileUrl,savePath,fileName,callBack, template,domainClazz);
		}else if("csv".equals(fileType)){
			abstractImporter = new CsvImporter(fileUrl,savePath,fileName,callBack, template,domainClazz);
		}
		httpProxyPort = proxyPort;
		httpProxyUrl = proxyUrl;
		
		return abstractImporter;
	}
	
	/**
	 * 导入处理
	 */
	public  void handle(){  
		publishTaskEvent(TaskLogEvent.working(taskContext));
		List<String> result = TemplateUtil.check(template.getColumns(), domainClazz);
		if(result.size() > 0) {
    		publishTaskEvent(TaskLogEvent.fail(taskContext,result.toString()));
			return;
		}
		
		FileInfo fileInfo = read();
        List dataObjs =  null;
        try {
        	dataObjs =  TemplateUtil.convert(template.getColumns(), fileInfo.getData(), domainClazz);
		} catch (Exception e) {
			logger.error("数据转换对象失败",e);
			publishTaskEvent(TaskLogEvent.fail(taskContext,"数据转换对象失败"));
		}
        callBack.exec(this,dataObjs);    
   }
	
	/**
	 * 读取文件返回Workbook
	 * 
	 * @param file
	 * @return
	 * @throws IOException 
	 * @throws Exception
	 */
	@SuppressWarnings("unused")
	public FileInfo read(){
		HttpURLConnection conn = null;
		
		FileInfo fileInfo = new FileInfo();
		//获取文件名称和文件类型
		fileInfo.setFileName(fileName);	
		fileInfo.setFileType(fileName.substring(fileName.lastIndexOf(".") + 1, fileName.length()));
		try {
			URL url = new URL(fileUrl);
			//设置代理
            Proxy proxy = ProxyCreateUtil.createUrlProxy(httpProxyUrl, httpProxyPort);
            if(proxy != null){
            	conn = (HttpURLConnection) url.openConnection(proxy);
            } else {
            	conn = (HttpURLConnection) url.openConnection();
            }
            
			//防止屏蔽程序抓取而返回403错误
	        conn.setRequestProperty("User-Agent", "Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt)");
	      //得到输入流
	        InputStream inputStream = conn.getInputStream();
	        if(true){
	        	List<String[]> data=parse(inputStream,template);
	        	fileInfo.setData(data);
	        }else{
	        	  //获取自己数组
	            byte[] buffer = new byte[1024];
	            int len = 0;
	            ByteArrayOutputStream bos = new ByteArrayOutputStream();
	            while((len = inputStream.read(buffer)) != -1) {
	                bos.write(buffer, 0, len);
	            }
	            bos.close();
	            
	            byte[] getData = bos.toByteArray();
	            //文件保存位置
	            File saveDir = new File(savePath);
	            if(!saveDir.exists()){
	                saveDir.mkdir();
	            }
	            File file = new File(saveDir+File.separator+fileName);
	            FileOutputStream fos = new FileOutputStream(file);
	            fos.write(getData);
	            if(fos!=null){
	                fos.close();
	            }
	            if(inputStream!=null){
	                inputStream.close();
	            }
        	}
    
		} catch (Exception e) {
			logger.error("文件解析失败",e);
			publishTaskEvent(TaskLogEvent.fail(taskContext,"文件解析失败,请先确保导入文件内容格式和模板一致"));
		}
		return fileInfo;
	}
	
	
	/**
	 * 解析文件
	 * @param fileInfo
	 * @return
	 */
	protected abstract  List<String[]> parse(FileInfo fileInfo,Template template);
	
	/**
	 * 解析文件
	 * @param inputStream
	 * @param template
	 * @return
	 */
	protected abstract  List<String[]> parse(InputStream inputStream,Template template);


}
