package com.yizhi.application.filter;

import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.util.HTTPRequestUtils;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.token.TokenHelper;
import com.yizhi.system.application.model.AuthAccountSiteModel;
import com.yizhi.system.application.system.remote.AccountClient;
import com.yizhi.system.application.system.remote.AuthClient;
import com.yizhi.system.application.system.remote.CompanyClient;
import com.yizhi.system.application.system.remote.SiteClient;
import com.yizhi.system.application.vo.AccountVO;
import com.yizhi.system.application.vo.CompanyVO;
import com.yizhi.system.application.vo.SiteVO;
import com.yizhi.util.application.constant.GlobalConstant;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.json.JsonUtil;
import io.jsonwebtoken.Claims;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;

/**
 * @Author: XieHaijun
 * @Description:
 * @Date: Created in 14:47 2018/5/15
 * @Modified By
 */
@Component
public class EntryFilter extends BaseZuulFilter {

    @Autowired
    private TokenHelper tokenHelper;

    @Autowired
    private RedisCache redisCache;

    @Autowired
    private CompanyClient companyClient;

    @Autowired
    private AccountClient accountClient;

    @Autowired
    private SiteClient siteClient;

    @Autowired
    private AuthClient authClient;
    /**
     * 登陆的接口url
     */
    private static final String[] LOGIN_URL = {"system/api/login", "system/api/loginApp", "system/api/loginSms"
            , "system/api/loginManage", "system/api/loginWeChat", "system/api/authApi",
            "wechat/get/user", "system/api/getWechat", "system/api/loginAppNoUser", "system/api/loginBindWechat", "system/api/account/public/getAccountByOpenId", "system/api/checkToken"};

    private static final String private_company = ",cpic,";// 使用 ,关键字, 形式

    /**
     * 可见范围、用户组织相关等后台公共url（几乎每个模块都会用到）
     */
    private static final String[] SYSTEM_MANAGE_PUBLIC_URL = {"/manage/org/userGroup/list",
            "/system/manage/account/name/search",
            "/message/manage/message/template/list",
            "/manage/site/site/menus",
            "/system/manage/dashboard/activeAccount",
            "/system/manage/dashboard/courseTime",
            "/system/manage/dashboard/courseFinish",
            "/system/manage/dashboard/tpFinish",
            "/system/manage/dashboard/accountYesterday",
            "/system/manage/dashboard/accountToWeek",
            "/system/manage/dashboard/accountToMonth",
            "/system/manage/dashboard/accountReadyTask",
            "/system/manage/dashboard/usedFunctions",
            "/system/manage/site/get",
            "/manage/group/list/page",
            "/manage/systemMailbox/selectMyPage",
            "/manage/systemMailbox/save",
            "/manage/systemMailbox/getById",
            "/manage/leaveWord/save",
            "/manage/leaveWord/getById",
            "/manage/site/classify/publication/insert",
            "/manage/site/classify/publication/list",
            "manage/api/eventTrack",
            "/manage/point/user/new/rank/list",
            "/manage/prompt/scroll/data",
            "/manage/classify/childnode/find"
    };

    /**
     * 无论前台后台，只要是能登录，就可以访问，不带上下文信息
     */
    private static final String[] ACCOUNT_COMMON_URL = {"/aliyun/upload/policy", "/aliyun/upload/download/url", "/manage/authz/resource/new/scan", "/authz/get/id"};

    private static Logger logger = LoggerFactory.getLogger(EntryFilter.class);

    /**
     * 报表的 url
     */
    private static final String[] REPORT_URL = {"/report/", "/offlineCourseReport/", "/point/user/rank/"};
    /**
     * 公共的url
     */
    private static String PUBLIC_URl = "/public/";

    /**
     * 管理端的url
     */
    private static String MANAGE_URl = "/manage/";

    @Override
    public boolean shouldFilter() {
        return true;
    }

    @Override
    public Object run() {
        // 获取请求上下文N
        RequestContext ctx = RequestContext.getCurrentContext();
        Map<String, List<String>> queryParams = HTTPRequestUtils.getInstance().getQueryParams();

        // 获取url
        String url = ctx.getRequest().getRequestURL().toString();
        try {
            // 获取请求ID
            ctx.addZuulRequestHeader(GlobalConstant.REQUEST_ID, ContextHolder.get().getRequestId());
            logger.info("请求参数={}", JSON.toJSONString(queryParams));
            // 公开访问url，不需要登陆处理。
            if (StringUtils.indexOf(url, PUBLIC_URl) >= 0) {
                if (!CollectionUtils.isEmpty(queryParams)) {
                    List<String> companyCodes = queryParams.get(GlobalConstant.COMPANY_CODE);
                    CompanyVO companyInfo = null;
                    if (!CollectionUtils.isEmpty(companyCodes)) {
                        companyInfo = getCompanyVO(companyCodes.get(0));
                        if (companyInfo != null && companyInfo.getId() != null) {
                            // 设置企业信息上下文
                            ctx.addZuulRequestHeader(GlobalConstant.COMPANY_ID, String.valueOf(companyInfo.getId()));
                            ctx.addZuulRequestHeader(GlobalConstant.COMPANY_CODE, companyInfo.getCode());
                            ctx.addZuulRequestHeader(GlobalConstant.COMPANY_NAME, urlEncoder(companyInfo.getName()));
                        }
                    }
                    List<String> siteCodes = queryParams.get(GlobalConstant.SITE_CODE);
                    if (!CollectionUtils.isEmpty(siteCodes) && companyInfo != null) {
                        SiteVO siteVO = getSiteVO(companyInfo.getId(), siteCodes, companyCodes.get(0));
                        if (siteVO != null && siteVO.getId() != null) {
                            ctx.addZuulRequestHeader(GlobalConstant.SITE_ID, String.valueOf(siteVO.getId()));
                            ctx.addZuulRequestHeader(GlobalConstant.SITE_CODE, String.valueOf(siteVO.getCode()));
                            ctx.addZuulRequestHeader(GlobalConstant.SITE_NAME, urlEncoder(siteVO.getName()));
                        }
                    }
                }

                // 获取令牌对象
                String token = ctx.getRequest().getHeader(GlobalConstant.TOKEN_HEADER);
                if (StringUtils.isNotBlank(token)) {
                    Claims claims = tokenHelper.parseToken(token);
                    if (claims != null) {
                        // 获取用户ID
                        Long accountId = Long.valueOf(claims.get(GlobalConstant.ACCOUNT_ID, String.class));
                        // 获取用户的账号信息
                        AccountVO accountVO = accountClient.findById(accountId);
                        // 用户信息
                        ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_ID, String.valueOf(accountVO.getId()));
                        ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_NAME, urlEncoder(accountVO.getName()));
                        ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_FULL_NAME, urlEncoder(accountVO.getFullName()));
                        ctx.addZuulRequestHeader(GlobalConstant.HEAD_PORTRAIT, urlEncoder(accountVO.getHeadPortrait()));
                        ctx.addZuulRequestHeader(GlobalConstant.ORG_ID, String.valueOf(accountVO.getOrgId()));
                    }
                }
                // 获取语言类型
                String languageType = ctx.getRequest().getHeader(GlobalConstant.LANGUAGE_TYPE);
                if (languageType != null) {
                    ctx.addZuulRequestHeader(GlobalConstant.LANGUAGE_TYPE, languageType);
                }
                return null;
            }

            if (isCommonUrl(url)) {
                // 判断用户是否有站点访问权限
                return null;
            }

            // 获取请求企业code
            List<String> companyCodes = queryParams.get(GlobalConstant.COMPANY_CODE);
            if (CollectionUtils.isEmpty(companyCodes)) {
                logger.info("请求url没有指定企业code");
                setFailCtx(ctx, ReturnCode.NOT_COMPANY_PARAM_INVALID);
                return null;
            }
            CompanyVO companyVO = null;
            String companyCode = companyCodes.get(0);
            companyVO = getCompanyVO(companyCode);

            // 如果是登陆请求
            if (isLoginUrl(url)) {
                if (companyVO == null || companyVO.getId() == null) {
                    logger.info("企业信息不存在");
                    setFailCtx(ctx, ReturnCode.NOT_COMPANY_EXTIS);
                    return null;
                }

                // 设置上下文
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_ID, String.valueOf(companyVO.getId()));
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_CODE, companyCode);
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_NAME, urlEncoder(companyVO.getName()));

                //ctx.addZuulRequestHeader(GlobalConstant.AUTH_CODE,companyVO.getAuthCode());

                SiteVO siteVO = setSiteVO(ctx, queryParams, companyVO.getId(), companyCode);
                if (siteVO == null) {
                    return null;
                }

                ctx.addZuulRequestHeader(GlobalConstant.SITE_ID, String.valueOf(siteVO.getId()));
                ctx.addZuulRequestHeader(GlobalConstant.SITE_CODE, String.valueOf(siteVO.getCode()));
                ctx.addZuulRequestHeader(GlobalConstant.SITE_NAME, urlEncoder(siteVO.getName()));

                return null;
            } else {
                // 获取令牌对象
                String token = ctx.getRequest().getHeader(GlobalConstant.TOKEN_HEADER);
                logger.info("令牌信息=token{}", token);
                if (StringUtils.isEmpty(token)) {
                    logger.info("令牌信息为空，请重新登陆获取令牌信息");
                    setFailCtx(ctx, ReturnCode.NO_TOKEN);
                    return null;
                }
                Claims claims = tokenHelper.parseToken(token);
                if (claims == null) {
                    logger.info("解析令牌为空，令牌失效或者过期，请重新登陆");
                    setFailCtx(ctx, ReturnCode.TOKEN_INVALID);
                    return null;
                }
                // 获取企业信息
                Long companyId = Long.valueOf(claims.get(GlobalConstant.COMPANY_ID, String.class));
                // 获取企业code
                companyCode = claims.get(GlobalConstant.COMPANY_CODE, String.class);
                // 获取企业名称
                String companyName = claims.get(GlobalConstant.COMPANY_NAME, String.class);
                logger.info("获取企业信息，COMPANY_ID={}，COMPANY_CODE={}，COMPANY_NAME={}", companyId, companyCode, companyName);
                // 设置企业信息上下文
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_ID, companyId.toString());
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_CODE, companyCode);
                ctx.addZuulRequestHeader(GlobalConstant.COMPANY_NAME, urlEncoder(companyName));
                // 判断访问的企业code是否和令牌的企业code一致
                if (!StringUtils.equals(companyCodes.get(0), companyCode)) {
                    logger.info("访问的企业和登陆企业不匹配");
                    setFailCtx(ctx, ReturnCode.COMPANY_NOT_LOGIN);
                    return null;
                }
                SiteVO siteVO = setSiteVO(ctx, queryParams, companyId, companyCode);
                if (siteVO == null) {
                    return null;
                }
                // 检查访问的站点是否和登陆站点一致
                String siteCodeTmp = siteVO.getCode();
                // 如果不是管理端并且不在特殊指定的企业，对令牌的站点id和访问的站点做比对检查
                /*if (StringUtils.indexOf(url, MANAGE_URl) < 0 && private_company.indexOf("," + companyCode + ",") < 0) {
                    String siteIdStr = claims.get(GlobalConstant.SITE_ID, String.class);
                    Long siteId = Long.valueOf(siteIdStr == null ? "0" : siteIdStr);
                    if (siteId.compareTo(siteVO.getId()) != 0) {
                        logger.info("您已切换了访问的站点，请刷新页面。");
                        setFailCtx(ctx, ReturnCode.SITE_NOT_LOGIN);
                        return null;
                    }
                }*/
                ctx.addZuulRequestHeader(GlobalConstant.SITE_ID, String.valueOf(siteVO.getId()));
                ctx.addZuulRequestHeader(GlobalConstant.SITE_CODE, String.valueOf(siteVO.getCode()));
                ctx.addZuulRequestHeader(GlobalConstant.SITE_NAME, urlEncoder(siteVO.getName()));
                //ctx.addZuulRequestHeader(GlobalConstant.SITE_TYPE, String.valueOf(siteVO.getType()));
                // 设定站点关联的管辖范围ID
                ctx.addZuulRequestHeader(GlobalConstant.SITE_MEMBER, String.valueOf(siteVO.getUserGroupId()));

                // 获取用户ID
                Long accountId = Long.valueOf(claims.get(GlobalConstant.ACCOUNT_ID, String.class));
                // 获取用户的账号信息
                AccountVO accountVO = accountClient.findById(accountId);
                // 检查用户是否过期
                if (checkUser(ctx, accountVO)) {
                    return null;
                }
                Boolean admin = false;

                // 获取语言类型
                String languageType = ctx.getRequest().getHeader(GlobalConstant.LANGUAGE_TYPE);
                if (languageType != null) {
                    ctx.addZuulRequestHeader(GlobalConstant.LANGUAGE_TYPE, languageType);
                }

                // 非管理获取relationIds
                if (StringUtils.indexOf(url, MANAGE_URl) < 0) {
                    ctx.addZuulRequestHeader(GlobalConstant.REQUEST_TYPE, com.yizhi.core.application.context.RequestContext.RequestType.STUDENT.name());
                    // 判断用户是否有站点访问权限
                    boolean siteAccount = authClient.isAccessToSiteFront(companyId, accountVO.getId(), companyCode, siteVO.getId());
                    logger.info("siteAccount={}", siteAccount);
                    if (!siteAccount) {
                        logger.info("没有指定站点访问权限");
                        setFailCtx(ctx, ReturnCode.NOT_SITE_ACCESS);
                        return null;
                    }
                    // 获取relationIds
                    List<Long> relationIds = authClient.getRelationIds(companyId, accountId, siteVO.getId());
                    if (!CollectionUtils.isEmpty(relationIds)) {
                        ctx.addZuulRequestHeader(GlobalConstant.RELATION_IDS, Joiner.on(",").join(relationIds));
                    }
                    //logger.info("relationIds={}", relationIds);

                } else { // 管理端接口
                    ctx.addZuulRequestHeader(GlobalConstant.REQUEST_TYPE, com.yizhi.core.application.context.RequestContext.RequestType.MANAGE.name());
                    String keyPrefix = null;
                    boolean isReportUrl = isReportUrl(url);
                    // 如果是报表接口
                    if (isReportUrl) {
                        keyPrefix = CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ORGIDS;
                    } else {
                        keyPrefix = CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS;
                    }

                    // 获取用户权限资源
                    AuthAccountSiteModel authAccountModel = getAuthAccountSiteModel(companyId, companyCode, accountVO.getId(), siteVO.getId(), siteVO.getCode(), keyPrefix);

                    if (null == authAccountModel && !isManagePublic(url)) {
                        logger.info("没有指定站点访问权限");
                        setFailCtx(ctx, ReturnCode.NOT_SITE_ACCESS);
                        return null;
                    }

                    // 过滤查询可见范围、用户组织相关url
                    if (isManagePublic(url)) {
                        if (authAccountModel != null && authAccountModel.getSiteAdmin()) {
                            admin = Boolean.TRUE;
                        }
                    } else {
                        // 1. 站点管理员
                        if (authAccountModel.getSiteAdmin()) {
                            admin = Boolean.TRUE;
                        }
                        // 2. 不是站点管理员
                        else {
                            //  model 中没有任何接口权限
                            Map<String, Set<Long>> urlRanges = authAccountModel.getUrlRanges();
                            if (CollectionUtils.isEmpty(urlRanges)) {
                                logger.info("没有任何接口权限");
                                setFailCtx(ctx, ReturnCode.NO_INTERFACE_ACCESS);
                                return null;
                            }

                            // model 中没有当前权限接口
                            Optional<String> roleUrl = urlRanges.keySet().stream().filter(item -> url.indexOf(item) >= 0).findFirst();
                            if (!roleUrl.isPresent()) {
                                logger.info("没有当前接口权限");
                                setFailCtx(ctx, ReturnCode.NO_CURRENT_INTERFACE_ACCESS);
                                return null;
                            }

                            // model 中当前接口权限下没有管理对象
                            Set<Long> ids = urlRanges.get(roleUrl.get());

                            // 如果是报表 url, 组装 orgIds
                            if (isReportUrl) {
                                if (checkUrlAdmin(ids, companyVO.getRootOrgId())) {
                                    admin = Boolean.TRUE;
                                } else {
                                    // 获取子部门
                                    if (!CollectionUtils.isEmpty(ids)) {
                                        Set<Long> withChildOrgIds = authClient.getSiteUrlAutho(ids);
                                        if (!CollectionUtils.isEmpty(withChildOrgIds)) {
                                            String orgIds = Joiner.on(",").join(withChildOrgIds);
                                            ctx.addZuulRequestHeader(GlobalConstant.ORG_IDS, orgIds);
                                        }
                                    }
                                }
                            }
                            // 除了报表的管理端访问 url
                            else {
                                if (CollectionUtils.isEmpty(ids)) {
                                    logger.info("当前接口权限下没有管理对象");
                                    setFailCtx(ctx, ReturnCode.NO_CURRENT_INTERFACE_MANAGER);
                                    return null;
                                }

                                String managerIdsStr = Joiner.on(",").join(ids);
                                ctx.addZuulRequestHeader(GlobalConstant.MANAGER_IDS, managerIdsStr);
                            }
                        }
                    }
                }
                ctx.addZuulRequestHeader(GlobalConstant.ADMIN_FLG, admin.toString());
                // 用户信息
                ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_ID, String.valueOf(accountVO.getId()));
                ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_NAME, urlEncoder(accountVO.getName()));
                ctx.addZuulRequestHeader(GlobalConstant.ACCOUNT_FULL_NAME, urlEncoder(accountVO.getFullName()));
                ctx.addZuulRequestHeader(GlobalConstant.HEAD_PORTRAIT, urlEncoder(accountVO.getHeadPortrait()));
                ctx.addZuulRequestHeader(GlobalConstant.ORG_ID, String.valueOf(accountVO.getOrgId()));
            }

            return null;
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("*************************************网关处理异常{}", e);
            setFailCtx(ctx, ReturnCode.SERVICE_UNAVAILABLE);
            return null;
        }
    }

    private SiteVO setSiteVO(RequestContext ctx, Map<String, List<String>> queryParams, Long companyId, String companyCode) {
        // 获取请求站点信息
        List<String> siteCodes = queryParams.get(GlobalConstant.SITE_CODE);
        SiteVO siteVO = null;
        if (CollectionUtils.isEmpty(siteCodes)) {
            logger.info("没有指定组织code");
            setFailCtx(ctx, ReturnCode.NOT_SITE_CODE);
            return null;
        }
        siteVO = getSiteVO(companyId, siteCodes, companyCode);
        if (siteVO == null || siteVO.getId() == null) {
            logger.info("指定的组织code信息不存在");
            setFailCtx(ctx, ReturnCode.NOT_SITE_INFO_CODE);
            return null;
        }
        return siteVO;
    }

    /**
     * 检查用户是否有效
     *
     * @param ctx
     * @param accountVO
     * @return
     */
    private static boolean checkUser(RequestContext ctx, AccountVO accountVO) {
        if (!accountVO.getEnabled() || accountVO.getLocked()) {
            setFailCtx(ctx, ReturnCode.ACCOUNT_CLOSE);
            return true;
            // 前面时间大于后面时间==1
        } else if (accountVO.getExpiredType() == 2 && accountVO.getEndTime().compareTo(new Date()) <= 0) {
            setFailCtx(ctx, ReturnCode.ACCOUNT_INVALID);
            return true;
        } else if (accountVO.getExpiredType() == 3 && accountVO.getExpiredTime().compareTo(new Date()) <= 0) {
            setFailCtx(ctx, ReturnCode.ACCOUNT_INVALID);
        }
        return false;
    }

    /**
     * 根据站点code查询站点信息
     *
     * @param companyId 企业
     * @param siteCodes 站点信息
     * @return
     */
    private SiteVO getSiteVO(Long companyId, List<String> siteCodes, String companyCode) {
        // 根据站点code获取站点信息
        SiteVO siteVO = null;
        Object object = redisCache.hget(CacheNamespace.REDIS_SITES,
                companyCode + CacheNamespace.REDIS_HASH_SEPARATOR + siteCodes.get(0));
        if (object == null) {
            siteVO = siteClient.findByName(siteCodes.get(0), companyId);
        } else {
            siteVO = JsonUtil.json2Ojbect(object.toString(), SiteVO.class);
        }
        return siteVO;
    }

    /**
     * 根据企业code获取企业信息
     *
     * @param companyCode
     * @return
     */
    private CompanyVO getCompanyVO(String companyCode) {
        // 根据企业code获取企业信息
        CompanyVO companyVO = null;
        Object cObject = redisCache.hget(CacheNamespace.REDIS_COMPANIES, companyCode);
        if (cObject == null) {
            companyVO = companyClient.findByCode(companyCode);
        } else {
            companyVO = JsonUtil.json2Ojbect(cObject.toString(), CompanyVO.class);
        }
        logger.info("url存在企业code，获取缓存的企业信息" + JSON.toJSONString(companyVO));
        return companyVO;
    }

    /**
     * 获取权限
     *
     * @param companyId
     * @param accountId
     * @param siteId
     * @param key       如果是报表接口，需要组织 id
     * @return
     */
    private AuthAccountSiteModel getAuthAccountSiteModel(Long companyId, String companyCode, Long accountId, Long siteId, String siteCode, String key) {
        AuthAccountSiteModel authAccountModel = null;
        Object object = redisCache.hget(key + companyCode, siteCode + CacheNamespace.REDIS_HASH_SEPARATOR + accountId);
        if (null == object) {
            authAccountModel = authClient.getSiteAutho(companyId, companyCode, accountId, siteId, key);
        } else {
            authAccountModel = JsonUtil.json2Ojbect(object.toString(), AuthAccountSiteModel.class);
        }
        return authAccountModel;
    }

    /**
     * 如果管辖范围中包含根组织，该url下是admin
     *
     * @param orgIds
     * @param rootOrgId
     * @return
     */
    private boolean checkUrlAdmin(Set<Long> orgIds, Long rootOrgId) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return false;
        }
        return orgIds.contains(rootOrgId);
    }

    /**
     * 是否是登录请求
     *
     * @param url 请求的url
     * @return
     */
    private static boolean isLoginUrl(String url) {
        for (String item : LOGIN_URL) {
            if (url.contains(item)) {
                return true;
            }
        }
        return false;
    }

    /**
     * url编码字符
     *
     * @param val
     * @return
     */
    private static String urlEncoder(String val) {
        String retStr = "";
        try {
            retStr = URLEncoder.encode(StringUtils.trimToEmpty(val), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            logger.info("中文字符串通过url编码失败，异常信息=" + JSON.toJSONString(e));
        }
        return retStr;
    }

    /**
     * 设定错误上下文
     *
     * @param ctx
     * @param returnCode
     */
    private static void setFailCtx(RequestContext ctx, ReturnCode returnCode) {
        ctx.setResponseStatusCode(403);
        ctx.setResponseBody(returnCode.toString());
        ctx.getResponse().setContentType("application/json;charset=UTF-8");
        ctx.setSendZuulResponse(false);
    }

    /**
     * 可见范围、用户组织相关等后台公共url（几乎每个模块都会用到）
     *
     * @param url
     * @return
     */
    private Boolean isManagePublic(String url) {
        return Arrays.stream(SYSTEM_MANAGE_PUBLIC_URL).filter((item) -> url.indexOf(item) != -1).findFirst().isPresent();
    }

    /**
     * 无论前台后台，只要是能登录，就可以访问，不带上下文信息
     *
     * @param url
     * @return
     */
    private Boolean isCommonUrl(String url) {
        return Arrays.stream(ACCOUNT_COMMON_URL).filter((item) -> url.indexOf(item) != -1).findFirst().isPresent();
    }

    /**
     * 是否是报表请求 url
     *
     * @param url
     * @return
     */
    public boolean isReportUrl(String url) {
        return Arrays.stream(REPORT_URL).filter((item) -> url.indexOf(item) > 0).findFirst().isPresent();
    }
}
