package com.yizhi.application.filter;

import cn.hutool.http.HttpUtil;
import cn.hutool.http.useragent.UserAgent;
import cn.hutool.http.useragent.UserAgentUtil;
import com.alibaba.fastjson.JSON;
import com.netflix.zuul.context.RequestContext;
import com.yizhi.logging.dto.RequestLogDTO;
import com.yizhi.util.application.constant.GlobalConstant;
import com.yizhi.util.application.constant.QueueConstant;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Map;

/**
 * @ClassName LogPostFilter
 * @Description TODO
 * @Author shengchenglong
 * @DATE 2021/1/11 11:19
 * @Version 1.0
 */
@Component
@Slf4j
public class LogPostFilter extends BaseZuulFilter {

    @Value("${gateway.logSwitch:false}")
    private Boolean logSwitch;
/*
    @Autowired
    private RabbitTemplate rabbitTemplate;*/

    @Override
    public boolean shouldFilter() {
        return logSwitch;
    }

    @Override
    public Object run() {
        RequestContext requestContext = RequestContext.getCurrentContext();
        HttpServletRequest request = requestContext.getRequest();
        HttpServletResponse response = requestContext.getResponse();

        String errMsg = "";

        String localIp = null;
        try {
            localIp = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            errMsg += "未知主机错误\n" + e.getLocalizedMessage() + "\n";
            log.error("未知主机错误", e);
        }

        Map<String, List<String>> requestParams = requestContext.getRequestQueryParams();
        String requestBody = null;
        try {
            requestBody = StreamUtils.copyToString(request.getInputStream(), Charset.forName("UTF-8"));
        } catch (IOException e) {
            errMsg += "获取请求body错误\n" + e.getLocalizedMessage() + "\n";
            log.error("获取请求body错误", e);
        }

        String responseBody = requestContext.getResponseBody();

        Map<String, String> zuulRequestHeaders = requestContext.getZuulRequestHeaders();

        String accountId = zuulRequestHeaders.get(GlobalConstant.ACCOUNT_ID.toLowerCase());
        String companyId = zuulRequestHeaders.get(GlobalConstant.COMPANY_ID.toLowerCase());
        String siteId = zuulRequestHeaders.get(GlobalConstant.SITE_ID.toLowerCase());
        String orgId = zuulRequestHeaders.get(GlobalConstant.ORG_ID.toLowerCase());

        RequestLogDTO requestLogDTO = RequestLogDTO.builder()
                .requestId((String) requestContext.get(GlobalConstant.REQUEST_ID))
                .requestLocalIp(localIp)
                .requestRemoteIp(getRemoteIp(request))
                .requestUrl(request.getRequestURL().toString())
                .requestUri(request.getRequestURI())
                .requestMethod(request.getMethod())
                .requestParams(CollectionUtils.isEmpty(requestParams) ? null : JSON.toJSONString(requestParams))
                .requestBody(requestBody)

                .responseStatus(response.getStatus())
                .responseBody(responseBody)
                .errorMsg(errMsg)

                .accountId(!StringUtils.isEmpty(accountId) ? Long.valueOf(accountId) : null)
                .orgId(!StringUtils.isEmpty(orgId) ? Long.valueOf(orgId) : null)
                .siteId(!StringUtils.isEmpty(siteId) ? Long.valueOf(siteId) : null)
                .companyId(!StringUtils.isEmpty(companyId) ? Long.valueOf(companyId) : null)
                .startTime(LocalDateTime
                        .ofInstant(Instant.ofEpochMilli(Long.valueOf(requestContext.getZuulRequestHeaders().get(LogPreFilter.GATEWAY_LOG_START_TIME_HEADER))),
                                ZoneId.systemDefault()))
                .endTime(LocalDateTime.now())
                .terminalType(getUserAgent(request.getHeader("User-Agent")))
                .build();

        //rabbitTemplate.convertAndSend(QueueConstant.REQUEST_LOG_QUEUE, requestLogDTO);
        return null;
    }

    private String getRemoteIp(HttpServletRequest request) {
        // 通过nginx配置自定义的ip
        String remoteIp = request.getHeader("nginx_user_ip");
        if (!StringUtils.isEmpty(remoteIp)) {
            return remoteIp;
        }

        String remoteAddr = request.getRemoteAddr();
        String forwarded = request.getHeader("X-Forwarded-For");
        String realIp = request.getHeader("X-Real-IP");
        if (realIp == null) {
            if (forwarded == null) {
                remoteIp = remoteAddr;
            } else {
                remoteIp = remoteAddr + "/" + forwarded.split(",")[0];
            }
        } else {
            if (realIp.equals(forwarded)) {
                remoteIp = realIp;
            } else {
                if (forwarded != null) {
                    forwarded = forwarded.split(",")[0];
                }
                remoteIp = realIp + "/" + forwarded;
            }
        }
        return remoteIp;
    }

    private String getUserAgent(String userAgent) {
        UserAgent userAgentParsed = UserAgentUtil.parse(userAgent);
        if (userAgentParsed == null) {
            return "unknown";
        }
        return userAgentParsed.getOs() + " | " + userAgentParsed.getBrowser() + " | " + (userAgentParsed.isMobile() ? "移动端" : "PC端");
    }
}
