package com.yizhi.application.orm.id;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import com.yizhi.core.application.cache.RedisCache;

import cn.ms.sequence.Sequence;

/**
 * 分布式snowflake Id生成器
 * 
 * @author scotthu
 *
 */
@Component
public class DistributeSnowFlakeIdGenerator implements IdGenerator {

	private Sequence sequence;

	@Autowired
	RedisCache redisCache;

	// id生成器进程标识注册之缓存key
	private final static String CACHE_REGISTRY_KEY = "ID_GENERATOR_FLAG_REGISTRY";

	// 服务节点注册时间上限,超过此时间则认为此节点已死
	final static int REGISTER_MAX_DAY = 1;

	// 本进程的id生成器进程标识键
	private static Long FLAG_ITEM_KEY = -1L;
	//注册尝试最大次数
	private final static int REGISTER_COUNT = 50;

	private static Logger log = LoggerFactory.getLogger(DistributeSnowFlakeIdGenerator.class);

	@PostConstruct
	public void init() {

		registerKey();
		
		removeDeadIdGenerateNodes(redisCache.hmget(CACHE_REGISTRY_KEY));

		Long dataCenterId = FLAG_ITEM_KEY / 32;
		Long realWorkerId = FLAG_ITEM_KEY % 32;
		sequence = new Sequence(realWorkerId, dataCenterId);
	}

	/**
	 * 定时刷新本节点的注册值,报告本节点还活着(2小时一次)
	 */
	@Scheduled(cron = "0 0 0/2 * * ? ")
	public void refreshFlagRegistry() {
		redisCache.hset(CACHE_REGISTRY_KEY, FLAG_ITEM_KEY.toString(), String.valueOf(System.currentTimeMillis()));
	}
	
	/**
	 * 注册本节点到缓存中
	 */
	void registerKey() {
		Map<Object, Object> registries = null;
		Long candidateWorkId = null;
		boolean result = false;
		//重复注册一定次数
		for (int i = 0; i < REGISTER_COUNT; i++) {
			registries = redisCache.hmget(CACHE_REGISTRY_KEY);
			candidateWorkId = getCandidateWorkId(registries);
			result = redisCache.hsetIfAbsent(CACHE_REGISTRY_KEY, candidateWorkId.toString(), String.valueOf(System.currentTimeMillis()));
			if(result) {
				// 本地缓存竞争到的标识,以便定期刷新
				FLAG_ITEM_KEY = candidateWorkId;
				log.info("id生成器redis注册标识{}成功",FLAG_ITEM_KEY);
				break;
			}
		}
		
		if(!result) {
			log.error("id生成器redis注册失败");
			throw new RuntimeException("id生成器redis注册失败");
		}
	}

	/**
	 * 获得本节点可使用的占位标识
	 * @param registries  redis里注册的节点
	 * @return  本节点可使用的占位标识
	 */
	Long getCandidateWorkId(Map<Object, Object> registries) {
		if(registries == null || registries.size() == 0)
			return 1L;

		Integer id = getUnusedId(registries);
		if(id < 0)
			id = collectIdFromExpiredNodes(registries);

		if(id < 0) {
			log.error("雪花id生成器找不到可用的占位标识,进程退出");
			throw new RuntimeException("id生成器redis注册失败");
		}

		return Long.valueOf(id);
	}
	
	/**
	 * 从过期节点中收集可使用占位标识
	 * 已过期节点认为是死亡节点
	 * @param registries 占位标识注册信息
	 * @return 收集到的可使用的过期占位标识
	 */
	Integer collectIdFromExpiredNodes(Map<Object, Object> registries){
		if (registries == null || registries.size() == 0)
			return -1;
		
		Integer id = -1;
		for (Object itemKey : registries.keySet()) {
			String itemValue = registries.get(itemKey).toString();

			Long registerDays = (System.currentTimeMillis() - Long.valueOf(itemValue)) / (1000 * 3600 * 24);
			if (registerDays > REGISTER_MAX_DAY) {
				id = Integer.valueOf(itemKey.toString());
				break;
			}
				
		}
		
		return id;
	}
	
	/**
	 * 获得尚未使用的占位标识
	 * @param registries
	 * @return
	 */
	Integer getUnusedId(Map<Object, Object> registries) {
		if (registries == null || registries.size() == 0)
			return 1;
		
		List<Integer> unusedIds = new ArrayList<Integer>(40);

		for (Object itemKey : registries.keySet()) {
			unusedIds.add(Integer.valueOf(itemKey.toString()));
		}

		int count = unusedIds.size();
		if (count == 1)
			return unusedIds.get(0) <= 1022 ? unusedIds.get(0) + 1 : 1;

		unusedIds.sort(new Comparator<Integer>() {
			@Override
			public int compare(Integer o1, Integer o2) {
				return o1 - o2;
			}
		});
		
		Integer minUsed = unusedIds.get(0);
		if(minUsed > 1)
			return minUsed - 1;
		Integer maxUsed = unusedIds.get(count -1);
		if(maxUsed < 1023)
			return maxUsed + 1;
		
		Integer unusedId = -1;
		for (int i = 0; i < count - 1; i++) {
			if (unusedIds.get(i + 1) == unusedIds.get(i) + 1)
				continue;

			unusedId = unusedIds.get(i) + 1;
			break;
		}
		
		return unusedId;
	}
	
	/**
	 * 删除判断为死亡的节点的占位标识,为新节点注册占位标识提供可用资源
	 * @param registries redis里注册的节点
	 */
	void removeDeadIdGenerateNodes(Map<Object, Object> registries) {
		Set<Object> nodes = getDeadIdGenerateNodes(registries);
		if(nodes.size() == 0)
			return;
		
		redisCache.hdel(CACHE_REGISTRY_KEY, nodes.toArray());
		log.info("清除redis中的死亡节点的占位标识：{}",nodes.toString());
	}

	/**
	 * 搜集死亡的节点
	 * @param registries redis里注册的节点
	 * @return 死亡的节点key集合
	 */
	Set<Object> getDeadIdGenerateNodes(Map<Object, Object> registries) {
		Set<Object> itemKeys = new HashSet<Object>(40);
		for (Object itemKey : registries.keySet()) {
			String itemValue = registries.get(itemKey).toString();

			Long registerDays = (System.currentTimeMillis() - Long.valueOf(itemValue)) / (1000 * 3600 * 24);
			if (registerDays >= REGISTER_MAX_DAY)
				itemKeys.add(itemKey);
		}
		
		return itemKeys;
	}

	@Override
	public Long generate() {
		return sequence.nextId();
	}

}
 