package com.yizhi.system.application.controller.manage;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.application.orm.page.PageUtil;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.system.application.domain.AuthzRole;
import com.yizhi.system.application.service.IAuthzAccountRoleService;
import com.yizhi.system.application.service.IAuthzRoleService;
import com.yizhi.system.application.vo.*;
import com.yizhi.system.application.vo.domain.AuthzAccountRole;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.page.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.*;

@Api(tags = "角色管理接口")
@RestController
@RequestMapping(value = "/manage/role")
public class RoleController {

    @Autowired
    IAuthzRoleService roleService;
    @Autowired
    DomainConverter converter;

    @Autowired
    IAuthzAccountRoleService authzAccountRoleService;

    private static final Logger logger = LoggerFactory.getLogger(RoleController.class);

    /**
     * 新增角色接口
     *
     * @param roleVO
     * @return
     */
    @ApiOperation(value = "新增角色", notes = "新增角色")
    @PostMapping(value = "/add")
    public Response<String> addRole(@RequestBody RoleVO roleVO) {
        try {
            if (!StringUtils.isEmpty(roleVO.getCode())) {
                if (roleVO.getCode().indexOf(AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX) != -1) {
                    return Response.fail("4001", "_site_admin 不能被设置成角色 code 关键字");
                }
            }
            com.yizhi.system.application.domain.AuthzRole role = (com.yizhi.system.application.domain.AuthzRole) converter.toDO(roleVO, com.yizhi.system.application.domain.AuthzRole.class);
            role.setCompanyId(ContextHolder.get().getCompanyId());
            role.setSiteId(ContextHolder.get().getSiteId());
            // 过滤重复的模块 id
            List<Long> moduleIds = roleVO.getModuleIds();
            if (CollectionUtils.isNotEmpty(moduleIds)) {
                HashSet<Long> set = new HashSet<>();
                for (Long menuId : moduleIds) {
                    if (menuId < 10) {
                        continue;
                    }
                    set.add(menuId);
                }
                moduleIds.clear();
                moduleIds.addAll(set);
                role.setMenuIds(moduleIds);
            }
            role.setGroupIds(roleVO.getGroupIds());
            role.setManagerAccountIds(roleVO.getManagerAccountIds());
            roleService.add(role);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    @ApiOperation(value = "角色管理的管理员编辑", response = Boolean.class)
    @PostMapping(value = "/manager/edit")
    public Response<Boolean> managerList(@ApiParam("角色 id") @RequestBody ManagerEditVo editVo) {
        Long roleId = editVo.getRoleId();
        if (roleId == null) {
            return Response.fail("角色 id 不能为空");
        }
        return Response.ok(roleService.managerEdit(editVo));
    }

    @ApiOperation(value = "查看当前站点内的管理员", response = ManagerVo.class)
    @GetMapping(value = "/manager/list")
    public Response<ManagerVo> managerList(@ApiParam("角色 id") @RequestParam("roleId") Long roleId,
                                           @ApiParam("用户名、姓名、工号") @RequestParam(value = "keyWord", required = false) String keyWord,
                                           @ApiParam("职务") @RequestParam(value = "position", required = false) String position,
                                           @ApiParam("部门id") @RequestParam(value = "orgId", required = false) Long orgId,
                                           @ApiParam("部门名称") @RequestParam(value = "orgName", required = false) String orgName,
                                           @ApiParam("pageNo") @RequestParam(value = "pageNo", required = false, defaultValue = "1") Integer pageNo,
                                           @ApiParam("pageSize") @RequestParam(value = "pageSize", required = false, defaultValue = "10") Integer pageSize) {
        try {
            Page<ManagerVo> page = roleService.pageManager(roleId, keyWord, position, orgId, orgName, pageNo, pageSize);
            Map<String, Integer> pageMap = new HashMap<>();
            pageMap.put("pageTotal", page.getTotal());
            pageMap.put("pageNo", page.getCurrent());
            pageMap.put("pageSize", page.getSize());
            return Response.ok(page.getRecords(), pageMap);

        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 删除角色
     *
     * @param roleVO
     * @return
     */
    @ApiOperation(value = "删除角色", notes = "删除角色")
    @PostMapping(value = "/delete")
    public Response<RoleVO> delRole(@RequestBody RoleVO roleVO) {
        try {
            roleService.delete((com.yizhi.system.application.domain.AuthzRole) converter.toDO(roleVO, com.yizhi.system.application.domain.AuthzRole.class));
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 修改角色
     *
     * @param roleVO
     * @return
     */
    @ApiOperation(value = "修改角色", notes = "修改角色")
    @PostMapping(value = "/update")
    public Response<RoleVO> updateRole(@RequestBody RoleVO roleVO) {
        logger.info("*******************RoleController************************************打印下信息");
        logger.info("roleVO=" + JSON.toJSONString(roleVO));
        try {
            com.yizhi.system.application.domain.AuthzRole role = (com.yizhi.system.application.domain.AuthzRole) converter.toDO(roleVO, com.yizhi.system.application.domain.AuthzRole.class);
            role.setCompanyId(ContextHolder.get().getCompanyId());
            role.setSiteId(ContextHolder.get().getSiteId());
            // 过滤重复的模块 id
            List<Long> moduleIds = roleVO.getModuleIds();
            if (CollectionUtils.isNotEmpty(moduleIds)) {
                HashSet<Long> set = new HashSet<>();
                for (Long menuId : moduleIds) {
                    if (menuId < 10) {
                        continue;
                    }
                    set.add(menuId);
                }
                moduleIds.clear();
                moduleIds.addAll(set);
                role.setMenuIds(moduleIds);
            }
            role.setGroupIds(roleVO.getGroupIds());
            roleService.update(role);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 分页查询角色
     *
     * @param roleVO
     * @return
     */
    @ApiOperation(value = "分页查询角色", notes = "分页查询角色")
    @PostMapping(value = "/list")
    public Response<RoleVO> list(@RequestBody PageRoleVO roleVO) {
        logger.info("*******************RoleController************************************打印下信息");
        logger.info("roleVO=" + JSON.toJSONString(roleVO));
        try {
            com.yizhi.system.application.domain.AuthzRole role = (com.yizhi.system.application.domain.AuthzRole) converter.toDO(roleVO, com.yizhi.system.application.domain.AuthzRole.class);
            role.setCompanyId(ContextHolder.get().getCompanyId());
            role.setSiteId(ContextHolder.get().getSiteId());
            // 如果是不是站点管理员，则只能看到自己创建的
            if (!ContextHolder.get().isAdmin()) {
                role.setCreateById(ContextHolder.get().getAccountId());
            }
            PageInfo<com.yizhi.system.application.domain.AuthzRole> pageInfo = new PageInfo<>();
            pageInfo.setPageNo(roleVO.getPageNo());
            pageInfo.setPageSize(roleVO.getPageSize());
            Page<AuthzRole> page = roleService.list(role, PageUtil.vice(pageInfo));
            Page<RoleVO> roleVOPage = new Page<>();
            List<RoleVO> roleVOList = new ArrayList<>();
            BeanUtils.copyProperties(page, roleVOPage);
            for (AuthzRole a : page.getRecords()) {
                RoleVO rv = new RoleVO();
                BeanUtils.copyProperties(a,rv);
                rv.setIsSiteDefaultAdmin(false);
                if (a.getCode().contains(AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX)){
                    rv.setIsSiteDefaultAdmin(true);
                }
                roleVOList.add(rv);
            }
            roleVOPage.setRecords(roleVOList);
            return Response.ok(PageUtil.versa(roleVOPage));
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 检查公司内roleName是否存在
     *
     * @param name
     * @return
     */
    @ApiOperation(value = "检查公司内roleName是否存在", notes = "检查公司内roleName是否存在")
    @GetMapping(value = "/name/exist")
    public Response<String> roleNameExist(@RequestParam String name) {
        try {
            roleService.checkExistName(name, ContextHolder.get().getSiteId());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 检查公司内roleCode是否存在
     *
     * @param code
     * @return
     */
    @ApiOperation(value = "检查公司内roleCode是否存在", notes = "检查公司内roleCode是否存在")
    @GetMapping(value = "/code/exist")
    public Response<String> roleCodeExitst(@RequestParam String code) {
        try {
            roleService.checkExistCode(code, ContextHolder.get().getSiteId());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 根据角色Id查询关联用户信息
     *
     * @param roleId
     * @return
     */
    @ApiOperation(value = "根据角色Id查询关联用户信息", notes = "根据角色Id查询关联用户信息")
    @GetMapping(value = "/account/list")
    public Response<AccountVO> searchAccountList(@RequestParam Long roleId) {
        try {
            List<com.yizhi.system.application.domain.Account> retList = roleService.getAccountByRoleId(roleId);
            return Response.ok(converter.toDOList(retList, AccountVO.class));
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 角色解除授权用户
     *
     * @param accountRoleVO
     * @return
     */
    @ApiOperation(value = "角色解除授权用户", notes = "角色解除授权用户")
    @PostMapping(value = "/account/delete")
    public Response<String> deleteAccount(@RequestBody AccountRoleVO accountRoleVO) {
        try {
            com.yizhi.system.application.domain.Site site = new com.yizhi.system.application.domain.Site();
            site.setId(ContextHolder.get().getSiteId());
            site = site.selectById();
            roleService.delAccountByRoleId(accountRoleVO.getRoleId(), site.getCode(), accountRoleVO.getAccountId());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 角色授权用户
     *
     * @param authorizeUsers
     * @return
     */
    @ApiOperation(value = "角色授权用户", notes = "角色授权用户")
    @PostMapping(value = "/auth/account")
    public Response<Boolean> authorize(@RequestBody AuthorizeUsersVO authorizeUsers) {
        try {
            com.yizhi.system.application.domain.Site site = new com.yizhi.system.application.domain.Site();
            site.setId(ContextHolder.get().getSiteId());
            site = site.selectById();
            roleService.authAccount(authorizeUsers.getRoleId(), site.getCode(), authorizeUsers.getUserIds());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    @ApiOperation(value = "查看角色", notes = "查看角色")
    @GetMapping(value = "/get")
    public Response<RoleVO> get(@RequestParam Long id) {
        try {
            com.yizhi.system.application.domain.AuthzRole role = roleService.get(id);
            RoleVO roleVO = (RoleVO) converter.toVO(role, RoleVO.class);
            roleVO.setGroupIds(role.getGroupIds());
            roleVO.setModuleIds(role.getMenuIds());
            return Response.ok(roleVO);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(), ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }


    @ApiOperation(value = "晚上跑报表需要用到,查看所有的管理员,不区分站点", notes = "晚上跑报表需要用到,查看所有的管理员,不区分站点")
    @GetMapping("/report/get/all/admin")
    public List<AuthzAccountRole> getAllAdmin() {
        List<com.yizhi.system.application.domain.AuthzAccountRole> allAdmin = authzAccountRoleService.getAllAdmin();
        return converter.toDOList(allAdmin, AuthzAccountRole.class);
    }

    @GetMapping(value = "/get/adminIds")
    public List<Long> getAdminIdsByAccountId(
            @RequestParam(name = "companyId") Long companyId,
            @RequestParam(name = "siteId") Long siteId,
            @RequestParam(name = "accountId") Long accountId
    ) {
        return authzAccountRoleService.getAdminIdsByAccountId(companyId, siteId, accountId);
    }

}
