package com.yizhi.system.application.controller.manage;

import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.application.orm.page.PageUtil;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.TaskContext;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.core.application.task.TaskExecutor;
import com.yizhi.system.application.service.ISensitiveWordService;
import com.yizhi.system.application.task.SensitiveTask;
import com.yizhi.system.application.vo.SensitiveWordPageVO;
import com.yizhi.system.application.vo.SensitiveWordVO;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.page.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

/**
 * <p>
 * 敏感词前端控制器
 * </p>
 *
 * @author Moni.Yin
 * @date 2018.4.8
 * @since 1.0
 */
@Api(tags = "敏感词接口")
@RestController
@RequestMapping(value = "/manage/sensitive")
public class SensitiveWordController {

    @Autowired
    ISensitiveWordService sensitiveWordService;
    @Autowired
    DomainConverter converter;
    @Autowired
    RedisCache redisCache;
    @Autowired
    SensitiveTask sensitiveTask;
    @Autowired
    TaskExecutor taskExecutor;

    private static final Logger logger = LoggerFactory.getLogger(SensitiveWordController.class);

    /**
     * 增加敏感词
     *
     * @param sensitiveWordVO
     * @return
     */
    @ApiOperation(value = "新增敏感词", notes = "新增敏感词")
    @PostMapping(value = "/insert")
    public Response<String> insert(@RequestBody SensitiveWordVO sensitiveWordVO) {
        try {
            sensitiveWordVO.setCompanyId(ContextHolder.get().getCompanyId());
            sensitiveWordService.save((com.yizhi.system.application.domain.SensitiveWord) converter.toDO(sensitiveWordVO, com.yizhi.system.application.domain.SensitiveWord.class));
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 修改敏感词
     *
     * @param sensitiveWordVO
     * @return
     */
    @ApiOperation(value = "修改敏感词", notes = "修改敏感词")
    @PostMapping(value = "/update")
    public Response<String> update(@RequestBody SensitiveWordVO sensitiveWordVO) {
        try {
            sensitiveWordVO.setCompanyId(ContextHolder.get().getCompanyId());
            sensitiveWordService.update((com.yizhi.system.application.domain.SensitiveWord) converter.toDO(sensitiveWordVO, com.yizhi.system.application.domain.SensitiveWord.class));
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 删除敏感词
     *
     * @param sensitiveWordVO
     * @return
     */
    @ApiOperation(value = "删除敏感词", notes = "删除敏感词")
    @PostMapping(value = "/delete")
    public Response<String> delete(@RequestBody SensitiveWordVO sensitiveWordVO) {
        try {
            sensitiveWordService.delete(sensitiveWordVO.getId());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 检查名称是否存在
     *
     * @param name
     * @return
     */
    @ApiOperation(value = "检查敏感词name是否存在", notes = "检查敏感词name是否存在")
    @GetMapping(value = "/check/name")
    public Response<Boolean> checkName(@RequestParam(name = "name") String name) {
        try {
            sensitiveWordService.checkName(0L, ContextHolder.get().getCompanyId(), name);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 查询敏感词列表
     *
     * @param sensitiveWordPageVO
     * @return
     */
    @ApiOperation(value = "查询敏感词列表", notes = "查询敏感词列表")
    @PostMapping(value = "/list")
    public Response<PageInfo<SensitiveWordVO>> list(@RequestBody SensitiveWordPageVO sensitiveWordPageVO) {
        try {
            Long companyId = ContextHolder.get().getCompanyId();
            com.yizhi.system.application.domain.SensitiveWord sensitiveWord = (com.yizhi.system.application.domain.SensitiveWord) converter.toDO(sensitiveWordPageVO, com.yizhi.system.application.domain.SensitiveWord.class);
            sensitiveWord.setCompanyId(companyId);
            PageInfo<com.yizhi.system.application.domain.SensitiveWord> pageInfo = new PageInfo<>();
            pageInfo.setPageNo(sensitiveWordPageVO.getPageNo());
            pageInfo.setPageSize(sensitiveWordPageVO.getPageSize());
            Page<com.yizhi.system.application.domain.SensitiveWord> page = sensitiveWordService.list(sensitiveWord, PageUtil.vice(pageInfo));
            //同步敏感词缓存
            TaskContext taskContext = new TaskContext(null, null, null, null, null, null, companyId);
            sensitiveTask.bindContext(taskContext);
            taskExecutor.asynExecute(sensitiveTask);
            return Response.ok(PageUtil.versa(converter.toPageVO(page, SensitiveWordVO.class)));
        } catch (Exception e) {
            logger.error("查询数据出错", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }
}
