package com.yizhi.system.application.controller.manage;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.application.orm.page.PageUtil;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.core.application.task.AbstractTaskHandler;
import com.yizhi.core.application.task.TaskExecutor;
import com.yizhi.system.application.domain.AuthzUserGroup;
import com.yizhi.system.application.domain.Company;
import com.yizhi.system.application.enums.SiteType;
import com.yizhi.system.application.exception.AuthNoManagementException;
import com.yizhi.system.application.exception.AuthNotManageException;
import com.yizhi.system.application.model.AccessSite;
import com.yizhi.system.application.model.BaseModel;
import com.yizhi.system.application.request.SiteUnselectedResq;
import com.yizhi.system.application.service.AuthService;
import com.yizhi.system.application.service.IAuthzUserGroupService;
import com.yizhi.system.application.service.ICompanyService;
import com.yizhi.system.application.service.ISiteService;
import com.yizhi.system.application.vo.*;
import com.yizhi.system.application.vo.domain.Site;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.page.PageInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 公司管理前端控制器
 * </p>
 *
 * @author FEIDA.WANG
 * @date 2018.3.8
 * @since 1.0
 */
@Api(tags = "站点管理接口")
@RestController
@RequestMapping(value = "/manage/site")
public class SiteController {

    @Autowired
    ISiteService siteService;
    @Autowired
    private AuthService authService;
    @Autowired
    DomainConverter converter;
    @Autowired
    ICompanyService companyService;
    @Autowired
    private IAuthzUserGroupService authzUserGroupService;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private TaskExecutor taskExecutor;

    private static final Logger logger = LoggerFactory.getLogger(SiteController.class);

    /**
     * 增加站点
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "插入站点信息", notes = "插入站点信息")
    @PostMapping(value = "/insert")
    public Response<String> insert(@RequestBody SiteVO siteVO) {
        try {
            BaseModel<SiteVO> model = new BaseModel<>();
            model.setObj(siteVO);
            model.setDate(new Date());
            model.setContext(ContextHolder.get());
            siteService.save(model);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 修改站点
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "修改站点信息", notes = "修改站点信息")
    @PostMapping(value = "/update")
    public Response<String> update(@RequestBody SiteVO siteVO) {
        try {
            BaseModel<SiteVO> model = new BaseModel<>();
            model.setObj(siteVO);
            model.setDate(new Date());
            model.setContext(ContextHolder.get());
            siteService.update(model);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 删除站点
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "上架站点", notes = "上架站点")
    @PostMapping(value = "/up")
    public Response<String> up(@RequestBody SiteVO siteVO) {
        try {
            siteService.up(siteVO.getId(), ContextHolder.get().getCompanyCode());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 下架站点
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "下架站点", notes = "下架站点")
    @PostMapping(value = "/down")
    public Response<String> down(@RequestBody SiteVO siteVO) {
        try {
            siteService.down(siteVO.getId(), ContextHolder.get().getCompanyCode());
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 查询站点
     *
     * @return
     */
    @ApiOperation(value = "查找站点信息", notes = "查找站点信息")
    @GetMapping(value = "/get")
    public Response<SiteVO> findById() {
        try {
            SiteVO siteVO = (SiteVO) converter.toVO(siteService.findById(ContextHolder.get().getSiteId()), SiteVO.class);
            Company company = companyService.selectById(siteVO.getCompanyId());
            if (company != null) {
                siteVO.setExpiredTime(company.getExpiredTime());
            }
            if (siteVO.getUserGroupId() != null) {
                com.yizhi.system.application.domain.AuthzUserGroup userGroup = authzUserGroupService.selectById(siteVO.getUserGroupId());
                if (userGroup != null) {
                    siteVO.setUserGroupName(userGroup.getName());
                }
            }
            return Response.ok(siteVO);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 根据公司id查询
     *
     * @return
     */
    @ApiOperation(value = "根据公司id查找站点信息", notes = "根据公司id查找站点信息")
    @GetMapping(value = "/get/comid")
    public Response<SiteVO> findByCompanyId() {
        try {
            return Response.ok((SiteVO) converter.toVOList(siteService.findByCompanyId(ContextHolder.get().getCompanyId()), SiteVO.class));
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 检查code是否存在
     *
     * @param code
     * @return
     */
    @ApiOperation(value = "检查站点code是否存在", notes = "检查站点code是否存在")
    @GetMapping(value = "/check/code")
    public Response<Boolean> checkCode(@RequestParam(name = "code") String code) {
        try {
            siteService.checkCode(0L, ContextHolder.get().getCompanyId(), code);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    /**
     * 检查名称是否存在
     *
     * @param name
     * @return
     */
    @ApiOperation(value = "检查站点name是否存在", notes = "检查站点name是否存在")
    @GetMapping(value = "/check/name")
    public Response<Boolean> checkName(@RequestParam(name = "name") String name) {
        try {
            siteService.checkName(0L, ContextHolder.get().getCompanyId(), name);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
        return Response.ok();
    }

    @ApiOperation(value = "获取站点类型列表", notes = "获取站点类型列表")
    @GetMapping(value = "/get/site/type")
    public Response<SiteTypeVO> getSiteTypes() {
        try {
            List<SiteTypeVO> types = new ArrayList<>();
            for (SiteType type : SiteType.values()) {
                types.add(new SiteTypeVO(type.getCode(), type.getName()));
            }
            return Response.ok(types);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 分页查询站点列表
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "分页查询站点列表", notes = "分页查询站点列表")
    @PostMapping(value = "/list")
    public Response<SiteVO> list(@RequestBody SitePageVO siteVO) {
        try {
            com.yizhi.system.application.domain.Site site = (com.yizhi.system.application.domain.Site) converter.toDO(siteVO, com.yizhi.system.application.domain.Site.class);
            site.setCompanyId(ContextHolder.get().getCompanyId());
            site.setEnabled(siteVO.getEnabled());
            if (StringUtils.isNotBlank(siteVO.getName())) {
                site.setName(siteVO.getName());
                site.setCode(siteVO.getName());
            }
            Long accessSiteId = ContextHolder.get().getSiteId();

            PageInfo<com.yizhi.system.application.domain.Site> pageInfo = new PageInfo<>();
            pageInfo.setPageNo(siteVO.getPageNo());
            pageInfo.setPageSize(siteVO.getPageSize());
            Page<com.yizhi.system.application.domain.Site> page = siteService.list(site, PageUtil.vice(pageInfo), accessSiteId);

            PageInfo<SiteVO> resPageInfo = PageUtil.versa(converter.toPageVO(page, SiteVO.class));
            List<SiteVO> resRecords = resPageInfo.getRecords();
            if (CollectionUtils.isNotEmpty(resRecords)) {
                AuthzUserGroup userGroup;
                for (SiteVO resRecord : resRecords) {
                    if (resRecord.getUserGroupId() != null) {
                        userGroup = authzUserGroupService.selectById(resRecord.getUserGroupId());
                        if (userGroup != null) {
                            resRecord.setUserGroupName(userGroup.getName());
                        }
                    }
                }
            }
            return Response.ok(resPageInfo);
        } catch (BizException be) {
            return Response.fail(be.getCode(), be.getMsg());
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    @ApiOperation(value = "课程授权获取站点列表", response = SiteVO.class, notes = "课程授权获取站点列表")
    @GetMapping(value = "/course/auth/sites")
    public Response<List<SiteVO>> getAuthSites() {
        // 获取上下文
        RequestContext requestContext = ContextHolder.get();
        List<com.yizhi.system.application.domain.Site> data = siteService.courseAuthOtherSite(requestContext);
        List<SiteVO> siteVOS = new ArrayList<>();
        if (CollectionUtils.isEmpty(data)) {
            return null;
        }
        for (com.yizhi.system.application.domain.Site item : data) {
            SiteVO vo = new SiteVO();
            BeanUtils.copyProperties(item, vo);
            siteVOS.add(vo);
        }
        return Response.ok(siteVOS);
    }


    @ApiOperation(value = "多站点管理-设置管理员")
    @PostMapping(value = "/admin/set")
    public Response<Integer> setAdmin(
            @ApiParam("站点设置管理员") @RequestBody SiteAdminVo siteAdminVo
    ) {
        try {
            BaseModel<SiteAdminVo> model = new BaseModel<>();
            model.setObj(siteAdminVo);
            model.setDate(new Date());
            model.setContext(ContextHolder.get());
            return Response.ok(siteService.setSiteAdmin(model));
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }


    @ApiOperation(value = "多站点管理-查询管理员")
    @GetMapping(value = "/admin/find")
    public Response<AccountVO> findAdmin(@ApiParam("站点id") @RequestParam Long siteId) {
        try {
            return Response.ok(converter.toVOList(siteService.getSiteAdmin(siteId), AccountVO.class));
        } catch (Exception e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    @ApiOperation(value = "获取权限", response = AccessSite.class)
    @GetMapping("/menus/get")
    public Response<List<AccessSite>> getSiteMenus() {
        try {
            RequestContext context = ContextHolder.get();
            List<AccessSite> result = authService.getAccessSiteWithMenu(context.getCompanyCode(), context.getAccountId(), context.getSiteId());
            return Response.ok(result);
        } catch (AuthNoManagementException e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        } catch (AuthNotManageException e) {
            logger.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    @ApiOperation(value = "获取所有站点列表", response = Site.class)
    @GetMapping("/get/site/list")
    public Response<List<CompanySiteVO>> getSiteList() {
        List<CompanySiteVO> siteList = null;
        try {
            siteList = siteService.getSiteList();
        } catch (Exception e) {
            logger.info("站点查询异常{}", e.getMessage());
        }
        return Response.ok(siteList);
    }

    @ApiOperation(value = "条件查询站点列表")
    @GetMapping("/get/unselected/site/list")
    public Response<List<SiteProtectDefinedVO>> getUnselectedSiteList(SiteUnselectedResq siteReq) {
        logger.info("站点维护查询站点信息入参: {}",siteReq.toString());
        Page<SiteProtectDefinedVO> page = siteService.getUnselectedSiteList(siteReq);
        return Response.ok(page);
    }

    /**
     * 分页查询站点列表
     *
     * @return
     */
    @ApiOperation(value = "分页查询站点列表", notes = "分页查询站点列表")
    @PostMapping(value = "/getPage")
    public Response<Page<SiteVO>> getPage(@RequestBody TrSitePageVO trSitePageVO) {

        com.yizhi.system.application.domain.Site site = (com.yizhi.system.application.domain.Site) converter.toDO(trSitePageVO, com.yizhi.system.application.domain.Site.class);
        site.setName(trSitePageVO.getKeyword());
        Page<com.yizhi.system.application.domain.Site> page = new Page<>(trSitePageVO.getPageNo(), trSitePageVO.getPageSize());
        Page<SiteVO> voPage = new Page<>(trSitePageVO.getPageNo(), trSitePageVO.getPageSize());
        page = siteService.getPage(site, page);
        List<com.yizhi.system.application.domain.Site> records = page.getRecords();
        if (CollectionUtils.isNotEmpty(records)) {
            ArrayList<SiteVO> list = new ArrayList<>(records.size());
            records.forEach(a -> {
                list.add((SiteVO) converter.toDO(a, SiteVO.class));
            });
            voPage.setRecords(list);
            voPage.setTotal(page.getTotal());
        }
        return Response.ok(voPage);
    }

    /**
     * 分页查询站点列表
     *
     * @return
     */
    @ApiOperation(value = "初始化缺失的站点登录code")
    @PostMapping(value = "/init/siteLoginCode")
    public void initSiteLoginCode() {
        siteService.initSiteLoginCode();
    }

}
