package com.yizhi.system.application.controller.manage;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.system.application.constant.SiteProtectConstants;
import com.yizhi.system.application.constant.SiteProtectExceptionEnum;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.system.application.domain.dto.SiteProtectTaskDTO;
import com.yizhi.system.application.exception.SiteProtectBusinessException;
import com.yizhi.system.application.request.QuerySiteProtectResq;
import com.yizhi.system.application.request.SiteProtectDelReq;
import com.yizhi.system.application.request.SiteProtectExportReq;
import com.yizhi.system.application.request.SiteProtectReq;
import com.yizhi.system.application.service.SiteProtectService;
import com.yizhi.system.application.task.SiteProtectExport;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.page.PageInfo;
import com.yizhi.system.application.vo.SiteProtectDetailVO;
import com.yizhi.system.application.vo.SiteProtectNoticeVO;
import com.yizhi.system.application.vo.SiteProtectUserVO;
import com.yizhi.system.application.vo.SiteProtectVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;

@RestController
@RequestMapping("/manage/site/protect")
@Slf4j
@Api(tags = "站点维护")
public class SiteProtectController {

    @Autowired
    SiteProtectService siteProtectService;
    @Autowired
    RedisCache redisCache;
    @Autowired
    SiteProtectExport siteProtectExport;

    @PostMapping("/querylist")
    @ApiOperation(value = "查询站点维护列表")
    public Response<SiteProtectVO> queryList(@RequestBody QuerySiteProtectResq querySiteProtectResq){
        log.info("SiteProtectController............queryList:----------入参-----{}", JSON.toJSONString(querySiteProtectResq));
        if(querySiteProtectResq.getPageSize()==null||0==querySiteProtectResq.getPageSize()){
            querySiteProtectResq.setPageSize(10);
        }
        if(querySiteProtectResq.getPageNo()==null||0==querySiteProtectResq.getPageNo()){
            querySiteProtectResq.setPageNo(1);
        }
        PageInfo<SiteProtectVO> siteProtectDTOPageInfo = siteProtectService.queryList(querySiteProtectResq);
        return Response.ok(siteProtectDTOPageInfo);
    }

    @PostMapping("/add")
    @ApiOperation(value = "新增站点维护")
    public Response<String> add(@RequestBody SiteProtectReq siteProtectReq){
        log.info("[站点维护] 新增维护入参=============：{}",siteProtectReq.toString());
        RequestContext context = ContextHolder.get();
        String checkResult = siteProtectReq.checkFieldLimitCondition();
        if (!"ok".equals(checkResult)){
            return setError(checkResult);
        }
        String dataValid = siteProtectReq.checkDataValid();
        if (!"ok".equals(dataValid)){
            return setError(dataValid);
        }
        String formatDate = siteProtectReq.formatDate();
        if (!"ok".equals(formatDate)){
            return setError(formatDate);
        }
        String startTimeValid = siteProtectReq.checkStartTimeValid();
        if (!"ok".equals(startTimeValid)){
            return setError(startTimeValid);
        }
        String checkEndTimeValid = siteProtectReq.checkEndTimeValid();
        if (!"ok".equals(checkEndTimeValid)){
            return setError(checkEndTimeValid);
        }
        siteProtectReq.setCreateById(context.getAccountId());
        siteProtectReq.setCreateByName(context.getAccountName());
        try {
            Long protectId = siteProtectService.addSiteProtect(siteProtectReq);
            if (null == protectId){
                return Response.fail(SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getMsg());
            }
            return Response.ok();
        }catch (SiteProtectBusinessException siteException){
            String errCode = siteException.getErrCode();
            if (!SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getErrCode().equals(errCode)){
                redisCache.delete(SiteProtectConstants.SITE_PROTECT_ADD_REDIS_PREF + siteProtectReq.getName());
            }
            log.info("[新增站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 新增实体 siteProtectReq = {} ,站点维护返回信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString(),siteException.getMessage());
            return Response.fail(siteException.getErrCode(),siteException.getMsg());
        }catch (Exception e){
            redisCache.delete(SiteProtectConstants.SITE_PROTECT_ADD_REDIS_PREF + siteProtectReq.getName());
            log.error("[新增站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 新增实体 siteProtectReq = {} ,异常信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString(),e);
            return Response.fail(SiteProtectExceptionEnum.PROTECT_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_FAIL.getMsg());
        }
    }

    //设置错误信息
    private Response<String> setError(String errMsg) {
        RequestContext context = ContextHolder.get();
        log.info("[站点维护][siteId = {} ,siteCode = {},companyId = {} , accountId = {}] "
                ,context.getSiteId(),context.getSiteCode()
                ,context.getCompanyId(),context.getAccountId());
        return Response.fail(SiteProtectExceptionEnum.PROTECT_DATA_INVALID.getErrCode(),errMsg);
    }

    @PostMapping("/update")
    @ApiOperation(value = "修改站点维护")
    public Response<String> update(@RequestBody SiteProtectReq siteProtectReq){
        log.info("[站点维护] 修改维护入参=============：{}",siteProtectReq.toString());
        RequestContext context = ContextHolder.get();
        String checkResult = siteProtectReq.checkFieldLimitCondition();
        if (!"ok".equals(checkResult)){
            return setError(checkResult);
        }
        String dataValid = siteProtectReq.checkDataValid();
        if (!"ok".equals(dataValid)){
            return setError(dataValid);
        }
        String formatDate = siteProtectReq.formatDate();
        if (!"ok".equals(formatDate)){
            return setError(formatDate);
        }

        siteProtectReq.setCreateById(context.getAccountId());
        siteProtectReq.setCreateByName(context.getAccountName());
        try {
            Long protectId = siteProtectService.updateSiteProtect(siteProtectReq);
            if (null == protectId){
                return Response.fail(SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getMsg());
            }
            return Response.ok();
        }catch (SiteProtectBusinessException siteException){
            String errCode = siteException.getErrCode();
            if (!SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getErrCode().equals(errCode)){
                redisCache.delete(SiteProtectConstants.SITE_PROTECT_UP_REDIS_PREF + siteProtectReq.getId());
            }
            log.info("[修改站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 修改实体  siteProtectReq = {} ,站点维护返回信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString(),siteException.getMessage());
            return Response.fail(siteException.getErrCode(),siteException.getMsg());
        }catch (Exception e){
            redisCache.delete(SiteProtectConstants.SITE_PROTECT_UP_REDIS_PREF + siteProtectReq.getId());
            log.error("[修改站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 修改实体 siteProtectReq = {} ,异常信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString(),e);
            return Response.fail(SiteProtectExceptionEnum.PROTECT_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_FAIL.getMsg());
        }
    }

    @PostMapping("/delete")
    @ApiOperation(value = "删除站点维护")
    public Response<String> update(@RequestBody SiteProtectDelReq protectDelReq){
        log.info("[站点维护] 删除维护入参=============：{}",protectDelReq.toString());
        RequestContext context = ContextHolder.get();
        try {
            siteProtectService.deleteSiteProtect(protectDelReq);
            return Response.ok();
        }catch (SiteProtectBusinessException siteException){
            log.info("[删除站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] ，protectDelReq = {} ,站点维护返回信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),protectDelReq.toString(),siteException.getMessage());
            return Response.fail(siteException.getErrCode(),siteException.getMsg());
        }catch (Exception e){
            log.error("[修改站点维护异常siteId = {} ,siteCode = {},companyId = {} , accountId = {}] protectDelReq = {} ,异常信息：{}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),protectDelReq.toString(),e);
            return Response.fail(SiteProtectExceptionEnum.PROTECT_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_FAIL.getMsg());
        }
    }

    @PostMapping("/export")
    public Response<String> export(@RequestBody SiteProtectExportReq siteProtectExportReq){
        if(CollectionUtils.isEmpty(siteProtectExportReq.getSiteNoticeIdList())){
            return Response.fail(SiteProtectExceptionEnum.NO_CHOOSE_SITE.getMsg());
        }
        RequestContext res = ContextHolder.get();
        Date date = new Date();
        Long taskId = date.getTime();
        String taskName = "导出站点维护-"+taskId;
        String result = "任务名称："+taskName;
        SiteProtectTaskDTO siteProtectTaskDTO = new SiteProtectTaskDTO();
        siteProtectTaskDTO.setTaskId(taskId);
        siteProtectTaskDTO.setTaskName(taskName);
        siteProtectTaskDTO.setSiteNoticeIdList(siteProtectExportReq.getSiteNoticeIdList());
        siteProtectTaskDTO.setRequestContext(res);
        siteProtectExport.execute(siteProtectTaskDTO, true);
        return Response.ok(result);
    }

    @ApiOperation("查询站点维护详情")
    @GetMapping("/detail/{id}")
    public Response detail(@ApiParam("站点维护记录id") @PathVariable("id") Long id){
        if(id==null||0==id){
            return Response.fail(SiteProtectExceptionEnum.NO_DETAIL_ID.getErrCode(),SiteProtectExceptionEnum.NO_DETAIL_ID.getMsg());
        }
        SiteProtectDetailVO result = null;
        try {
            result = siteProtectService.queryOne(id);
        } catch (SiteProtectBusinessException e) {
            log.error("异常信息---------站点维护记录id:{},--------异常信息:--------{}",id,e);
            return Response.fail(SiteProtectExceptionEnum.NO_DETAIL_ID.getErrCode(),SiteProtectExceptionEnum.NO_DETAIL_ID.getMsg());
        }catch (Exception e) {
            log.error("异常信息---------站点维护记录id:{},--------异常信息:--------{}",id,e);
            return Response.fail(SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getErrCode(),SiteProtectExceptionEnum.PROTECT_UNKNOWN_FAIL.getMsg());
        }
        return Response.ok(result);
    }


    @GetMapping("/user/protect/list")
    public String userProtectList(){

        RequestContext context = ContextHolder.get();
        Long siteId = context.getSiteId();
        Long companyId = context.getCompanyId();
        try {
            List<SiteProtectUserVO> userProtect = siteProtectService.getUserProtect(siteId, companyId);
            if (null != userProtect && !userProtect.isEmpty()){
                SiteProtectUserVO userVO = userProtect.get(0);
                List<SiteProtectNoticeVO> noticeList = userVO.getNoticeList();
                if (null != noticeList && noticeList.size()>0){
                    String message = noticeList.get(0).getMessage();
                    JSONObject object = new JSONObject();
                    object.put("msg",message);
                    return message;
                }
            }
        }catch (SiteProtectBusinessException protectException){
            log.info("站点维护message{}",protectException.getMsg());
            return "";

        }catch (Exception e){
            log.error("站点维护message{}",e.getMessage());
            return "";
        }
        return "";
    }

}
