package com.yizhi.system.application.controller.remote;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.enums.SqlLike;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.system.application.controller.manage.OrganizationController;
import com.yizhi.system.application.domain.*;
import com.yizhi.system.application.mapper.AccountMapper;
import com.yizhi.core.application.vo.DroolsVo;
import com.yizhi.system.application.model.AccountNameForIdsModel;
import com.yizhi.system.application.model.RangeModel;
import com.yizhi.system.application.model.SiteOrgIdModel;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.page.PageUtil;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.system.application.service.*;
import com.yizhi.system.application.util.OrgIndexUtil;
import com.yizhi.system.application.util.RightLeftIndex;
import com.yizhi.util.application.domain.Response;
import com.yizhi.util.application.encrypt.ShaEncrypt;
import com.yizhi.util.application.page.PageInfo;
import com.yizhi.system.application.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.text.DateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 账号远程调用接口
 *
 * @author yinyuyan
 */
@RestController
@RequestMapping(value = "/remote/account")
@Slf4j
public class RemoteAccountController {

    private static final Logger LOG = LoggerFactory.getLogger(RemoteAccountController.class);

    @Autowired
    IAccountService accountService;
    @Autowired
    private AuthorityService authorityService;
    @Autowired
    private DomainConverter converter;
    @Autowired
    RedisCache redisCache;
    @Autowired
    IdGenerator idGenerator;
    @Autowired
    IOrganizationService iOrganizationService;
    @Autowired
    private ISiteService siteService;
    @Autowired
    private ICompanyService companyService;
    @Autowired
    private IUserWechatService userWechatService;
    @Autowired
    private AccountMapper accountMapper;

    /**
     * 更新账号
     *
     * @param accountVO
     * @return
     */
    @PostMapping(value = "/update")
    public Boolean updateAccount(@RequestBody AccountVO accountVO) {
        RequestContext context = ContextHolder.get();
        Account account = new Account();
        BeanUtils.copyProperties(accountVO, account);
        boolean b = false;
        try {
            b = accountService.updateById(account);
        } catch (Exception e) {
            e.printStackTrace();
            LOG.error("更新账号异常,调用方ip:" + context.getRemoteAddr());
        }
        return b;

    }

    @GetMapping(value = "/name/search")
    public List<AccountVO> fuzzySearchAccountByName(String name, Long companyId) {
        return converter.toVOList(accountService.fuzzySearchByName(name, companyId), AccountVO.class);
    }

    @PostMapping(value = "/add")
    public boolean addAccount(@RequestBody AccountRegisterVO accountVO) {
        Account findAccount = accountService.find(accountVO.getCompanyId(), accountVO.getName());
        if (findAccount != null) {
            LOG.info("方法名:addAccount" +"已存在相应用户，不新增" + JSON.toJSONString(findAccount));
            return false;
        }
        LOG.info("wachat 新增用户信息=" + JSON.toJSONString(accountVO));
        Account account = new Account();
        BeanUtils.copyProperties(accountVO, account);
        //3.密码加密
        account.setPassword(ShaEncrypt.encryptNewPassword(account.getPassword()));
        //4.生成账号id
        Long accountId = idGenerator.generate();
        account.setId(accountId);
        accountService.insert(account);
        LOG.info("方法名:addAccount" + "--新增的一个用户：" + JSON.toJSONString(account));
        return true;
    }


    @PostMapping(value = "/add/get")
    public AccountVO addAccountGet(@RequestBody AccountRegisterVO accountVO) {
        Account findAccount = accountService.find(accountVO.getCompanyId(), accountVO.getName());
        if (findAccount != null) {
            LOG.info("方法名:addAccount" +"已存在相应用户，不新增" + JSON.toJSONString(findAccount));
            return (AccountVO) converter.toVO(findAccount, AccountVO.class);
        }
        LOG.info("wachat 新增用户信息=" + JSON.toJSONString(accountVO));
        Account account = new Account();
        BeanUtils.copyProperties(accountVO, account);
        //3.密码加密
        account.setPassword(ShaEncrypt.encryptNewPassword(account.getPassword()));
        //4.生成账号id
        Long accountId = idGenerator.generate();
        account.setId(accountId);
        boolean addStatus = accountService.insert(account);
        if (addStatus) {
            LOG.info("方法名:addAccount" + "--新增的一个用户：" + JSON.toJSONString(account));
            return (AccountVO) converter.toVO(accountService.selectById(accountId), AccountVO.class);
        } else {
            return null;
        }

    }

    @GetMapping(value = "/access/site")
    public boolean isAccessToSite(@RequestParam Long accountId, @RequestParam Long siteId) {
        return accountService.isAccessToSite(accountId, siteId);
    }

    @GetMapping(value = "/get")
    public AccountVO get(Long id) {
        List<Long> ids = new ArrayList<>();
        ids.add(id);
        Account account = accountService.selectById(id);
        AccountVO accountVO = new AccountVO();
        if (account != null) {
            BeanUtils.copyProperties(account, accountVO);
            Organization organization = iOrganizationService.selectById(accountVO.getOrgId());
            if (Objects.nonNull(organization)) {
                accountVO.setOrgName(organization.getName());
            }

            return accountVO;
        }
        return null;
    }

    @PostMapping(value = "/ids/get")
    public List<AccountVO> idsGet(@RequestBody List<Long> ids) {
        return accountService.idsGet(ids);
    }

    @PostMapping(value = "/list")
    public List<AccountVO> selectByIds(@RequestBody List<Long> ids) {
        return converter.toVOList(accountService.getAccountByIds(ids), AccountVO.class);
    }

    /**
     * 获取站点下账号
     *
     * @param siteId
     * @return
     */
    @GetMapping(value = "/list/siteid")
    public List<AccountVO> getAccountBySiteId(@RequestParam Long siteId) {
        return converter.toDOList(accountService.selectBySiteId(siteId), AccountVO.class);
    }

    /**
     * 根据部门名称查询员工信息
     *
     * @param orgName
     * @param companyId
     * @return
     */
    @GetMapping(value = "/list/orgname")
    public List<AccountVO> getAccountByOrgName(@RequestParam String orgName, @RequestParam Long companyId) {
        return converter.toDOList(accountService.getAccountByOrgName(orgName, companyId), AccountVO.class);
    }

    /**
     * 根据accountId，查询siteId站点下包含accountId的部门id列表和用户组id列表
     *
     * @param accountId
     * @param siteId
     * @return
     */
    @GetMapping(value = "/list/chief/ids")
    public List<Long> getCheifIds(@RequestParam("accountId") Long accountId, @RequestParam("siteId") Long siteId) {
        return authorityService.getAuthOrgAndGroup(accountId, siteId);
    }

    /**
     * 取部门下所有的用户
     *
     * @param orgId
     * @param companyId
     * @return
     */
    @GetMapping(value = "/list/orgid")
    public List<AccountVO> getAccountByOrgId(@RequestParam Long orgId, @RequestParam Long companyId) {
        return converter.toDOList(accountService.selectByOrgId(orgId, companyId), AccountVO.class);
    }

    /**
     * 取用户组下所有的用户
     *
     * @param groupId
     * @param companyId
     * @return
     */
    @GetMapping(value = "/list/groupid")
    public List<AccountVO> getAccountByGroupId(@RequestParam Long groupId, @RequestParam Long companyId) {
        return converter.toDOList(accountService.selectByGroupId(groupId, companyId), AccountVO.class);
    }

    /**
     * 根据用户名，姓名，或部门名称查询账号列表
     *
     * @param userInfoVO
     * @return
     */
    @PostMapping(value = "/list/userinfo")
    public Map<Long, UserInfoVO> selectUserInfo(@RequestBody UserInfoVO userInfoVO) {
        return accountService.selectUserInfo(userInfoVO);
    }

    @PostMapping(value = "/list/getUser")
    public Map<Long, UserInfoVO> getUser(@RequestBody UserInfoVO userInfoVO) {
        return accountService.getUser(userInfoVO);
    }

    /**
     *
     * 获取用户列表 用户名和组织id
     * @param userInfoVO
     * @return
     */
    @PostMapping(value = "/find/userInfo/list")
    public List<Account> getUserInfoVOList(@RequestBody UserInfoVO userInfoVO){
        return accountService.getUserInfoVOList(userInfoVO);


    }

    /**
     * 排序用户的id
     * @param accountIdList
     * @return
     */
    @PostMapping(value = "/find/accountId/list")
   public List<Long> selectAccountIdList(@RequestBody List<Long> accountIdList){
        return accountService.selectAccountIdList(accountIdList);
    }

    /**
     * @param accountVO
     * @return
     */
    @PostMapping(value = "/list/page")
    public PageInfo<AccountVO> list(@RequestBody AccountPageVO accountVO) {
        if (null == accountVO.getOrgId() || accountVO.getOrgId() == 0L) {
            accountVO.setOrgId(ContextHolder.get().getOrgId());//
        }
        accountVO.setCompanyId(ContextHolder.get().getCompanyId());//
        PageInfo<Account> pageInfo = new PageInfo<>();
        pageInfo.setPageNo(accountVO.getPageNo());
        pageInfo.setPageSize(accountVO.getPageSize());
        Page<Account> page = accountService.list((Account) converter.toDO(accountVO, Account.class), PageUtil.vice(pageInfo));
        return PageUtil.versa(converter.toPageVO(page, AccountVO.class));
    }

    /**
     * 查询站点的管理员
     * @param accountVO
     * @return
     */
    @PostMapping(value = "/manager/page")
    public PageInfo<AccountVO> mangerList(@RequestBody AccountPageVO accountVO) {
        accountVO.setCompanyId(ContextHolder.get().getCompanyId());
        Page<Account> page = accountService.managerList(accountVO);
        return PageUtil.versa(converter.toPageVO(page, AccountVO.class));
    }

    /**
     * 根据相关类型和id查询用户信息
     * key为count时，value为人数
     * key为account时，value为用户列表
     *
     * @return
     */
    @PostMapping(value = "/find/members")
    public Map<String, Object> findMemberInfo(@RequestBody List<RangeModel> models) {
        return accountService.findMemberInfo(models);
    }

    /**
     * 查取站点下所有的员工(管辖区指定调整）
     *
     * @param orgId
     * @param companyId
     * @return
     */
    @GetMapping(value = "/find/orgid")
    public List<AccountVO> accountsFindByOrgId(@RequestParam Long orgId, @RequestParam Long companyId) {
        EntityWrapper<Account> ew = QueryUtil.condition(new Account());
        ew.where("company_id={0}", companyId).and("org_id={0}", orgId).and("enabled={0}", 1);
        return converter.toDOList(accountService.selectList(ew), AccountVO.class);
    }

    @GetMapping("/site/count")
    public Integer getSiteAccountNum(@RequestParam("siteId") Long siteId, @RequestParam("companyId") Long companyId) {
        try {
            LOG.info("******************************************站点id={},企业id={}统计站点下员工数量，", siteId, companyId);
            return accountService.getSiteAccountNum(siteId, companyId);
        } catch (Exception e) {
            LOG.info("错误信息= {}", e);
            return 0;
        }
    }

    @GetMapping(value = "/site/org/count")
    public Integer getSiteOrgWithChildAccountNum(@RequestBody SiteOrgIdModel model) {
        Long siteId = model.getSiteId();
        List<Long> orgIds = model.getOrgIds();
        if (CollectionUtils.isEmpty(orgIds)) {
            return 0;
        }
        Site site = siteService.selectById(siteId);
        if (null != site) {
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(iOrganizationService.selectBatchIds(orgIds));
            if (CollectionUtils.isEmpty(indices)) {
                return 0;
            }
            return accountMapper.orgAccountListCount(indices, 1, site.getCompanyId());
        }
        return 0;
    }


    @GetMapping(value = "/card/get")
    public AccountVO findCardById(@RequestParam(value = "accountId") long accountId) {
        try {
            AccountVO accountVO = (AccountVO) converter.toVO(accountService.getCard(accountId), AccountVO.class);
            return accountVO;
        } catch (Exception e) {
            return null;
        }
    }

    @GetMapping(value = "/accountIds/keyword/get")
    public List<Long> findAIdsByKeyword(@RequestParam("key") String key,
                                        @RequestParam("companyId") Long companyId) {

        RequestContext requestContext = ContextHolder.get();
        EntityWrapper<Account> ew = new EntityWrapper<>(new Account());
        ew.where("company_id={0}", companyId);
        //ew.eq("site_id",requestContext.getSiteId());
        List<Long> orgIds = requestContext.getOrgIds();
        if (!CollectionUtils.isEmpty(orgIds)) {
            ew.in("org_id", orgIds);
        } else {
            Site site = siteService.selectById(requestContext.getSiteId());
            Set<Long> orgs = iOrganizationService.getOrgIdsInSite(site.getId(), requestContext.getCompanyCode(),
                    site.getCode());
            ew.in("org_id", orgs);
        }

        ew.andNew().like("name", key, SqlLike.DEFAULT).
                or().like("full_name", key, SqlLike.DEFAULT);
        List<Account> findData = accountService.selectList(ew);
        if (CollectionUtils.isNotEmpty(findData)) {
            return findData.stream().map(obj -> obj.getId()).collect(Collectors.toList());
        }
        return null;
    }

    @PostMapping(value = "/accountIds/keyword/byVo")
    public List<Long> findAIdsByKeywordByVo(@RequestBody AccountIdsByKeyReqVO vo) {
        EntityWrapper<Account> ew = new EntityWrapper<>(new Account());
        ew.where("company_id={0}", vo.getCompanyId());
        //ew.eq("site_id",vo.getSiteId());
        List<Long> orgIds = vo.getOrgIds();
        if (!CollectionUtils.isEmpty(orgIds)) {
            ew.in("org_id", orgIds);
        } else {
            Site site = siteService.selectById(vo.getSiteId());
            Company company = companyService.selectById(vo.getCompanyId());
            Set<Long> orgs = iOrganizationService.getOrgIdsInSite(site.getId(), company.getCode(),
                    site.getCode());
            ew.in("org_id", orgs);
        }
        ew.andNew().like("name", vo.getKey(), SqlLike.DEFAULT).
                or().like("full_name", vo.getKey(), SqlLike.DEFAULT);
        List<Account> findData = accountService.selectList(ew);
        if (CollectionUtils.isNotEmpty(findData)) {
            return findData.stream().map(obj -> obj.getId()).collect(Collectors.toList());
        }
        return null;
    }

    /**
     * 企业微信的用户的新增
     *
     * @param weChatUserVO 参数
     * @return 用户实体
     */
    @PostMapping(value = "/insert/cwechatuser")
    public Account saveUser(@RequestBody WeChatUserVO weChatUserVO) {
        LOG.info("保存微信的用户信息：{}", JSON.toJSON(weChatUserVO));
        Account account = userWechatService.saveUser(weChatUserVO);
        return account;
    }

    /**
     * 查询用户的相信信息
     *
     * @param name      用户名称
     * @param companyId 公司id
     * @return
     */
    @RequestMapping(value = "/name/find", method = RequestMethod.GET)
    public AccountVO findUser(@RequestParam("name") String name, @RequestParam("companyId") Long companyId) {
        return accountService.findUser(name, companyId);
    }

    @PostMapping(value = "/find/accountOrgs")
    public List<AccountOrgsVO> findCcountOrgs(@RequestBody AccountOrgsRepVO vo) {
        List<AccountOrgsVO> retData = new ArrayList<>();
        EntityWrapper<Account> ew = new EntityWrapper<>(new Account());

        int idsType = vo.getIdsType();
        List<Long> accountIds = vo.getIds();
        if (idsType == 1) {
            ew.where("1=1").in("id", accountIds);
        } else if (idsType == 2) {
            ew.where("1=1").in("org_id", accountIds);
        } else {
            ew.where("1=1").andNew().in("id", accountIds).or().in("org_id", accountIds);
        }
        List<Account> findData = accountService.selectList(ew);
        Boolean isFindParent = vo.getIsFindParent();
        if (isFindParent == null || !isFindParent) {
            List<Long> orgIds = findData.stream().map(obj -> obj.getOrgId()).collect(Collectors.toList());
            List<Organization> orgsInfo = iOrganizationService.selectBatchIds(orgIds);
            Map<Long, Organization> dataMap = orgsInfo.stream().collect(Collectors.toMap(key -> key.getId(), val -> val));
            for (Account obj : findData) {
                AccountOrgsVO accountOrgsVO = new AccountOrgsVO();
                BeanUtils.copyProperties(obj, accountOrgsVO);
                accountOrgsVO.setOrgName(dataMap.get(obj.getOrgId()).getName());
                String status = getAccountStatus(obj);
                accountOrgsVO.setStatus(status);
                retData.add(accountOrgsVO);
            }
        } else {
            for (Account obj : findData) {
                AccountOrgsVO accountOrgsVO = new AccountOrgsVO();
                BeanUtils.copyProperties(obj, accountOrgsVO);
                List<String> names = iOrganizationService.listParentNameIds(obj.getOrgId());
                accountOrgsVO.setOrgNames(names);
                int size = names.size();
                accountOrgsVO.setOrgName(names.get(size - 1));
                String status = getAccountStatus(obj);
                accountOrgsVO.setStatus(status);
                retData.add(accountOrgsVO);
            }
        }
        return retData;
    }

    /**
     * 根据用户名来查询账号 id，包括数据权限的校验
     *
     * @param model
     * @param companyId
     * @param siteId
     * @param moduleType
     * @return
     */
    @PostMapping("/name/id/get")
    public Map<String, Long> getAccountIdByNames(@RequestBody AccountNameForIdsModel model,
                                                 @RequestParam("companyId") Long companyId,
                                                 @RequestParam("siteId") Long siteId,
                                                 @RequestParam("moduleType") String moduleType) {
        List<String> names = model.getNames();
        RequestContext managerContext = model.getContext();
        return accountService.getAccountIdByNames(names, companyId, siteId, managerContext, moduleType);
    }

    /**
     * 丁小伟添加,外部经验导入需要用户的部门id和部门名称
     *
     * @param model
     * @param companyId
     * @param siteId
     * @param moduleType
     * @return
     */
    @PostMapping("/name/account/get")
    Map<String, Map<String, Object>> getAccountByNames(@RequestBody AccountNameForIdsModel model,
                                                       @RequestParam("companyId") Long companyId,
                                                       @RequestParam("siteId") Long siteId,
                                                       @RequestParam("moduleType") String moduleType) {
        List<String> names = model.getNames();
        RequestContext managerContext = model.getContext();
        return accountService.getAccountByNames(names, companyId, siteId, managerContext, moduleType);
    }

    private static String getAccountStatus(Account obj) {
        String status = "正常";
        if (!obj.getEnabled()) {
            status = "未启用";
            // 前面时间大于后面时间==1
        } else if (obj.getLocked()) {
            status = "锁定";
        } else if (obj.getExpiredType() == 2 && obj.getEndTime().compareTo(new Date()) <= 0) {
            status = "超期";
        } else if (obj.getExpiredType() == 3 && obj.getExpiredTime().compareTo(new Date()) <= 0) {
            status = "过时";
        }
        return status;
    }

    @PostMapping(value = "/add/swhy")
    public AccountVO addSwhyAccount(@RequestBody com.yizhi.system.application.vo.domain.Account account) {
        //密码加密
        account.setPassword(ShaEncrypt.encryptNewPassword(account.getPassword()));
        Account findAccount = accountService.find(account.getCompanyId(), account.getName());
        if (findAccount != null) {
            //更新
            findAccount.setPassword(account.getPassword());
            findAccount.setFullName(account.getFullName());
            findAccount.setEmail(account.getEmail());
            findAccount.setUpdateTime(account.getCreateTime());
            findAccount.setUpdateById(account.getCreateById());
            findAccount.setUpdateByName(account.getCreateByName());
            findAccount.setIdNumber(account.getIdNumber());
            findAccount.setMobile(account.getMobile());
            accountService.updateById(findAccount);

            AccountVO accountVO = new AccountVO();
            BeanUtils.copyProperties(findAccount, accountVO);

            return accountVO;
        }
        //生成账号id
        Long accountId = idGenerator.generate();
        account.setId(accountId);
        Account account1 = new Account();
        BeanUtils.copyProperties(account,account1);
        boolean addStatus = accountService.insert(account1);
        if (addStatus) {
            return (AccountVO) converter.toVO(accountService.selectById(accountId), AccountVO.class);
        } else {
            return null;
        }

    }

    @PostMapping(value = "/disable/swhy")
    public Boolean disableSwhyAccount(@RequestParam("companyId") Long companyId, @RequestParam("userName") String userName) {
        Account findAccount = accountService.find(companyId, userName);
        if (findAccount == null) {
            return false;
        }

        AccountEnableLog log = new AccountEnableLog();
        log.setAccountId(findAccount.getId());
        Boolean isOk = accountService.close(log);
        return isOk;
    }

    @PostMapping(value = "/site/org/counts")
    public List<Account> getSiteOrgWithChildAccountNums(@RequestBody SiteOrgIdModel model) {
        Long siteId = model.getSiteId();
        List<Long> orgIds = model.getOrgIds();
        if (CollectionUtils.isEmpty(orgIds)) {
            return null;
        }
        // 获取站点下的所有部门，并组成树形结构
        Site site = siteService.selectById(siteId);
        if (null != site) {
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(iOrganizationService.selectBatchIds(orgIds));
            if (CollectionUtils.isEmpty(indices)) {
                return null;
            }
            return accountMapper.orgAccountList(indices, 1, site.getCompanyId());
        }
        return null;
    }

    @GetMapping("/org/isEnable/list")
    public List<Long> listAccountIdByOrgIdAndEnable(@RequestParam("orgId") Long orgId, @RequestParam("isEnable") Boolean isEnable) {
        return accountService.listAccountIdByOrgIdAndEnable(orgId, isEnable);
    }

    @GetMapping("/getAccountByMobile")
    public List<Account> getAccountByMobile(@RequestParam("mobile") String mobile) {
        Account account = new Account();
        account.setMobile(mobile);
        account.setEnabled(true);
        EntityWrapper<Account> wrapper = new EntityWrapper<Account>(account);
        List<Account> list = accountService.selectList(wrapper);
        return list;
    }

    @GetMapping("getAccountCount")
    public Map<Long, Integer> getAccountCount(@RequestParam("companyId") Long companyId) {
        Map<Long, Integer> map = new HashMap<Long, Integer>();
        List<AccountCountVO> list = accountService.getAccountCount(companyId);
        for (AccountCountVO vo : list) {
            map.put(vo.getOrgId(), vo.getCount());
        }
        return map;
    }

    @GetMapping("/getPageByDrools")
    PageInfo<DroolsVo> getPageByDrools(@RequestParam("field") String field,
                                       @RequestParam(value = "value", required = false) String value,
                                       @RequestParam("pageNo") Integer pageNo,
                                       @RequestParam("pageSize") Integer pageSize) {
        PageInfo<DroolsVo> pageInfo = new PageInfo<>();
        pageInfo.setPageNo(pageNo);
        pageInfo.setPageSize(pageSize);
        Page<DroolsVo> page = accountService.getPageByDrools(field, value, PageUtil.vice(pageInfo));
        return PageUtil.versa(converter.toPageVO(page, DroolsVo.class));
    }

    @PostMapping("/getPageByRuleItems")
    PageInfo<DroolsAccountVO> getPageByRuleItems(@RequestBody RuleVisibleRangeVo vo) {
        PageInfo<DroolsAccountVO> pageInfo = new PageInfo<>();
        pageInfo.setPageNo(vo.getPageNo());
        pageInfo.setPageSize(vo.getPageSize());
        Page page = accountService.getPageByRuleItems(vo, PageUtil.vice(pageInfo));
        return PageUtil.versa(page);
    }

    /**
     * 保存用户
     * @param account
     * @return
     */
    @PostMapping("/insert/third/account")
    public Account saveThirdAccount(@RequestBody com.yizhi.system.application.vo.domain.Account account){
        Account account1 = new Account();
        BeanUtils.copyProperties(account,account1);
        return accountService.saveThirdAccount(account1);
    }

    @Autowired
    private OrganizationController organizationController;

    /**
     * 查询管理员管辖区和组织下的
     * @return
     */
    @GetMapping("/org/ids")
    public Response<String> findAccountIds() {
        List<Long> accountIds = organizationController.getAccountIds();
        return Response.ok(accountIds);
    }

    /**
     * 查询用户
     * @param id
     * @return
     */
    @GetMapping("/byId")
    public Account getAccountById(@RequestParam(name = "id") Long id) {
        return accountService.selectById(id);
    }

    /**
     * 获取用户信息
     * @param authorizeUsersVO
     * @return
     */
    @PostMapping("/accountIds")
    public Map<Long,Account> getAccountList(@RequestBody AuthorizeUsersVO authorizeUsersVO) {
        Account account = new Account();
        account.setCompanyId(authorizeUsersVO.getRoleId());
        EntityWrapper<Account> condition = QueryUtil.condition(account);
        condition.in("id",authorizeUsersVO.getUserIds());
        List<Account> list = accountService.selectList(condition);

        Map<Long, Account> userInfoMap =
                list.stream()
                        .collect(Collectors.toMap(a -> a.getId(), a -> a));
        return userInfoMap;
    }

    @GetMapping("/findAccountByName")
    public Account findAccountByName(@RequestParam(name = "name") String name,@RequestParam(name = "companyId") Long companyId) {
		Account user = accountService.find(companyId, name);
		return user;
    }

    /**
     * 保存供应商的用户的信息
     * @param weChatUserVO
     * @return
     */
    @PostMapping(value = "/insert/provider/user")
    public Account saveProviderUser(@RequestBody WeChatUserVO weChatUserVO) {
        LOG.info("保存供应商微信的用户信息：{}", JSON.toJSON(weChatUserVO));

        Account account = userWechatService.saveProviderUser(weChatUserVO);

        return account;
    }

    /**
     * 社群人员同步
     * @param indexTime
     * @return
     */
    @PostMapping(value="/public/community/syncAccount")
    public List<Account> syncAccount(@RequestParam("indexTime") long indexTime){
        log.info("===================社群syncAccount同步==========================");
        EntityWrapper<Account> wrapper = new EntityWrapper<>();
        wrapper.gt("index_time", DateUtil.formatDateTime(new Date(indexTime)));
        List<Account> accounts = accountService.selectList(wrapper);
        if (CollectionUtils.isEmpty(accounts)){
            return null;
        }
        return  accounts;
    }

    @GetMapping("searchByConditionWithPriority")
    public ResponseEntity<Account> searchByConditionWithPriority(@RequestParam(name = "companyId") Long companyId,
                                                        @RequestParam(name = "account") String account,
                                                        @RequestParam(name = "mobile") String mobile,
                                                        @RequestParam(name = "email") String email) {
        Account accountEW = new Account();
        accountEW.setEnabled(true);
        accountEW.setDeleted(0);
        accountEW.setLocked(false);
        accountEW.setCompanyId(companyId);

        List<Account> resAccount = null;

        EntityWrapper<Account> ew = new EntityWrapper<>(accountEW);
        ew.eq("name", account);

        resAccount = accountService.selectList(ew);

        if (StringUtils.isNotEmpty(mobile) && CollectionUtils.isEmpty(resAccount)) {
            ew = new EntityWrapper<>(accountEW);
            ew.eq("mobile", mobile);
            resAccount = accountService.selectList(ew);
        }

        if (StringUtils.isNotEmpty(email) && CollectionUtils.isEmpty(resAccount)) {
            ew = new EntityWrapper<>(accountEW);
            ew.eq("email", email);
            resAccount = accountService.selectList(ew);
        }

        if (CollectionUtils.isEmpty(resAccount) || resAccount.size() > 1) {
            return ResponseEntity.ok(null);
        }

        return ResponseEntity.ok(resAccount.get(0));
    }

}
