package com.yizhi.system.application.service.impl;

import com.baomidou.mybatisplus.enums.SqlLike;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.system.application.constant.SecurityError;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.system.application.domain.*;
import com.yizhi.system.application.service.IAuthzAccountRoleService;
import com.yizhi.system.application.service.IAuthzRoleMenuService;
import com.yizhi.system.application.service.IAuthzRoleService;
import com.yizhi.system.application.service.IAuthzRoleUserGroupService;
import com.yizhi.system.application.mapper.*;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.system.application.vo.ManagerEditVo;
import com.yizhi.system.application.vo.ManagerVo;
import com.yizhi.system.application.vo.ReportAccountRespVO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

@Service
@Transactional(rollbackFor = Exception.class)
public class AuthzRoleServiceImpl extends ServiceImpl<AuthzRoleMapper,AuthzRole> implements IAuthzRoleService {

    private static final Logger logger = LoggerFactory.getLogger(AuthzRoleServiceImpl.class);

    @Autowired
    private IdGenerator idGenerator;
    @Autowired
    private AuthzRoleMapper roleMapper;
    @Autowired
    private AuthzAccountRoleMapper accountRoleMapper;
    @Autowired
    private IAuthzRoleMenuService roleModuleService;
    @Autowired
    private IAuthzRoleUserGroupService roleGroupService;
    @Autowired
    private IAuthzAccountRoleService accountRoleService;
    @Autowired
    AuthzRoleMenuMapper roleModuleMapper;
    @Autowired
    AuthzRoleUserGroupMapper roleGroupMapper;
    @Autowired
    AccountMapper accountMapper;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private AuthzRoleManagerMapper authzRoleManagerMapper;

    /**
     * 新增角色
     *
     * @param role
     */
    @Override
    public void add(AuthzRole role) {
        logger.info("新增角色{}", role);
        if (roleMapper.countExistCode(0L, role.getSiteId(), role.getCode()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "角色编码已存在");
        }
        if (roleMapper.countExistName(0L, role.getSiteId(), role.getName()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "角色名称已存在");
        }
        // 生产新的roleId
        Long roleId = idGenerator.generate();
        role.setId(roleId);
        // 添加role
        if (this.insert(role)) {
            if (CollectionUtils.isNotEmpty(role.getMenuIds())) {
                List<AuthzRoleMenu> modules = new ArrayList<>();
                for (Long menuId : role.getMenuIds()) {
                    modules.add(new AuthzRoleMenu(idGenerator.generate(), roleId, menuId));
                }
                if (!roleModuleService.insertBatch(modules)) {
                    throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
                }
            }
            if (CollectionUtils.isNotEmpty(role.getGroupIds())) {
                List<AuthzRoleUserGroup> groups = new ArrayList<>();
                for (Long userGroupId : role.getGroupIds()) {
                    groups.add(new AuthzRoleUserGroup(idGenerator.generate(), roleId, userGroupId));
                }
                if (!roleGroupService.insertBatch(groups)) {
                    throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
                }
            }
            List<Long> managerAccountIds = role.getManagerAccountIds();
            if (CollectionUtils.isNotEmpty(managerAccountIds)) {
                List<AuthzRoleManager> list = new ArrayList<>(managerAccountIds.size());
                managerAccountIds.forEach((item) -> list.add(new AuthzRoleManager(idGenerator.generate(), roleId, item)));
                if (CollectionUtils.isNotEmpty(list)) {
                    if (authzRoleManagerMapper.insertBatch(list) != managerAccountIds.size()) {
                        throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
                    }
                }
            }

            // 删除权限缓存
            deleteAccountAuthzInfo(ContextHolder.get().getCompanyCode(), ContextHolder.get().getSiteCode());
        } else {
            throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
        }
    }

    /**
     * 更新角色
     *
     * @param role
     */
    @Override
    public void update(AuthzRole role) {
        logger.info("更新角色:{}", role);
        //如果修改name，检查name是否重复
        if (StringUtils.isNotEmpty(role.getName())
                && roleMapper.countExistName(role.getId(), role.getSiteId(), role.getName()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "角色名称已存在");
        }
        //如果修改管辖区域，先把旧数据删除
        if (CollectionUtils.isNotEmpty(role.getGroupIds())) {
            //删除旧数据
            AuthzRoleUserGroup condition = new AuthzRoleUserGroup();
            condition.setRoleId(role.getId());
            roleGroupService.delete(QueryUtil.condition(condition));
            //插入新数据
            List<AuthzRoleUserGroup> groups = new ArrayList<>();
            for (Long userGroupId : role.getGroupIds()) {
                groups.add(new AuthzRoleUserGroup(idGenerator.generate(), role.getId(), userGroupId));
            }
            if (!roleGroupService.insertBatch(groups)) {
                throw new BizException(ReturnCode.UPDATE_FAIL.getCode(), ReturnCode.UPDATE_FAIL.getMsg());
            }
        }
        //如果修改管理模块，先把旧数据删除
        if (CollectionUtils.isNotEmpty(role.getMenuIds())) {
            //删除旧数据
            AuthzRoleMenu condition = new AuthzRoleMenu();
            condition.setRoleId(role.getId());
            roleModuleService.delete(QueryUtil.condition(condition));
            //插入新数据
            List<AuthzRoleMenu> modules = new ArrayList<>();
            for (Long menuId : role.getMenuIds()) {
                modules.add(new AuthzRoleMenu(idGenerator.generate(), role.getId(), menuId));
            }
            if (!roleModuleService.insertBatch(modules)) {
                throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
            }
        }
        //修改角色
        if (StringUtils.isNotEmpty(role.getName())) {
            role.setCode(null);
            if (!this.updateById(role)) {
                throw new BizException(ReturnCode.UPDATE_FAIL.getCode(), ReturnCode.UPDATE_FAIL.getMsg());
            }
        }
        // 清除缓存
        Site site = new Site();
        site.setId(role.getSiteId());
        site = site.selectById();
        deleteAccountAuthzInfo(ContextHolder.get().getCompanyCode(), site.getCode());
    }

    /**
     * 分页查询
     *
     * @param role
     * @param page
     */
    @Override
    public Page<AuthzRole> list(AuthzRole role, Page<AuthzRole> page) {
        logger.info("查询角色列表条件:{}", role.toString());
        role.setEnabled(1);
        EntityWrapper<AuthzRole> ew = null;
        if (StringUtils.isNotEmpty(role.getName()) && StringUtils.isNotEmpty(role.getCode()) && role.getName().equals(role.getCode())) {
            ew = QueryUtil.condition(new AuthzRole());
            ew.where("company_id = {0} and site_id = {1} and enabled = {2} ", role.getCompanyId(), role.getSiteId(), 1)
                    .andNew().like("name", role.getName(), SqlLike.DEFAULT).or().like("code", role.getCode(), SqlLike.DEFAULT);
            if (role.getCreateById()!=null) {
            	logger.info("分级管理员id："+role.getCreateById());
				ew.and("create_by_id = {0}", role.getCreateById());
			}
        } else {
            ew = QueryUtil.condition(role);
        }
        ew.orderBy("create_time", false);
        Page<AuthzRole> pageRole = selectPage(page, ew);
        List<AuthzRole> roles = pageRole.getRecords();
        if (CollectionUtils.isNotEmpty(roles)) {
            List<Long> roleIds = new ArrayList<>();
            Set<Long> accountIds = null;
            List<Account> accounts = null;
            for (AuthzRole role1 : roles) {
                // 设置被关联数
                AuthzAccountRole accountRole = new AuthzAccountRole();
                accountRole.setRoleId(role1.getId());
                Integer count = accountRoleMapper.selectCount(QueryUtil.condition(accountRole));
                role1.setRelevanceNum(count == null ? 0 : count);
                // 设置分级授权管理用户名
                roleIds.clear();
                roleIds.add(role1.getId());
                accountIds = authzRoleManagerMapper.getManagerAccountIdsByRoleIds(role.getSiteId(), roleIds);
                if (CollectionUtils.isNotEmpty(accountIds)) {
                    accounts = accountMapper.selectBatchIds(accountIds);
                    if (CollectionUtils.isNotEmpty(accounts)) {
                        accounts.forEach((account -> {
                            role1.setManagerAccountNames(role1.getManagerAccountNames().concat(",").concat(account.getName()));
                        }));
                        role1.setManagerAccountNames(role1.getManagerAccountNames().substring(1));
                    }
                }
            }
            pageRole.setRecords(roles);
        }
        return pageRole;
    }

    /**
     * 根据roleId查询关联的账户信息
     *
     * @param roleId
     * @return
     */
    @Override
    public List<Account> getAccountByRoleId(Long roleId) {
        //查询账号列表
        List<Long> accountIds = accountRoleMapper.getAccountIdsByRoleId(roleId);
        if (CollectionUtils.isNotEmpty(accountIds)) {
            List<Account> accounts = accountMapper.selectBatchIds(accountIds);
            return accounts;
        }
        return new ArrayList<>();
    }

    /**
     * 删除角色下关联的账户
     *
     * @param roleId
     * @param siteCode
     * @param accountId
     * @return
     */
    @Override
    public void delAccountByRoleId(Long roleId, String siteCode, Long accountId) {
        AuthzAccountRole accountRole = new AuthzAccountRole();
        accountRole.setRoleId(roleId);
        accountRole.setAccountId(accountId);
        if (accountRoleMapper.delete(QueryUtil.condition(accountRole)) < 1) {
            throw new BizException(SecurityError.ROLE_UNAUTH_ACCOUNT_ERROR, SecurityError.ROLE_UNAUTH_ACCOUNT_ERROR_MSG);
        }
        // 清除权限缓存
        deleteAccountAuthzInfo(ContextHolder.get().getCompanyCode(), siteCode);
    }

    @Override
    public void delete(AuthzRole role) {
        //检查角色是否是站点默认管理员 默认管理员不能删除
        role = selectById(role.getId());
        String code = role.getCode();
        if (code.contains(AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX)){
            throw new BizException(SecurityError.ROLE_IS_DEFAULT_ADMINISTRATOR,SecurityError.ROLE_IS_DEFAULT_ADMINISTRATOR_MSG);
        }
        //检查角色是否关联账号，关联的话不能删除
        AuthzAccountRole condition = new AuthzAccountRole();
        condition.setRoleId(role.getId());
        List<AuthzAccountRole> accountRoles = accountRoleMapper.selectList(QueryUtil.condition(condition));
        if (CollectionUtils.isNotEmpty(accountRoles)) {
            throw new BizException(SecurityError.ROLE_HAS_ACCOUNT, SecurityError.ROLE_HAS_ACCOUNT_MSG);
        }
        role.setEnabled(0);
        if (!this.updateById(role)) {
            throw new BizException(ReturnCode.DELETE_FAIL.getCode(), ReturnCode.DELETE_FAIL.getMsg());
        }
        // 删除关联关系
        AuthzRoleUserGroup authzRoleUserGroup = new AuthzRoleUserGroup();
        authzRoleUserGroup.setRoleId(role.getId());
        authzRoleUserGroup.delete(new EntityWrapper(authzRoleUserGroup));

        AuthzRoleMenu authzRoleMenu = new AuthzRoleMenu();
        authzRoleMenu.setRoleId(role.getId());
        authzRoleMenu.delete(new EntityWrapper(authzRoleMenu));

        AuthzRoleManager authzRoleManager = new AuthzRoleManager();
        authzRoleManager.setRoleId(role.getId());
        authzRoleManager.delete(new EntityWrapper(authzRoleManager));
    }

    @Override
    public void checkExistCode(String code, Long siteId) {
        if (roleMapper.countExistCode(0L, siteId, code) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "角色编码已存在");
        }
    }

    @Override
    public void checkExistName(String name, Long siteId) {
        if (roleMapper.countExistName(0L, siteId, name) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "角色名称已存在");
        }

    }

    @Override
    public void authAccount(Long roleId, String siteCode, List<Long> accountIds) {
        RequestContext context = ContextHolder.get();
        Set<Long> beforeAccountIds = new HashSet<>();
        // 先查出以前的
        AuthzAccountRole accountRole = new AuthzAccountRole();
        accountRole.setRoleId(roleId);
        accountRole.setSiteId(context.getSiteId());
        List<AuthzAccountRole> oldAccountRoles = accountRole.selectList(new EntityWrapper(accountRole));
        // 先删除以前的，并记录影响的用户id
        if (CollectionUtils.isNotEmpty(oldAccountRoles)) {
            for (AuthzAccountRole ar : oldAccountRoles) {
                beforeAccountIds.add(ar.getId());
            }
            accountRoleMapper.deleteBatchIds(beforeAccountIds);
        }
        // 新增现在的
        if (CollectionUtils.isNotEmpty(accountIds)) {
            List<AuthzAccountRole> accountRoles = new ArrayList<>();
            for (Long accountId : accountIds) {
                accountRoles.add(new AuthzAccountRole(idGenerator.generate(), accountId, roleId, context.getSiteId()));
            }
            if (!accountRoleService.insertBatch(accountRoles)) {
                throw new BizException(SecurityError.ROLE_AUTH_ACCOUNT_ERROR, SecurityError.ROLE_AUTH_ACCOUNT_ERROR_MSG);
            }
        }
        // 清除权限缓存
        deleteAccountAuthzInfo(context.getCompanyCode(), siteCode);
    }

    @Override
    public AuthzRole get(Long id) {
        AuthzRole role = this.selectById(id);
        if (null == role) {
            throw new BizException(ReturnCode.RESOURCE_NOT_FOUND.getCode(), ReturnCode.RESOURCE_NOT_FOUND.getMsg());
        }
        role.setMenuIds(roleModuleMapper.getMenuIdsByRoleId(id));
        role.setGroupIds(roleGroupMapper.getUserGroupIdsByRoleId(id));
        return role;
    }

    @Cacheable(cacheNames = "auth:manage", key = "'company:'+ #companyId " +
            "+':site:' + #siteId + ':account:' + #accountId + ':moduleCode:' + #moduleCode")
    @Override
    public List<Long> getIdByModuleCodeInSite(Long companyId, Long siteId, Long accountId, String moduleCode) {
        return roleMapper.getIdByModuleCodeInSite(moduleCode, accountId, siteId);
    }

    /**
     * 删除权限缓存
     *
     * @param companyCode
     * @param siteCode
     */
    @Override
    public void deleteAccountAuthzInfo(String companyCode, String siteCode) {
        try {
            // 删除权限cache
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ORGIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ACCOUNT_ACCESS_SITE + companyCode);
        } catch (Exception e) {
            logger.error("删除权限缓存redis错误：try-catch处理，不影响返回", e);
        }
    }

    @Override
    public Boolean managerEdit(ManagerEditVo editVo) {
        long roleId = editVo.getRoleId();
        // 先删除以前的
        AuthzRoleManager roleManager = new AuthzRoleManager();
        roleManager.setRoleId(roleId);
        boolean result = roleManager.delete(new EntityWrapper(roleManager));
        Set<Long> accountIds = editVo.getAccountIds();
        if (result && !CollectionUtils.isEmpty(accountIds)) {
            List<AuthzRoleManager> list = new ArrayList<>(accountIds.size());
            accountIds.forEach((accountId) -> {
                AuthzRoleManager authzRoleManager = new AuthzRoleManager();
                authzRoleManager.setRoleId(roleId);
                authzRoleManager.setId(idGenerator.generate());
                authzRoleManager.setManagerAccountId(accountId);
                list.add(authzRoleManager);
            });
            authzRoleManagerMapper.insertBatch(list);
        }
        redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS.concat(ContextHolder.get().getCompanyCode()));
        return Boolean.TRUE;
    }

    @Override
    public List<ManagerVo> listManager(Long roleId) {
        Long siteId = ContextHolder.get().getSiteId();
        Set<Long> roleIdsInSite = roleMapper.getIdsBySiteId(siteId, ContextHolder.get().getAccountId());
        if (CollectionUtils.isNotEmpty(roleIdsInSite)) {
            Set<Long> accountIds = accountRoleMapper.getAccountIdsByRoleIds(roleIdsInSite);
            Set<Long> siteAdminAccountIds = accountRoleMapper.getSiteAdminAccountId(siteId);
            if (CollectionUtils.isNotEmpty(accountIds)) {
                siteAdminAccountIds.forEach(item -> accountIds.remove(item));
                if (CollectionUtils.isNotEmpty(accountIds)) {
                    List<Account> accounts = accountMapper.selectBatchIds(accountIds);
                    if (CollectionUtils.isNotEmpty(accounts)) {
                        List<ManagerVo> managerVos = new ArrayList<>(accounts.size());
                        // 查询该角色已经关联管理的管理员
                        List<Long> roleIds = new ArrayList<>();
                        roleIds.add(roleId);
                        Set<Long> managerIds = authzRoleManagerMapper.getManagerAccountIdsByRoleIds(siteId, roleIds);
                        accounts.forEach((account) -> {
                            ManagerVo vo = new ManagerVo();
                            vo.setAccountId(account.getId());
                            vo.setIsChecked(managerIds.contains(account.getId()));
                            vo.setName(account.getName());
                            managerVos.add(vo);
                        });
                        return managerVos;
                    }
                }
                return new ArrayList<>();
            }
        }
        return new ArrayList<>();
    }

    @Override
    public Page<ManagerVo> pageManager(Long roleId, String keyWord, String position, Long orgId, String orgName, Integer pageNo, Integer pageSize) {

        Page  page = new Page<>(pageNo, pageSize);
        Long siteId = ContextHolder.get().getSiteId();
        //获取该站点下所有的角色id
        Set<Long> roleIdsInSite = roleMapper.getIdsBySiteId(siteId, ContextHolder.get().getAccountId());
        if (CollectionUtils.isNotEmpty(roleIdsInSite)) {
            //获取所有角色关联的用户
            Set<Long> accountIds = accountRoleMapper.getAccountIdsByRoleIds(roleIdsInSite);
            //获取当前站点下所有站点管理员的账号
            Set<Long> siteAdminAccountIds = accountRoleMapper.getSiteAdminAccountId(siteId);
            if (CollectionUtils.isNotEmpty(accountIds)) {
                siteAdminAccountIds.forEach(item -> accountIds.remove(item));
                //剔除站点管理员的id
                if (CollectionUtils.isNotEmpty(accountIds)) {
                    List<Long> accountLists = new ArrayList<>(accountIds);
                    List<ReportAccountRespVO> accounts = accountMapper.getVoByIds(accountLists, keyWord, position, orgId, orgName, page);
                    if (CollectionUtils.isNotEmpty(accounts)) {
                        List<ManagerVo> managerVos = new ArrayList<>(accounts.size());
                        // 查询该角色已经关联管理的管理员
                        List<Long> roleIds = new ArrayList<>();
                        roleIds.add(roleId);
                        Set<Long> managerIds = authzRoleManagerMapper.getManagerAccountIdsByRoleIds(siteId, roleIds);
                        accounts.forEach((account) -> {
                            ManagerVo vo = new ManagerVo();
                            vo.setAccountId(account.getUserId());
                            vo.setIsChecked(managerIds.contains(account.getUserId()));
                            vo.setName(account.getUserName());
                            vo.setFullName(account.getUserFullName());
                            vo.setOrgName(account.getOrgName());
                            vo.setOrgId(account.getOrgId());
                            vo.setWorkNum(account.getWorkNum());
                            vo.setPosition(account.getPosition());
                            managerVos.add(vo);
                        });
                        page.setRecords(managerVos);
                    }
                }
            }
        }
        return page;
    }
}
