package com.yizhi.system.application.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.mapper.Wrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.enums.MemberType;
import com.yizhi.system.application.enums.OrderField;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.system.application.mapper.AccountMapper;
import com.yizhi.system.application.mapper.AuthzGroupMemberMapper;
import com.yizhi.system.application.mapper.AuthzUserGroupMapper;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.system.application.service.AuthorityService;
import com.yizhi.system.application.service.IAccountService;
import com.yizhi.system.application.service.IAuthzGroupMemberService;
import com.yizhi.system.application.service.IAuthzUserGroupService;
import com.yizhi.system.application.service.using.OrganizationIndexService;
import com.yizhi.system.application.domain.Account;
import com.yizhi.system.application.domain.AuthzGroupMember;
import com.yizhi.system.application.domain.AuthzUserGroup;
import com.yizhi.system.application.domain.Site;
import com.yizhi.system.application.util.RightLeftIndex;
import com.yizhi.util.application.constant.ReturnCode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * <p>
 * 学习组 服务实现类
 * </p>
 *
 * @author yinyuyan123
 * @since 2018-04-16
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class AuthzUserGroupServiceImpl extends ServiceImpl<AuthzUserGroupMapper, AuthzUserGroup> implements IAuthzUserGroupService {
    private static final Logger logger = LoggerFactory.getLogger(AuthzUserGroupServiceImpl.class);

    @Autowired
    IdGenerator idGenerator;
    @Autowired
    AuthzUserGroupMapper groupMapper;
    @Autowired
    AuthzGroupMemberMapper groupMemberMapper;
    @Autowired
    AccountMapper accountMapper;
    @Autowired
    AuthorityService authorityService;
    @Autowired
    IAuthzGroupMemberService groupMemberService;
    @Autowired
    private AuthzUserGroupMapper userGroupMapper;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private IAccountService accountService;
    @Autowired
    private OrganizationIndexService organizationIndexService;

    @Override
    public Boolean save(AuthzUserGroup group) {
        if (groupMapper.countExistName(0L, group.getName(), group.getCompanyId(), group.getSiteId()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "管辖区名称已存在");
        }
        Long groupId = idGenerator.generate();
        group.setId(groupId);
        List<AuthzGroupMember> members = new ArrayList<>();
        //关联员工
        if (CollectionUtils.isNotEmpty(group.getAccountIds())) {
            for (Long accountId : group.getAccountIds()) {
                members.add(new AuthzGroupMember(idGenerator.generate(), groupId, accountId, MemberType.USER.getCode()));
            }
        }
        //关联部门
        if (CollectionUtils.isNotEmpty(group.getOrgIds())) {
            for (Long orgId : group.getOrgIds()) {
                members.add(new AuthzGroupMember(idGenerator.generate(), groupId, orgId, MemberType.ORG.getCode()));
            }
        }
        //批量插入
        if (CollectionUtils.isNotEmpty(members)) {
            if (!groupMemberService.insertBatch(members)) {
                throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
            }
        }
        if (this.insert(group)) {
            return true;
        } else {
            throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
        }
    }

    @Override
    public Boolean update(AuthzUserGroup group) {
        if (StringUtils.isNotEmpty(group.getName())) {
            if (groupMapper.countExistName(group.getId(), group.getName(), group.getCompanyId(), group.getSiteId()) > 0) {
                throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "管辖区名称已存在");
            }
        }
        if (this.updateById(group)) {
            //删除旧关联数据
            AuthzGroupMember condition = new AuthzGroupMember();
            condition.setGroupId(group.getId());
            condition.setDeleted(0);
            groupMemberService.delete(QueryUtil.condition(condition));

            if (CollectionUtils.isNotEmpty(group.getAccountIds()) || CollectionUtils.isNotEmpty(group.getOrgIds())) {
                List<AuthzGroupMember> members = new ArrayList<>();
                //关联员工
                if (CollectionUtils.isNotEmpty(group.getAccountIds())) {
                    for (Long accountId : group.getAccountIds()) {
                        members.add(new AuthzGroupMember(idGenerator.generate(), group.getId(), accountId, MemberType.USER.getCode()));
                    }
                }
                //关联部门
                if (CollectionUtils.isNotEmpty(group.getOrgIds())) {
                    for (Long orgId : group.getOrgIds()) {
                        members.add(new AuthzGroupMember(idGenerator.generate(), group.getId(), orgId, MemberType.ORG.getCode()));
                    }
                }
                //批量插入
                if (!groupMemberService.insertBatch(members)) {
                    throw new BizException(ReturnCode.UPDATE_FAIL.getCode(), ReturnCode.UPDATE_FAIL.getMsg());
                }
            }
        } else {
            throw new BizException(ReturnCode.UPDATE_FAIL.getCode(), ReturnCode.UPDATE_FAIL.getMsg());
        }
        return true;
    }

    @Override
    public Boolean delete(Long id) {
        AuthzUserGroup group = new AuthzUserGroup();
        group.setId(id);
        group.setDeleted(true);
        if (this.updateById(group)) {
            AuthzGroupMember condition = new AuthzGroupMember();
            condition.setGroupId(group.getId());
            condition.setDeleted(0);
            groupMemberMapper.delete(QueryUtil.condition(condition));
        } else {
            throw new BizException(ReturnCode.DELETE_FAIL.getCode(), ReturnCode.DELETE_FAIL.getMsg());
        }
        return true;
    }

    @Override
    public AuthzUserGroup findById(Long id) {
        AuthzUserGroup group = this.selectById(id);
        if (null == group) {
            throw new BizException(ReturnCode.RESOURCE_NOT_FOUND.getCode(), ReturnCode.RESOURCE_NOT_FOUND.getMsg());
        }
        List<AuthzGroupMember> groupMembers = groupMemberMapper.listByGroupId(id);
        if (CollectionUtils.isNotEmpty(groupMembers)) {
            List<Long> orgIds = new ArrayList<>();
            List<Long> accountIds = new ArrayList<>();
            for (AuthzGroupMember gm : groupMembers) {
                if (gm.getType().equals(MemberType.ORG.getCode())) {
                    orgIds.add(gm.getMemberId());
                } else {
                    accountIds.add(gm.getMemberId());
                }
            }
            group.setOrgIds(groupMemberMapper.getMemberIdsByGroupId(1, id));
            group.setAccountIds(groupMemberMapper.getMemberIdsByGroupId(2, id));
        }
        return group;
    }

    @Override
    public List<AuthzUserGroup> list(AuthzUserGroup group, Long manageAccountId) {
        group.setDeleted(Boolean.valueOf(false));
        group.setCreateById(manageAccountId);
        group.setName(null);
        Wrapper<AuthzUserGroup> wrapper = QueryUtil.condition(group);
        if (StringUtils.isNotEmpty(group.getName())) {
            wrapper.like("name", group.getName());
        }
        return this.selectList(wrapper);
    }

    @Override
    public Boolean checkName(Long id, String name, Long companyId, Long siteId) {
        if (groupMapper.countExistName(id, name, companyId, siteId) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "管辖区名称已存在");
        } else {
            return true;
        }
    }

    @Override
    public Page<AuthzUserGroup> listByPage(AuthzUserGroup group, Page<AuthzUserGroup> page) {
        logger.info("查询用户组列表条件：{}", group);
        page.setOrderByField(OrderField.CREATETIME.getCode());
        page.setAsc(false);
        group.setDeleted(false);
        EntityWrapper<AuthzUserGroup> wrapper = new EntityWrapper<>(group);
        if (!StringUtils.isEmpty(group.getName())) {
            wrapper.like("name", group.getName().trim());
            group.setName(null);
        }
        Page<AuthzUserGroup> pageRtn = this.selectPage(page, wrapper);
        return pageRtn;
    }

    @Override
    public Map<String, List<Long>> getOrgIdAndAccountIdByGroupId(Long groupId) {
        Map<String, List<Long>> retMap = new HashMap<>();
        AuthzGroupMember condition = new AuthzGroupMember();
        condition.setDeleted(0);
        condition.setGroupId(groupId);
        List<AuthzGroupMember> members = groupMemberMapper.selectList(QueryUtil.condition(condition));
        if (CollectionUtils.isNotEmpty(members)) {
            List<Long> accountIds = new ArrayList<>();
            List<Long> orgIds = new ArrayList<>();
            for (AuthzGroupMember member : members) {
                if (MemberType.ORG.getCode().intValue() == member.getType().intValue()) {
                    orgIds.add(member.getMemberId());
                } else {
                    accountIds.add(member.getMemberId());
                }
            }
            retMap.put("orgIds", orgIds);
            retMap.put("accountIds", accountIds);
        }
        return retMap;
    }

    @Cacheable(cacheNames = "auth:manage", key = "'role:' + #roleId + ':userGroup'")
    @Override
    public Set<Long> getByRoleId(List<Long> roleIds) {
        return userGroupMapper.getByRoleId(roleIds);
    }

    private Integer selectCountByGroupId(Long groupId) {
        Map<String, List<Long>> countMap = this.getOrgIdAndAccountIdByGroupId(groupId);
        Integer count = 0;
        if (null != countMap) {
            if (CollectionUtils.isNotEmpty(countMap.get("orgIds"))) {
                Wrapper<Account> wrapper = QueryUtil.condition(new Account());
                wrapper.in("org_id", countMap.get("orgIds"));
                count += accountMapper.selectCount(wrapper);
            }
            if (CollectionUtils.isNotEmpty(countMap.get("accountIds"))) {
                count += countMap.get("accountIds").size();
            }
        }
        return count;
    }

    /**
     * 删除权限缓存
     *
     * @param companyCode
     * @param groupId
     */
    @Override
    public void deleteAccountAuthzInfo(String companyCode, String siteCode, Long groupId) {
        try {
//            // 根据管辖区查询角色id
//            Set<Long> roleIds = roleUserGroupMapper.getUserIdByGroupId(groupId);
//            if (CollectionUtils.isNotEmpty(roleIds)) {
//                // 查询角色关联的管理员id
//                Set<Long> accountIds = accountRoleMapper.getAccountIdsByRoleIdsWithoutSiteAdmin(roleIds);
//                String[] keyItems = new String[accountIds.size()];
//                int index = 0;
//                for (Long accountId : accountIds) {
//                    keyItems[index] = siteCode + CacheNamespace.REDIS_HASH_SEPARATOR + accountId;
//                }
//                if (CollectionUtils.isNotEmpty(accountIds)) {
//                    // 删除权限cache
//                    redisCache.hdel(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + companyCode, keyItems);
//                }
//            }
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ORGIDS + companyCode);
            // 删除站点下部门信息
            Site site = new Site();
            site.setUserGroupId(groupId);
            site.setEnabled(true);
            site = site.selectOne(new EntityWrapper(site));
            if (null != site) {
                redisCache.hdel(CacheNamespace.REIDS_COMPANY_SITE_ORGS + companyCode, new String[]{site.getCode()});
                redisCache.hdel(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + companyCode, new String[]{site.getCode()});
            }
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_FRONT_COMPANY_ACCOUNT_ACCESS_SITE + companyCode);
        } catch (Exception e) {
            logger.error("删除权限缓存redis错误：try-catch处理，不影响返回", e);
        }
    }

    @Override
    public Integer getGroupAccountNum(Long companyId, Long groupId) {
        Set<Long> orgIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId, MemberType.ORG.getCode());
        Set<Long> accountIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId, MemberType.USER.getCode());
        if (CollectionUtils.isEmpty(orgIds) && CollectionUtils.isEmpty(accountIds)) {
            return 0;
        }
        if (CollectionUtils.isEmpty(orgIds) && CollectionUtils.isNotEmpty(accountIds)) {
            return accountIds.size();
        }
        List<RightLeftIndex> indices = organizationIndexService.getIndices(orgIds);
        Long result = accountService.getReportAccountRespVOCount(companyId, null, indices, new ArrayList<>(accountIds), false);
        return Integer.valueOf(String.valueOf(result));
    }

    private int getOrgAndChildAccountNums(Long companyId, Set<Long> orgIds) {
        Account account = new Account();
        account.setCompanyId(companyId);
        account.setEnabled(true);
        EntityWrapper<Account> ew = new EntityWrapper<>(account);
        ew.in("org_id", orgIds);
        return accountMapper.selectCount(ew);
    }
}
