package com.yizhi.system.application.service.impl;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.system.application.constant.SiteProtectConstants;
import com.yizhi.system.application.constant.SiteProtectExceptionEnum;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.system.application.domain.SiteProtect;
import com.yizhi.system.application.domain.SiteProtectNotice;
import com.yizhi.system.application.domain.SiteProtectRel;
import com.yizhi.system.application.domain.dto.SiteProtectDTO;
import com.yizhi.system.application.domain.dto.SiteProtectExportDTO;
import com.yizhi.system.application.enums.SiteNoticeEnum;
import com.yizhi.system.application.enums.SiteProtectEnum;
import com.yizhi.system.application.exception.SiteProtectBusinessException;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.system.application.mapper.*;
import com.yizhi.system.application.service.SiteProtectNoticeService;
import com.yizhi.system.application.service.SiteProtectRelService;
import com.yizhi.system.application.service.SiteProtectService;
import com.yizhi.system.application.service.condition.SiteProjectCondition;
import com.yizhi.system.application.request.*;
import com.yizhi.system.application.vo.*;
import com.yizhi.system.application.domain.Company;
import com.yizhi.system.application.domain.Site;
import com.yizhi.util.application.page.PageInfo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class SiteProtectServiceImpl extends ServiceImpl<SiteProtectMapper, SiteProtect>implements SiteProtectService {

    @Autowired
    IdGenerator idGenerator;
    @Autowired
    RedisCache redisCache;
    @Autowired
    SiteProtectMapper siteProtectMapper;
    @Autowired
    SiteProtectRelService siteProtectRelService;
    @Autowired
    SiteProtectNoticeMapper protectNoticeMapper;
    @Autowired
    SiteProtectRelMapper siteProtectRelMapper;
    @Autowired
    DomainConverter converter;
    @Autowired
    SiteProtectNoticeService protectNoticeService;

    @Autowired
    SiteProtectNoticeMapper siteProtectNoticeMapper;
    @Autowired
    SiteProtectRelMapper protectRelMapper;

    @Autowired
    SiteMapper siteMapper;
    @Autowired
    CompanyMapper companyMapper;

    @Override
    @Transactional
    public PageInfo<SiteProtectVO> queryList(QuerySiteProtectResq querySiteProtectResq) {
        //构建查询条件
        SiteProjectCondition siteProjectCondition = new SiteProjectCondition();

        //设置其余查询条件
        setCondition(siteProjectCondition,querySiteProtectResq);

        List<SiteProtectDTO> siteProtectDTOs = baseMapper.querySiteProtectList(siteProjectCondition);
        LocalDateTime nowTime = siteProjectCondition.getNowTime();
        final List<SiteProtectVO> siteProtectVOS = Lists.newArrayList();
        // 封装vo设置状态
        siteProtectDTOs.forEach(a ->{
            SiteProtectVO siteProtectVO = new SiteProtectVO();
            siteProtectVO.setId(a.getId().toString());
            siteProtectVO.setName(a.getName());
            siteProtectVO.setSiteNoticeId(a.getSiteNoticeId().toString());
            siteProtectVO.setSiteNoticeEnum(a.getSiteNoticeEnum());
            siteProtectVO.setSiteProtectStatus(a.getSiteProtectStatus());
            if(nowTime.isBefore(a.getStartTime())){
                siteProtectVO.setSiteProtectStatus(SiteProtectConstants.NOT_START);
            }else if(nowTime.isAfter(a.getStartTime())&&nowTime.isBefore(a.getEndTime())){
                siteProtectVO.setSiteProtectStatus(SiteProtectConstants.RUNNING);
            }else{
                siteProtectVO.setSiteProtectStatus(SiteProtectConstants.FINISH);
            }
            siteProtectVO.setStartTime(a.getStartTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm")));
            siteProtectVO.setEndTime(a.getEndTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm")));
            siteProtectVOS.add(siteProtectVO);
        });
        int count = baseMapper.countQuerySiteProtectList(siteProjectCondition);
        PageInfo<SiteProtectVO> pageInfo = new PageInfo<>();

        pageInfo.setRecords(siteProtectVOS);
        pageInfo.setPageNo(querySiteProtectResq.getPageNo());
        pageInfo.setPageSize(querySiteProtectResq.getPageSize());
        // 计算总页数
        int mod = count%querySiteProtectResq.getPageSize();
        if(mod!=0){
            pageInfo.setPageTotal(count/querySiteProtectResq.getPageSize() + 1);
        }else {
            pageInfo.setPageTotal(count/querySiteProtectResq.getPageSize());
        }
        pageInfo.setPageRecords(count);
        return pageInfo;
    }

    /**
     * 设置基础条件
     * @param siteProjectCondition
     */
    private void setCondition(SiteProjectCondition siteProjectCondition,QuerySiteProtectResq querySiteProtectResq) {
        if(org.apache.commons.lang3.StringUtils.isNotBlank(querySiteProtectResq.getName())) {
            siteProjectCondition.setName(querySiteProtectResq.getName());
        }
        if(querySiteProtectResq.getSiteNoticeEnum()!=null) {
            siteProjectCondition.setSiteNoticeEnum(querySiteProtectResq.getSiteNoticeEnum());
        }

        Date startTime = querySiteProtectResq.getStartTime();
        if(startTime!=null){
            siteProjectCondition.setStartTime(startTime);
        }

        Date endTime = querySiteProtectResq.getEndTime();
        if(endTime!=null){
            siteProjectCondition.setEndTime(endTime);
        }
        siteProjectCondition.setStart((querySiteProtectResq.getPageNo()-1)*querySiteProtectResq.getPageSize());
        siteProjectCondition.setPageSize(querySiteProtectResq.getPageSize());
        // 根据状态设置条件,状态是根据当前时间和通知时间动态判断的,数据本身无状态
        siteProjectCondition.setNowTime(LocalDateTime.now());
        siteProjectCondition.setSiteProtectStatus(querySiteProtectResq.getSiteProtectStatus());
    }


    /**
     * 新增站点维护
     *
     * @param siteProtectReq 站点维护实体
     * @return 新增成功返回站点唯一Id，否则返回null
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Long addSiteProtect(SiteProtectReq siteProtectReq) {

        RequestContext context = ContextHolder.get();
        String siteProtectName = siteProtectReq.getName();
        String addProtectRedisKey = SiteProtectConstants.SITE_PROTECT_ADD_REDIS_PREF+siteProtectName;
        boolean lock = redisCache.setIfAbsent(addProtectRedisKey, JSONObject.toJSONString(siteProtectReq));
        if (!lock){
            log.info("[新增站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 重复提交! 新增实体 siteProtectReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getErrCode(),SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getMsg());
        }
        redisCache.expire(addProtectRedisKey,300);

        checkSiteProtectIsRepeat(siteProtectReq,context);

        Long siteProtectId = setAndInsertSiteProtect(siteProtectReq);

        //维护通知类型 站内通知、关站通知
        setAndInsertSiteProtectNotice(siteProtectReq, addProtectRedisKey, siteProtectId);

        String siteType = siteProtectReq.getSiteType().toString();
        if (SiteProtectEnum.PART.toString().equals(siteType)){
            //自定义选择站点
            setAndInsertProtectRelCompany(siteProtectReq, context, siteProtectId);
        }
        redisCache.delete(addProtectRedisKey);
        return siteProtectId;
    }

    //自定义选择站点维护通知
    private void setAndInsertProtectRelCompany(SiteProtectReq siteProtectReq, RequestContext context, Long siteProtectId) {
        // part sites protect notice
        List<SiteResq> siteList = siteProtectReq.getSiteList();
        if (CollectionUtils.isEmpty(siteList)){
            log.warn("[新增站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 自定义选择站点为空!新增实体 siteProtectReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_PART_SITE_IS_NULL.getErrCode(),SiteProtectExceptionEnum.PROTECT_PART_SITE_IS_NULL.getMsg());
        }
        List<SiteProtectRel> protectRelList = new ArrayList<>();
        siteList.forEach(siteResq -> {
            SiteProtectRel siteProtectRel = new SiteProtectRel();
            BeanUtils.copyProperties(siteProtectReq,siteProtectRel);
            siteProtectRel.setId(idGenerator.generate());
            siteProtectRel.setSiteProtectId(siteProtectId);
            //自定选择的站点id
            siteProtectRel.setSiteId(siteResq.getSiteId());
            protectRelList.add(siteProtectRel);
        });
        siteProtectRelService.insertBatch(protectRelList);
    }

    //设置并插入维护通知方式
    private void setAndInsertSiteProtectNotice(SiteProtectReq siteProtectReq, String addProtectRedisKey, Long siteProtectId) {
        List<SiteNoticeReq> siteNoticeReqList = siteProtectReq.getSiteNoticeReqList();
        List<SiteProtectNotice> noticeList = new ArrayList<>();
        for (SiteNoticeReq SiteNoticeReq : siteNoticeReqList){
            SiteProtectNotice siteProtectNotice = new SiteProtectNotice();
            Long siteProtectNoticeId = idGenerator.generate();
            BeanUtils.copyProperties(siteProtectReq,siteProtectNotice);
            BeanUtils.copyProperties(SiteNoticeReq,siteProtectNotice);

            Date startTime = SiteNoticeReq.getStartTime();
            Date endTime = SiteNoticeReq.getEndTime();
            if (null == startTime || null == endTime){
                redisCache.delete(addProtectRedisKey);
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_TIME_IS_NULL.getErrCode(),SiteProtectExceptionEnum.PROTECT_TIME_IS_NULL.getMsg());
            }
            siteProtectNotice.setSiteProtectId(siteProtectId);
            siteProtectNotice.setId(siteProtectNoticeId);
            noticeList.add(siteProtectNotice);
        }
        protectNoticeService.insertBatch(noticeList);
    }


    /**
     * check site protect time is valid
     */
    private void checkProtectTimeIsValid(SiteNoticeReq newProtectNoticeReq) {

        Long noticeId = newProtectNoticeReq.getId();
        if (null == noticeId || noticeId < 1){
            //throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_DATA_INVALID.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_DATA_INVALID.getMsg());
            return;
        }
        Date newStartTime = newProtectNoticeReq.getStartTime();
        Date newEndTime = newProtectNoticeReq.getEndTime();
        String newMessage = newProtectNoticeReq.getMessage();

        SiteProtectNotice oldNoticeEntity = getSiteProtectNoticeById(noticeId);
        Date oldStartTime = oldNoticeEntity.getStartTime();
        Date oldEndTime = oldNoticeEntity.getEndTime();
        String oldMessage = oldNoticeEntity.getMessage();
        Date currentTime = new Date();
        if (currentTime.before(oldStartTime)){
            //维护通知未开始
            if (!(newStartTime.after(currentTime) && newEndTime.after(currentTime))){
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_TIME_INVALID.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_TIME_INVALID.getMsg());
            }
        }else if (oldStartTime.before(currentTime) && oldEndTime.after(currentTime)){
            //进行中
            if (!newStartTime.equals(oldStartTime)){
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_START_TIME_INVALID.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_START_TIME_INVALID.getMsg());
            }
            if (newEndTime.before(currentTime)){
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_TIME_INVALID.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_TIME_INVALID.getMsg());
            }
        }else {
            if (!newStartTime.equals(oldStartTime) || !newEndTime.equals(oldEndTime) || !oldMessage.equals(newMessage)){
                //已结束;判断是否修改；修改则返回错误；否则通过
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_END.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_END.getMsg());
            }
        }
    }

    private SiteProtectNotice getSiteProtectNoticeById(Long noticeId) {
        SiteProtectNotice selectNotice = new SiteProtectNotice();
        selectNotice.setId(noticeId);
        EntityWrapper<SiteProtectNotice> noticeWrapper = new EntityWrapper<>();
        noticeWrapper.setEntity(selectNotice);
        return protectNoticeService.selectOne(noticeWrapper);
    }

    /**
     * 修改站点维护
     *
     * @param siteProtectReq 需要修改的站点维护数据
     * @return 修改成功返回该维护唯一id；否则返回null
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public Long updateSiteProtect(SiteProtectReq siteProtectReq) {
        Long siteProtectId = siteProtectReq.getId();
        RequestContext context = ContextHolder.get();
        String upProtectRedisKey = SiteProtectConstants.SITE_PROTECT_UP_REDIS_PREF+siteProtectId;
        boolean lock = redisCache.setIfAbsent(upProtectRedisKey, JSONObject.toJSONString(siteProtectReq));
        if (!lock){
            log.info("[修改站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 重复提交! 修改实体 siteProtectReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getErrCode(),SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getMsg());
        }
        //check site protect repeat
        checkSiteProtectIsRepeat(siteProtectReq,context);
        //获取修改的维护通知
        List<SiteNoticeReq> newSiteNoticeReqList = siteProtectReq.getSiteNoticeReqList();
        if (null == newSiteNoticeReqList || newSiteNoticeReqList.isEmpty()){
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_IS_NULL.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_IS_NULL.getMsg());
        }
        newSiteNoticeReqList.forEach(noticeReq -> {
            checkProtectTimeIsValid(noticeReq);
        });
        List<SiteProtectNotice> oldNoticeList = getProtectNoticeByProtectId(siteProtectId);
        if (null == newSiteNoticeReqList || newSiteNoticeReqList.size() == 0){
            //删除整个维护操作
            SiteProtectDelReq delReq = new SiteProtectDelReq();
            delReq.setSiteProtectId(siteProtectId.toString());
            for (SiteProtectNotice notice : oldNoticeList){

                delReq.setNoticeId(notice.getId().toString());
                deleteSiteProtect(delReq);
            }
            return siteProtectId;
        }
        if (null != oldNoticeList && oldNoticeList.size() > 0 && newSiteNoticeReqList.size() > 0){
            if (oldNoticeList.size() == newSiteNoticeReqList.size()){
                //全部修改，判断条件
                for (SiteNoticeReq newNotice : newSiteNoticeReqList){
                    Long noticeId = newNotice.getId();
                    SiteProtectNotice oldNotice = protectNoticeService.selectById(noticeId);
                    checkProtectTimeIsValid(newNotice);
                    SiteProtectNotice upNotice = compactNotice(newNotice,oldNotice);
                    protectNoticeService.updateById(upNotice);
                }
            }else {
                //有删除，有修改

                List<Long> newNoticeIds = getNoticeIds(newSiteNoticeReqList);
                List<Long> oldNoticeIds =  getOldNoticeIds(oldNoticeList);


                for (SiteNoticeReq newNotice : newSiteNoticeReqList){
                    Long noticeId = newNotice.getId();
                    if (oldNoticeIds.contains(noticeId)){
                        //修改
                        SiteProtectNotice oldNotice = protectNoticeService.selectById(noticeId);
                        checkProtectTimeIsValid(newNotice);
                        SiteProtectNotice upNotice = compactNotice(newNotice,oldNotice);
                        protectNoticeService.updateById(upNotice);
                    }else {
                        //有新增
                        SiteProtectNotice notice = getNewAddNotice(newNotice,siteProtectId);
                        notice.setSiteType(siteProtectReq.getSiteType());
                        protectNoticeService.insert(notice);
                    }
                }
                //查找删除
                if (null != oldNoticeIds && oldNoticeIds.size() > 0){
                    for (SiteProtectNotice oldNotice : oldNoticeList){
                        Long noticeId = oldNotice.getId();
                        if (!newNoticeIds.contains(noticeId)){
                            Date oldEndTime = oldNotice.getEndTime();
                            Date oldStartTime = oldNotice.getStartTime();
                            Date currentDate = new Date();
                            if (currentDate.before(oldEndTime) && oldStartTime.before(currentDate)){
                                //进行中的维护不能删除
                                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_START.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_START.getMsg());
                            }
                            if (currentDate.after(oldEndTime)){
                                //已结束的维护不能删除
                                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NOTICE_END.getErrCode(),SiteProtectExceptionEnum.PROTECT_NOTICE_END.getMsg());
                            }
                            //删除
                            protectNoticeService.deleteById(noticeId);
                        }
                    }

                }
            }
        }
        //update siteProtect
        SiteProtect oldSiteProtect = getSiteProtectById(siteProtectId);
        copyProtectProperties(siteProtectReq,oldSiteProtect);
        setFlag(oldSiteProtect,siteProtectReq);
        updateById(oldSiteProtect);
        //get old site type (ALL or PART)
        List<SiteProtectRel> oldProtectRelList = getProtectRelByProtectId(siteProtectId);
        List<SiteResq> newSiteList = siteProtectReq.getSiteList();
        List<SiteProtectRel> newProtectRelList = compactProtectRels(newSiteList,siteProtectId);
        protectRelMapper.deleteRelBatchByProtectId(siteProtectId);
        if (null != newProtectRelList && !newProtectRelList.isEmpty()){
            siteProtectRelService.insertBatch(newProtectRelList);
        }
        redisCache.delete(upProtectRedisKey);
        //delete redis cache
        String prefix = SiteProtectConstants.SITE_PROTECT_MESSAGE_PREF;
        HashMap<String,Set<String>> delMap = new HashMap<>();
        if (null != oldProtectRelList && !oldProtectRelList.isEmpty()){
            oldProtectRelList.forEach(siteProtectRel -> {
                Long siteId = siteProtectRel.getSiteId();
                Site site = siteMapper.selectById(siteId);
                String siteCode = site.getCode();
                Long companyId = site.getCompanyId();
                Company company = companyMapper.selectById(companyId);
                String companyCode = company.getCode();
                String key = prefix + companyCode;
                Set<String> siteSet;
                if (delMap.containsKey(key)){
                    siteSet = delMap.get(key);
                }else {
                    siteSet = new HashSet<>();
                }
                siteSet.add(siteCode);
                delMap.put(key,siteSet);
            });
            delMap.forEach((key,value) -> {
                redisCache.hdel(key,value.toArray());
            });
        }
        return siteProtectId;
    }

    private List<Long> getNoticeIds(List<SiteNoticeReq> list) {
        List<Long> ids = new ArrayList<>();
        if (null != list && list.size() > 0){
            list.forEach(req -> {
                Long noticeId = req.getId();
                ids.add(noticeId);
            });
        }
        return ids;
    }

    private SiteProtectNotice getNewAddNotice(SiteNoticeReq newNotice, Long siteProtectId) {
        RequestContext context = ContextHolder.get();
        Long accountId = context.getAccountId();
        String accountName = context.getAccountName();
        SiteProtectNotice addNotice = new SiteProtectNotice();
        BeanUtils.copyProperties(newNotice,addNotice);
        addNotice.setId(idGenerator.generate());
        addNotice.setCreateById(accountId);
        addNotice.setCreateByName(accountName);
        addNotice.setSiteProtectId(siteProtectId);

        return addNotice;
    }


    private List<Long> getOldNoticeIds(List<SiteProtectNotice> oldNoticeList) {
        List<Long> ids = new ArrayList<>();
        oldNoticeList.forEach(notice -> {
            ids.add(notice.getId());
        });
        return ids;
    }

    private SiteProtectNotice compactNotice(SiteNoticeReq newNotice, SiteProtectNotice oldNotice) {
        RequestContext context = ContextHolder.get();
        Long accountId = context.getAccountId();
        String accountName = context.getAccountName();
        oldNotice.setMessage(newNotice.getMessage());
        oldNotice.setStartTime(newNotice.getStartTime());
        oldNotice.setEndTime(newNotice.getEndTime());
        oldNotice.setUpdateByName(accountName);
        oldNotice.setUpdateById(accountId);
        return oldNotice;
    }

    /**
     * 根据前端传入自定义站点列表，组装关系表实体
     * @param newSiteList 前端传入站点列表
     * @return 关系表list
     */
    private List<SiteProtectRel> compactProtectRels(List<SiteResq> newSiteList,Long siteProtectId) {
        List<SiteProtectRel> list = new ArrayList<>();
        if (null == newSiteList || newSiteList.size() == 0){
            return list;
        }
        RequestContext context = ContextHolder.get();
        Long accountId = context.getAccountId();
        String accountName = context.getAccountName();
        newSiteList.forEach(newRel -> {
            SiteProtectRel rel = new SiteProtectRel();
            rel.setId(idGenerator.generate());
            rel.setSiteId(newRel.getSiteId());
            rel.setSiteProtectId(siteProtectId);
            rel.setCreateById(accountId);
            rel.setCreateByName(accountName);
            rel.setUpdateById(accountId);
            rel.setUpdateByName(accountName);
            list.add(rel);
        });
        return list;
    }

    //获取需要修改的自定义站点列表
    private List<SiteProtectRel> compareAndGetUpSiteProtectRelList(List<SiteProtectRel> oldRelList, List<SiteResq> newSiteList) {
        RequestContext context = ContextHolder.get();
        Long accountId = context.getAccountId();
        String accountName = context.getAccountName();
        List<Long> ids = getUpRelIds(newSiteList);
        List<SiteProtectRel> upRelList = new ArrayList<>();
        oldRelList.forEach(oldRel -> {
            Long relId = oldRel.getId();
            oldRel.setUpdateById(accountId);
            oldRel.setUpdateByName(accountName);
            if (!ids.contains(relId)){
                oldRel.setDeleted(true);
            }
            upRelList.add(oldRel);
        });
        return upRelList;
    }

    //获取修改的rel表id
    private List<Long> getUpRelIds(List<SiteResq> newSiteList) {
        List<Long> ids = new ArrayList<>();
        newSiteList.forEach(siteResq -> {
            Long id = siteResq.getId();
            if (null != id && id > 0){
                ids.add(id);
            }
        });
        return ids;
    }

    //获取需要新增的自定义站点列表
    private List<SiteProtectRel> compareAndGetInsertSiteProtectRelList(List<SiteResq> newSiteList,Long sitProtectId) {
        RequestContext requestContext = ContextHolder.get();
        List<SiteProtectRel> insertRelList = new ArrayList<>();
        newSiteList.forEach(siteResq -> {
            Long id = siteResq.getId();
            if (null == id || id == 0){
                SiteProtectRel rel = new SiteProtectRel();
                BeanUtils.copyProperties(siteResq,rel);
                rel.setSiteProtectId(sitProtectId);
                rel.setId(idGenerator.generate());
                rel.setCreateById(requestContext.getAccountId());
                rel.setCreateByName(requestContext.getAccountName());
                insertRelList.add(rel);
            }
        });
        return insertRelList;
    }

    private List<SiteProtectRel> getProtectRelByProtectId(Long siteProtectId) {

        SiteProtectRel rel = new SiteProtectRel();
        rel.setSiteProtectId(siteProtectId);
        rel.setDeleted(false);
        EntityWrapper wrapper = new EntityWrapper(rel);
        return siteProtectRelService.selectList(wrapper);
    }

    private void copyProtectProperties(SiteProtectReq siteProtectReq, SiteProtect oldSiteProtect) {
        oldSiteProtect.setName(siteProtectReq.getName());
        oldSiteProtect.setUpdateById(siteProtectReq.getUpdateById());
        oldSiteProtect.setUpdateByName(siteProtectReq.getUpdateByName());
    }

    private SiteProtect getSiteProtectById(Long siteProtectId) {
        SiteProtect select = new SiteProtect();
        select.setId(siteProtectId);
        select.setDeleted(false);
        EntityWrapper<SiteProtect> wrapper = new EntityWrapper<>(select);
        return selectOne(wrapper);
    }


    //设置站点通知类型标记
    private void setFlag(SiteProtect siteProtect, SiteProtectReq siteProtectReq) {
        List<SiteNoticeReq> siteNoticeReqList = siteProtectReq.getSiteNoticeReqList();
        if (null != siteNoticeReqList && siteNoticeReqList.size() > 0){
            for (SiteNoticeReq siteNoticeReq : siteNoticeReqList){
                String noticeType = siteNoticeReq.getNoticeType().toString();
                if (SiteNoticeEnum.INNER_NOTICE.toString().equals(noticeType)){
                    siteProtect.setInnerFlag(true);
                }else if (SiteNoticeEnum.CLOSE_NOTICE.toString().equals(noticeType)){
                    siteProtect.setCloseFlag(true);
                }
            }
        }
    }

    private Long setAndInsertSiteProtect(SiteProtectReq siteProtectReq) {
        SiteProtect siteProtect = new SiteProtect();
        BeanUtils.copyProperties(siteProtectReq,siteProtect);
        setFlag(siteProtect,siteProtectReq);
        Long siteProtectId = idGenerator.generate();
        siteProtect.setId(siteProtectId);
        insert(siteProtect);
        return siteProtectId;
    }

    //check site protect name
    private void checkSiteProtectIsRepeat(SiteProtectReq siteProtectReq,RequestContext context) {
        String protectName = siteProtectReq.getName();
        if (StringUtils.isEmpty(protectName)){
            log.error("[新增站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 新增站点维护失败! 维护名为空! 新增实体 siteProtectReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NAME_IS_NULL.getErrCode(),SiteProtectExceptionEnum.PROTECT_NAME_IS_NULL.getMsg());
        }
        SiteProtect selectProtect = new SiteProtect();
        selectProtect.setName(protectName);
        selectProtect.setDeleted(false);
        List<SiteProtect> selectByNameList = siteProtectMapper.selectList(new EntityWrapper<>(selectProtect));
        log.info("[站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}]. 站点维护根据维护name={} 查询结果：{} "
                ,context.getSiteId(),context.getSiteCode(),context.getCompanyId(),context.getAccountId()
                ,protectName, JSONObject.toJSONString(selectByNameList));
        if (null != selectByNameList && selectByNameList.size() > 1){
            log.error("[新增站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 新增站点维护失败! 维护名已存在! 新增实体 siteProtectReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),siteProtectReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NAME_REPEAT.getErrCode(),SiteProtectExceptionEnum.PROTECT_NAME_REPEAT.getMsg());
        }else if (null != selectByNameList && selectByNameList.size() == 1){
            //update operate check id is same
            SiteProtect siteProtect = selectByNameList.get(0);
            Long id = siteProtect.getId();
            Long updateProtectId = siteProtectReq.getId();

            if (!(null != updateProtectId && (id.longValue() == updateProtectId.longValue()))){
                throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_NAME_REPEAT.getErrCode(),SiteProtectExceptionEnum.PROTECT_NAME_REPEAT.getMsg());
            }
        }
    }

    /**
     * 删除站点维护
     *
     * @param protectDelReq 站点维护唯一id和通知id
     * @return 删除成功返回true，否则返回false
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean deleteSiteProtect(SiteProtectDelReq protectDelReq) {
        RequestContext context = ContextHolder.get();
        Long accountId = context.getAccountId();
        String accountName = context.getAccountName();
        Long siteProtectId = protectDelReq.getSiteProtectId();
        Long noticeId = protectDelReq.getNoticeId();
        boolean isValid = protectDelReq.isValid();
        if (!isValid){
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_DATA_INVALID.getErrCode(),SiteProtectExceptionEnum.PROTECT_DATA_INVALID.getMsg());
        }
        String delProtectRedisKey = SiteProtectConstants.SITE_PROTECT_DEL_REDIS_PREF + siteProtectId;
        boolean lock = redisCache.setIfAbsent(delProtectRedisKey, JSONObject.toJSONString(protectDelReq));
        if (!lock){
            log.info("[删除站点维护siteId = {} ,siteCode = {},companyId = {} , accountId = {}] 重复提交! delReq = {}"
                    ,context.getSiteId(),context.getSiteCode()
                    ,context.getCompanyId(),context.getAccountId(),protectDelReq.toString());
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getErrCode(),SiteProtectExceptionEnum.PROTECT_SUBMIT_REPEAT.getMsg());
        }

        SiteProtect siteProtect = getSiteProtectById(siteProtectId);
        List<SiteProtectNotice> noticeList = getProtectNoticeByProtectId(siteProtectId);
        //get and del notice
        protectNoticeService.deleteById(noticeId);
        if (null != noticeList && noticeList.size() == 1){
            //delete site protect relation table
            protectRelMapper.deleteRelBatchByProtectId(siteProtectId);
            //delete site protect
            siteProtect.setUpdateById(accountId);
            siteProtect.setUpdateByName(accountName);
            siteProtect.setDeleted(true);
            updateById(siteProtect);
        }
        return true;
    }

    private List<SiteProtectNotice> getProtectNoticeByProtectId(Long siteProtectId) {
        SiteProtectNotice selectNotice = new SiteProtectNotice();
        selectNotice.setSiteProtectId(siteProtectId);
        selectNotice.setDeleted(false);
        EntityWrapper<SiteProtectNotice> noticeWrapper = new EntityWrapper<>();
        noticeWrapper.setEntity(selectNotice);
        return protectNoticeService.selectList(noticeWrapper);
    }

    @Override
    @Transactional
    public List<SiteProtectExportDTO> list(List<Long> siteNoticeIdList) {
        List<SiteProtectExportDTO> result = Lists.newArrayList();

        // 一定要有维护通知记录的id
        if(org.apache.commons.collections.CollectionUtils.isNotEmpty(siteNoticeIdList)) {
            // 查询指定的通知记录
            EntityWrapper<SiteProtectNotice> wrapper = new EntityWrapper();
            wrapper.in("id",siteNoticeIdList);
            List<SiteProtectNotice> siteProtectNotices = siteProtectNoticeMapper.selectList(wrapper);
            // 通知所有站点
            List<SiteProtectNotice> allNotice = siteProtectNotices.stream().filter(a -> SiteProtectEnum.ALL == a.getSiteType()).collect(Collectors.toList());
            if(org.apache.commons.collections.CollectionUtils.isNotEmpty(allNotice)) {
                List<Long> collect = allNotice.stream().map(SiteProtectNotice::getId).collect(Collectors.toList());
                List<SiteProtectExportDTO> allSite = siteProtectNoticeMapper.selectExportDataALL(collect);
                result.addAll(allSite);
            }
            // 通知部分站点
            List<SiteProtectNotice> partNotice = siteProtectNotices.stream().filter(a -> SiteProtectEnum.PART == a.getSiteType()).collect(Collectors.toList());
            if(org.apache.commons.collections.CollectionUtils.isNotEmpty(partNotice)){
                List<Long> collect = partNotice.stream().map(SiteProtectNotice::getId).collect(Collectors.toList());
                List<SiteProtectExportDTO> allSite = siteProtectNoticeMapper.selectExportDataPart(collect);
                result.addAll(allSite);
            }
        }
        return result;
    }

    @Override
    @Transactional
    public SiteProtectDetailVO queryOne(Long id) {
        SiteProtectDetailVO siteProtectDetailVO = new SiteProtectDetailVO();
        if(id==null||0==id){
            return siteProtectDetailVO;
        }
        SiteProtect siteProtect = getSiteProtectById(id);
        List<SiteProtectNotice> notices = getProtectNoticeByProtectId(id);
        if(siteProtect==null|| org.apache.commons.collections.CollectionUtils.isEmpty(notices)){
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.NO_DETAIL_ID.getErrCode(),SiteProtectExceptionEnum.NO_DETAIL_ID.getMsg());
        }

        BeanUtils.copyProperties(siteProtect,siteProtectDetailVO);
        List<NoticeVO> noticesList = converterNoticeVO(notices);
        siteProtectDetailVO.setNoticeVOList(noticesList);
        log.info("-----------------noticeVO----{}",JSONObject.toJSONString(noticesList));
        List<SiteProtectRel> siteProtectRels = getProtectRelByProtectId(id);
        if(org.apache.commons.collections.CollectionUtils.isNotEmpty(siteProtectRels)){
            List<Long> siteIds = siteProtectRels.stream().map(SiteProtectRel::getSiteId).collect(Collectors.toList());
            log.info("-----------------siteIds-----{}",JSONObject.toJSONString(siteIds));
            List<SiteCompanyVO> siteCompanys = siteMapper.getSiteListByIds(siteIds);
            log.info("-----------------siteCompanys-----{}",JSONObject.toJSONString(siteCompanys));
            siteProtectDetailVO.setSiteCompanyS(siteCompanys);
        }
        return siteProtectDetailVO;
    }

    private List<NoticeVO> converterNoticeVO(List<SiteProtectNotice> notices) {
        List<NoticeVO> noticeVOList = Lists.newArrayList();
        notices.forEach(
                a->{
                    NoticeVO noticeVO = new NoticeVO();
                    noticeVO.setId(a.getId().toString());
                    noticeVO.setEndTime(DateUtil.format(a.getEndTime(),"yyyy-MM-dd HH:mm"));
                    noticeVO.setStartTime(DateUtil.format(a.getStartTime(),"yyyy-MM-dd HH:mm"));
                    noticeVO.setMessage(a.getMessage());
                    noticeVO.setSiteNoticeEnum(a.getNoticeType());
                    Date date = new Date();
                    if(date.before(a.getStartTime())){
                        noticeVO.setSiteProtectStatus(SiteProtectConstants.NOT_START);
                    }else if(date.after(a.getStartTime())&&date.before(a.getEndTime())){
                        noticeVO.setSiteProtectStatus(SiteProtectConstants.RUNNING);
                    }else{
                        noticeVO.setSiteProtectStatus(SiteProtectConstants.FINISH);
                    }
                    noticeVOList.add(noticeVO);
                }
        );
        return noticeVOList;
    }

    @Override
    public SiteProtectNotice findProtectNoticeBySiteCode(String siteCode, String companyCode) {

        // 查询指定公司指定站点是否存在
        Site site = siteMapper.selectSitebyCode(siteCode, companyCode);
        if (site == null) {
            // 不存在直接返回
            return null;
        }
        // 查询当前生效的关站通知信息
        List<SiteProtectNotice> validNotice = siteProtectNoticeMapper.selectValidList(site.getId());
        if (CollectionUtils.isEmpty(validNotice)) {
            return null;
        }
        return validNotice.get(0);
    }

    /**
     * 查询用户站点下的维护通知
     * @param siteId 站点id
     * @param companyId 公司id
     * @return 维护列表
     */
    @Override
    public List<SiteProtectUserVO> getUserProtect(Long siteId, Long companyId) {
        if (null == siteId){
            log.info("[用户端查询站点维护] 站点siteCode为空!");
            throw new SiteProtectBusinessException(SiteProtectExceptionEnum.PARAM_NULL.getErrCode(),SiteProtectExceptionEnum.PARAM_NULL.getMsg());
        }
        List<SiteProtectUserVO> userVOList = new ArrayList<>();
        //查询当前未结束的时间范围内的站内通知；判断是否包含ALL
//        List<SiteProtect> siteProtects = siteProtectMapper.selectSiteProtectByDate();
        List<SiteProtectNotice> noticeLists = protectNoticeMapper.selectListEndTimeBeforeCurrent();
        List<SiteProtectNoticeVO> noticeVOList = new ArrayList<>();
        for (SiteProtectNotice notice : noticeLists){
            SiteProtectEnum siteType = notice.getSiteType();
            if (SiteProtectEnum.ALL.toString().equals(siteType.toString())){
                SiteProtectUserVO userVO = new SiteProtectUserVO();
                String message = notice.getMessage();
                SiteProtectNoticeVO noticeVO = new SiteProtectNoticeVO();
                noticeVO.setMessage(message);
                noticeVOList.add(noticeVO);
                userVO.setNoticeList(noticeVOList);
                userVOList.add(userVO);
            }else if ((SiteProtectEnum.PART.toString().equals(siteType.toString()))){
                //不包含则去查询关系表
                List<SiteProtectRel> protectRelList = protectRelMapper.selectBySiteId(siteId);
                if (null != protectRelList && !protectRelList.isEmpty()){
                    Set<Long> siteProtectIdSet = new HashSet<>();
                    protectRelList.forEach(siteProtectRel -> {

                        Long siteProtectId = siteProtectRel.getSiteProtectId();
                        siteProtectIdSet.add(siteProtectId);
                    });
                    //根据维护id查询维护信息
                    siteProtectIdSet.forEach(siteProtectId -> {
                        SiteProtectUserVO userVO = new SiteProtectUserVO();
                        SiteProtect siteProtect = selectById(siteProtectId);
                        String siteProtectName = siteProtect.getName();
                        userVO.setName(siteProtectName);
                        List<SiteProtectNotice> noticeList = protectNoticeMapper.selectListByProtectId(siteProtectId);
                        if (null != noticeList && !noticeList.isEmpty()){
                            noticeList.forEach(siteProtectNotice -> {
                                SiteProtectNoticeVO noticeVO = new SiteProtectNoticeVO();
                                noticeVO.setMessage(siteProtectNotice.getMessage());
                                noticeVO.setStartTime(siteProtectNotice.getStartTime());
                                noticeVO.setEndTime(siteProtectNotice.getEndTime());
                                noticeVO.setNoticeType(siteProtectNotice.getNoticeType());
                                noticeVOList.add(noticeVO);
                            });
                            userVO.setNoticeList(noticeVOList);
                        }
                        userVOList.add(userVO);
                    });
                    return userVOList;
                }
            }
        }
        return userVOList;
    }
}
