package com.yizhi.system.application.service.impl;

import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.exceptions.ClientException;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.enums.SmsSendBizType;
import com.yizhi.core.application.enums.SmsSendMode;
import com.yizhi.core.application.enums.SmsTemplateType;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.system.application.domain.SmsBatchSend;
import com.yizhi.system.application.domain.SmsSendRecord;
import com.yizhi.system.application.mapper.SmsSendRecordMapper;
import com.yizhi.system.application.service.ISmsBatchSendService;
import com.yizhi.system.application.service.ISmsSendRecordService;
import com.yizhi.system.application.util.SmsUtil;
import com.yizhi.util.application.enums.i18n.Constants;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.stream.Collectors;

/**
 * <p>
 * 短信发送记录表 服务实现类
 * </p>
 *
 * @author 谢海军123
 * @since 2018-12-06
 */
@Service
public class SmsSendRecordServiceImpl extends ServiceImpl<SmsSendRecordMapper, SmsSendRecord> implements ISmsSendRecordService {

    private static final Logger log = LoggerFactory.getLogger(SmsSendRecordServiceImpl.class);

    // 默认的短信签名
    //public final static String DEFALAT_SIGN_NAME = "上海复深蓝软件股份有限公";
    public final static String DEFALAT_SIGN_NAME = "未木云";
    // 默认的短信模板id
    //public final static String DEFALAT_TEMPLATE_CODE = "SMS_152151075";
    public  static String DEFALAT_TEMPLATE_CODE;
    // 每分钟最大发送次数
    public final static int MAX_SEND_COUNT = 6;

    @Autowired
    private ISmsSendRecordService smsSendRecordService;

    @Autowired
    private ISmsBatchSendService smsBatchSendService;

    @Autowired
    private IdGenerator idGenerator;

    @Value("${sms.template.code:SMS_180056909}")
    public void setDefalatTemplateCode(String code){
        DEFALAT_TEMPLATE_CODE = code;
    }

    /**
     * 发送短信（可选参数，如果不填使用null）
     *
     * @param templateParam   短信模板的占位参数替换（可选 如果短信模板没有占位参数，不要填）
     * @param smsUpExtendCode 可选-上行短信扩展码(扩展码字段控制在7位或以下，无特殊需求用户请忽略此字段) "90997"
     * @param businessLogo 业务标识
     * @param outId           可选:outId为提供给业务方扩展字段,最终在短信回执消息中将此值带回给调用者
     * @param smsSendBizType  必填:发送短信的业务类型
     * @param phoneNumber     必填:待发送手机号。发送国际/港澳台消息时，接收号码格式为国际区号+号码，如85200000000;多个使用数组
     * @return
     */
    @Override
    @Transactional(rollbackFor = RuntimeException.class)
    public boolean sendSms(String templateParam, String smsUpExtendCode, Long businessLogo,
                           String outId, SmsSendBizType smsSendBizType, Long... phoneNumber) throws BizException
    {
        commSmsSend(templateParam, smsUpExtendCode, businessLogo, outId, smsSendBizType, phoneNumber,null);
        //businessLogo

        return true;
    }

    public void commSmsSend(String templateParam, String smsUpExtendCode, Long businessLogo,
                            String outId, SmsSendBizType smsSendBizType, Long[] phoneNumber,
                            String extend) {
        Calendar strateTime = Calendar.getInstance();
        long createTimelong = strateTime.getTimeInMillis();
        Date createTime = new Date(createTimelong);
        strateTime.set(Calendar.SECOND,0);

        log.info("发送短信参数：strateTime={},createTime={}",strateTime,createTime);

        RequestContext rc = ContextHolder.get();
        String signName = DEFALAT_SIGN_NAME;
        String templateCode = DEFALAT_TEMPLATE_CODE;
        String templateType = SmsTemplateType.VERIFICATION_CODE.getCode();
        int size = 0;
        if(phoneNumber!=null){
            size = phoneNumber.length;
        }
        if(size>0){
            if(size==1){
                // 判断手机号发送是否超限制
                EntityWrapper<SmsSendRecord> ew = QueryUtil.condition(new SmsSendRecord());
                ew.eq("phone_num",Long.valueOf(phoneNumber[0]));
                ew.eq("company_id",rc.getCompanyId());
                Date endDate = new Date(strateTime.getTimeInMillis()+60000);
                ew.between("create_date",strateTime.getTime(),endDate);
                ew.eq("biz_type",smsSendBizType.getCode());
                int countNum = smsSendRecordService.selectCount(ew);
                if(countNum>MAX_SEND_COUNT){
                    // 短信超过发送次数限制
                    throw new BizException(Constants.SYSTEM_MSG_SMS_SEND_MAX_LIMIT.getCode().toString()
                            ,Constants.SYSTEM_MSG_SMS_SEND_MAX_LIMIT.getName());
                }
            }

            if(smsSendBizType != null){
                // 根据类型查询配置信息，（短信签名和短信模板id,模板类型）
            }
        }

        // 单次发送
        if(size==1){
            try {
                SendSmsResponse ret = SmsUtil.simpleSend(signName,templateCode,templateParam,smsUpExtendCode,outId,phoneNumber[0]);
                if(ret.getCode() != null && ret.getCode().equals("OK")) {
                    String sendMode = SmsSendMode.SIMPLE.getCode();
                    saveSendRecord(templateParam, smsUpExtendCode, businessLogo, outId, smsSendBizType, createTime, rc, signName, templateCode, templateType, ret, sendMode, phoneNumber[0],extend);
                }else {
                    if(ret.getCode() != null && ret.getCode().equals("isv.MOBILE_NUMBER_ILLEGAL")){
                        // 非法手机号
                        throw new BizException(Constants.SYSTEM_MSG_MOBILE_NUMBER_ILLEGAL.getCode().toString()
                                ,Constants.SYSTEM_MSG_MOBILE_NUMBER_ILLEGAL.getName());
                    }else if(ret.getCode() != null && ret.getCode().equals("isv.isv.BUSINESS_LIMIT_CONTROL")){// 触发分钟级流控Permits:1
                        // 手机号发送太频繁限制
                        throw new BizException(Constants.SYSTEM_MSG_SMS_SEND_MAX_LIMIT.getCode().toString()
                                ,Constants.SYSTEM_MSG_SMS_SEND_MAX_LIMIT.getName());
                    }
                    // 短信发送异常，请重试或者联系管理员
                    throw new BizException(Constants.SYSTEM_MSG_SMS_SEND_ERROR.getCode().toString()
                            ,Constants.SYSTEM_MSG_SMS_SEND_ERROR.getName());
                }
            } catch (ClientException e) {
                log.info("发送短信异常{}",e);
                // 短信发送异常，请重试或者联系管理员
                throw new BizException(Constants.SYSTEM_MSG_SMS_SEND_ERROR.getCode().toString()
                        ,Constants.SYSTEM_MSG_SMS_SEND_ERROR.getName());
            }
        }else{// 多次发送
            int pageNo = 1;
            int pageSize = 1000;
            boolean next =true;
            do{
                String phoneNumbers = Arrays.stream(phoneNumber).
                        skip((pageNo-1)*pageSize).limit(pageNo*pageSize).
                        map(item->item.toString()).collect(Collectors.joining(","));
                if(StringUtils.isBlank(phoneNumbers)){
                    next =false;
                    break;
                }
                try {
                    SendSmsResponse ret = SmsUtil.batchSend(signName,templateCode,templateParam,smsUpExtendCode,outId,phoneNumbers);
                    String sendMode = SmsSendMode.BATCH.getCode();// 批量插入类型
                    SmsBatchSend smsBatchSend = new SmsBatchSend();
                    long batchId = idGenerator.generate();
                    smsBatchSend.setId(batchId);
                    smsBatchSend.setBatchPhoneNum(phoneNumbers);
                    smsBatchSendService.insert(smsBatchSend);
                    saveSendRecord(templateParam, smsUpExtendCode, businessLogo, outId, smsSendBizType, createTime, rc, signName, templateCode, templateType, ret, sendMode, batchId,extend);
                } catch (ClientException e) {
                    log.info("批量发送短信异常{}",e);
                    // 短信发送异常，请重试或者联系管理员
                    /*throw new BizException(Constants.SYSTEM_MSG_SMS_SEND_ERROR.getCode().toString()
                            ,Constants.SYSTEM_MSG_SMS_SEND_ERROR.getName());*/
                }
                pageNo++;
            }while (next);

        }
    }

    @Override
    public boolean sendSms(String templateParam, String smsUpExtendCode, Long businessLogo, String outId, SmsSendBizType smsSendBizType, String extend, Long... phoneNumber) {
        commSmsSend(templateParam, smsUpExtendCode, businessLogo, outId, smsSendBizType, phoneNumber,extend);
        return true;
    }


    private void saveSendRecord(String templateParam, String smsUpExtendCode, Long businessLogo, String outId, SmsSendBizType smsSendBizType, Date createTime, RequestContext rc,
                                String signName, String templateCode, String templateType,
                                SendSmsResponse ret, String sendMode, Long phone,String extend) {
        //请求成功
        SmsSendRecord sr = new SmsSendRecord();
        sr.setId(idGenerator.generate());
        sr.setBizId(ret.getBizId());
        sr.setPhoneNum(phone);
        sr.setBizType(smsSendBizType.getCode());
        sr.setBusinessLogo(businessLogo);
        sr.setCompanyId(rc.getCompanyId());
        sr.setCreateDate(createTime);
        sr.setOutId(outId);
        sr.setSignName(signName);
        sr.setSendType(sendMode);
        sr.setSiteId(rc.getSiteId());
        sr.setExtend(extend);
        sr.setSmsUpExtendCode(smsUpExtendCode);
        sr.setTemplateCode(templateCode);
        sr.setTemplateParam(templateParam);
        sr.setTemplateType(templateType);
        smsSendRecordService.insert(sr);
    }
}
