package com.yizhi.system.application.service.using;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.system.application.mapper.CompanyMapper;
import com.yizhi.system.application.mapper.OrganizationMapper;
import com.yizhi.system.application.service.IOrganizationService;
import com.yizhi.system.application.domain.Company;
import com.yizhi.system.application.domain.Organization;
import com.yizhi.system.application.util.OrgIndexUtil;
import com.yizhi.system.application.util.RightLeftIndex;
import com.yizhi.util.application.tree.CommonTree;
import com.yizhi.util.application.tree.TreeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @Author: shengchenglong
 * @Date: 2018/8/20 13:27
 */
@Component
public class OrganizationIndexService {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationIndexService.class);

    @Autowired
    private RedisTemplate redisTemplate;
    @Autowired
    private OrganizationMapper organizationMapper;
    @Autowired
    private IOrganizationService organizationService;
    @Autowired
    private OrganizationCacheService organizationCacheService;
    @Autowired
    private CompanyMapper companyMapper;

    public List<RightLeftIndex> getIndices(Collection<Long> orgIds) {
        if (CollectionUtils.isNotEmpty(orgIds)) {
            List<Organization> organizations = organizationMapper.selectBatchIds(orgIds);
            if (CollectionUtils.isNotEmpty(organizations)) {
                return OrgIndexUtil.getChildrenRange(organizations);
            }
        }
        return null;
    }

    public Long initLeftRight(String companyCode) {
        Company company = new Company();
        company.setCode(companyCode);
        company = company.selectOne(new EntityWrapper(company));

        if (company != null && company.getId() != null) {
            addInitTask(company.getId());
        }
        return company.getId();
    }

    /**
     * 初始化一个公司的组织的左右索引
     *
     * @param companyId
     */
    public void initLeftRight(Long companyId) {
        Organization organization = new Organization();
        organization.setDeleted(false);
        List<Organization> organizations = null;
        List<CommonTree> trees = null;

        organization.setCompanyId(companyId);
        organizations = organization.selectList(new EntityWrapper(organization));

        if (CollectionUtils.isEmpty(organizations)){
            LOGGER.info("初始化索引该公司下没有查询到部门，公司id：{}",companyId);
            return;
        }
        if (CollectionUtils.isNotEmpty(organizations)) {
            trees = new ArrayList<>();
            for (Organization o : organizations) {
                CommonTree tree = new CommonTree();
                tree.setId(o.getId());
                tree.setPId(o.getParentId());
                trees.add(tree);
            }

            trees = TreeUtil.buildTree(trees, 0L);
            trees = TreeUtil.buildAndTraverseRightLeftIndex(trees);

            organizations = new ArrayList<>();
            Organization o = new Organization();
            for (CommonTree tree : trees) {
                o.setId((Long) tree.getId());
                o.setLeftIndex(tree.getLeft());
                o.setRightIndex(tree.getRight());
                o.updateById();
            }
        }

        // 清除缓存
        LOGGER.info("初始化索引传参：{}",companyId);
        organizationService.deleteCompanyInfo(companyId);
        // 初始化企业下组织cache
        organizationCacheService.cacheCompanyOrganization(companyMapper.selectById(companyId));
    }

    /**
     * 处理添加一个部门
     * 未完成
     *
     * @param companyId
     * @param orgId
     */
    public void addOneOrg(Long companyId, Long orgId, Long parentId) {
        // 查出当前部门的父部门
        Organization parent = new Organization();
        parent.setId(parentId);
        parent.setCompanyId(companyId);
        parent = parent.selectById();
        if (null != parent) {
            int pRightIndex = parent.getRightIndex();
            // 更新右边节点索引
            organizationMapper.updateIndexAddOneSetpOne(pRightIndex);
            organizationMapper.updateIndexAddOneSetpTwo(pRightIndex - 1);
            // 计算当前新增节点索引
            Organization org = new Organization();
            org.setId(orgId);
            org.setLeftIndex(pRightIndex);
            org.setRightIndex(pRightIndex + 1);
            org.updateById();
            // 更新父级右边索引
            parent = new Organization();
            parent.setId(parentId);
            parent.setRightIndex(pRightIndex + 2);
            parent.updateById();
        }
    }

    public void initParentIdLayer(long companyId) {
        List<Organization> orgs = new ArrayList<>();
        List<Long> parentIds = new ArrayList<>();
        Map<Long, String> parentIdLayerMap = new HashMap<>();

        int layer = 0;

        Organization organization = new Organization();
        organization.setParentId(0L);
        organization.setCompanyId(companyId);
        List<Organization> organizations = organizationMapper.selectList(new EntityWrapper<>(organization));
        for (Organization parent : organizations) {
            parent.setRootId(0L);
            parent.setParentIdLayer(String.valueOf(parent.getId()));
            parent.setLayer(layer);
            parentIdLayerMap.put(parent.getId(), parent.getParentIdLayer());
            orgs.add(parent);
            parentIds.add(parent.getId());
        }
        recycle(companyId, parentIds, parentIdLayerMap, orgs, layer);

        Organization or = new Organization();
        for (Organization o : orgs) {
            or.setId(o.getId());
            or.setParentIdLayer(o.getParentIdLayer());
            or.setLayer(o.getLayer());
            or.setRootId(o.getRootId());
            or.updateById();
        }
    }

    private void recycle(long companyId, List<Long> parentIds, Map<Long, String> parentIdLayerMap, List<Organization> orgs, int layer) {
        if (CollectionUtils.isNotEmpty(parentIds)) {
            EntityWrapper<Organization> ew = new EntityWrapper<>();
            ew.in("parent_id", parentIds);
            ew.eq("company_id", companyId);
            List<Organization> organizations = organizationMapper.selectList(ew);
            parentIds = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(organizations)) {
                layer++;
                for (Organization parent : organizations) {
                    parent.setParentIdLayer(parentIdLayerMap.get(parent.getParentId()) + "_" + parent.getId());
                    parent.setLayer(layer);
                    parent.setRootId(Long.valueOf(parent.getParentIdLayer().split("_")[0]));
                    parentIdLayerMap.put(parent.getId(), parent.getParentIdLayer());
                    orgs.add(parent);
                    parentIds.add(parent.getId());
                }
                recycle(companyId, parentIds, parentIdLayerMap, orgs, layer);
            }
        }
    }

    /**
     * 增加一个待处理
     */
    public Boolean addInitTask(Long companyId) {
        return redisTemplate.opsForZSet().add(CacheNamespace.ORG_INDEX_INIT_KEY, String.valueOf(companyId),System.currentTimeMillis());
    }

}
