package com.yizhi.application;

import com.yizhi.util.application.constant.GlobalConstant;
import feign.*;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.autoconfigure.amqp.RabbitAutoConfiguration;
import org.springframework.cloud.netflix.zuul.EnableZuulProxy;
import org.springframework.cloud.openfeign.EnableFeignClients;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;
import org.springframework.web.filter.CorsFilter;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

//import org.springframework.cloud.netflix.feign.EnableFeignClients;

@EnableZuulProxy
@SpringBootApplication(exclude = {RabbitAutoConfiguration.class})
@ComponentScan(basePackages = {"com.yizhi"})
@EnableFeignClients(basePackages = "com.yizhi.system.application.system.remote")
public class GatewayApplication {

	@Value("${custom.zuul.allowedOrigin}")
	private List<String> zuulAllowedOrigin;

	public static void main(String[] args) {
		SpringApplication.run(GatewayApplication.class, args);
	}

	/***
	 * feign取消重试 默认5次（ConnectTimeout、ReadTimeout）
	 */
	@Bean
	Retryer feignRetryer() {
		return Retryer.NEVER_RETRY;
	}

	/***
	 * feign请求超时设置
	 */
	@Bean
	Request.Options requestOptions(ConfigurableEnvironment env) {
		int ribbonReadTimeout = env.getProperty("ribbon.ReadTimeout", int.class, 30000);
		int ribbonConnectionTimeout = env.getProperty("ribbon.ConnectTimeout", int.class, 6000);
		return new Request.Options(ribbonConnectionTimeout, ribbonReadTimeout);
	}

	@Bean
	public Feign.Builder feignBuilder() {
		return Feign.builder().requestInterceptor(new RequestInterceptor() {
			@Override
			public void apply(RequestTemplate requestTemplate) {
				ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
				HttpServletRequest request = attributes.getRequest();
				String token = request.getHeader(GlobalConstant.TOKEN_HEADER);
				requestTemplate.header(GlobalConstant.TOKEN_HEADER, token);
			}
		});
	}

	// 只要网关配置就好 ， 底下的服务不需要配置。两个同时配置还导致了 跨域失败 适得其反。
	@Bean
	public CorsFilter corsFilter() {
		final UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
		final CorsConfiguration config = new CorsConfiguration();
		config.setAllowCredentials(true); // 允许cookies跨域
		config.setAllowedOrigins(zuulAllowedOrigin);// #允许向该服务器提交请求的URI，*表示全部允许，在SpringMVC中，如果设成*，会自动转成当前请求头中的Origin
		config.addAllowedHeader("*");// #允许访问的头信息,*表示全部
		config.setMaxAge(18000L);// 预检请求的缓存时间（秒），即在这个时间段里，对于相同的跨域请求不会再预检了
		config.addAllowedMethod("*");// 允许提交请求的方法，*表示全部允许
		source.registerCorsConfiguration("/**", config);
		return new CorsFilter(source);
	}
}
