package com.yizhi.system.application.controller.remote;


import java.util.List;

import com.google.common.collect.Lists;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.system.application.vo.domain.Dictionary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.system.application.service.DictionaryService;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.domain.Response;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;

/**
 * <p>
 * 数据字典前端控制器
 * </p>
 *
 * @author shengchenglong
 * @since 2018-01-04
 */
@Api(tags = "DictionaryApi", description = "数据字典接口")
@RestController
@RequestMapping("/dictionary")
public class DictionaryController {

    private static final Logger LOG = LoggerFactory.getLogger(DictionaryController.class);

    @Autowired
    private DictionaryService dictionaryService;

    @Autowired
    private DomainConverter domainConverter;

    /**
     * 新增数据字典
     *
     * @param dictionary
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "新增一个字典数据", response = Response.class)
    @PostMapping("/insert")
    public Response<Long> insert(
            @ApiParam(
                    name = "dictionary",
                    value = "不能为空字段：</br>code</br>cnName</br>value</br>systemFlag</br>" +
                            "选传参数：</br>parentId:当为顶层节点时不传或传0</br>sort:同层级排序"
            )
            @RequestBody Dictionary dictionary) {
        try {
            com.yizhi.system.application.domain.Dictionary dictionary1 = new com.yizhi.system.application.domain.Dictionary();
            BeanUtils.copyProperties(dictionary,dictionary1);
            boolean success = dictionaryService.insertDetail(dictionary1);
            if (success) {
                return Response.ok(dictionary.getId());
            }else {
            	return Response.fail(ReturnCode.SAVE_FAIL);
            }
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }

    }

    /**
     * 新增搜索数据字典
     *
     * @param searchName
     * @return
     */
    @ApiOperation(value = "新增搜索数据字典", notes = "新增一个搜索的字典数据", response = Response.class)
    @GetMapping("/insertBySearchName")
    public boolean insertBySearchName(@RequestParam("searchName") String searchName) {
            return dictionaryService.insertBySearchName(searchName);
    }

    /**
     * 我的搜索历史记录
     *
     * @return
     */
    @ApiOperation(value = "我的搜索历史记录", notes = "查询所有我的搜索历史记录", response = Response.class)
    @GetMapping(value = "/home/searchListByMe")
    public Response<List<Dictionary>> searchListByMe() {
        try {
            List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.searchListByMe();

            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 删除我的搜索历史记录
     *
     * @return
     */
    @ApiOperation(value = "删除我的搜索历史记录", notes = "删除所有我的搜索历史记录", response = Response.class)
    @GetMapping(value = "/home/searchDeleteByMe")
    public Response<String> searchDeleteByMe() {
        try {
            dictionaryService.searchDeleteByMe();
            return Response.ok();
        } catch (Exception e) {
            LOG.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 大家都在搜索历史记录
     *
     * @return
     */
    @ApiOperation(value = "大家都在搜索历史记录", notes = "查询所有大家都在搜索历史记录", response = Response.class)
    @GetMapping(value = "/home/searchListByAll")
    public Response<List<Dictionary>> searchListByAll() {
        try {
            List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.searchListByAll();

            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }
    /**
     * 删除单个数据字典
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "删除单个数据字典（会同步循环删除该节点下所有子节点）", response = Response.class)
    @PostMapping("/delete")
    public Response<String> delete(@RequestParam("id") Long id) {
        try {
            boolean success = dictionaryService.batchDelete(new Long[]{id});
            if (success) {
            	return Response.ok();
            }else {
            	return Response.fail(ReturnCode.DELETE_FAIL);
            }
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 批量删除数据字典
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "批量删除数据字典（会同步循环删除节点下所有子节点）", response = Response.class)
    @PostMapping("/batch/delete")
    public Response<String> deleteBacth(
            @ApiParam(name = "id", value = "要删除的数据字典的id，可传入多个id") @RequestParam("id") Long[] id
    ) {
        try {
            boolean success = dictionaryService.batchDelete(id);
            if (success) {
            	return Response.ok();
            }else {
            	return Response.fail(ReturnCode.DELETE_FAIL);
            }
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 更新数据字典
     *
     * @param dictionary
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "根据id更新数据字典", response = Response.class)
    @PostMapping(value = "/update")
    public Response<String> update(@ApiParam("id不可不传") @RequestBody Dictionary dictionary) {
        try {
            com.yizhi.system.application.domain.Dictionary dictionary1 = new com.yizhi.system.application.domain.Dictionary();
            BeanUtils.copyProperties(dictionary,dictionary1);
            boolean success = dictionaryService.updateById(dictionary1);
            if (success) {
            	return Response.ok();
            }else {
            	return Response.fail(ReturnCode.SAVE_FAIL);
            }
            	
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }
    
    /**
     * 根据id查找单个字典
     *
     * @param id 字典Id
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "根据id查找单个字典", response = Response.class)
    @GetMapping(value = "/find")
    public Response<Dictionary> findById(@RequestParam("id") Long id) {
        try {
            com.yizhi.system.application.domain.Dictionary result = dictionaryService.selectById(id);
            Dictionary dictionary1 = new Dictionary();
            BeanUtils.copyProperties(result,dictionary1);
            return Response.ok(dictionary1);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }

    }

    /**
     * 根据code查询数据字典
     *
     * @param code
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "根据code查询数据字典", response = Response.class)
    @GetMapping(value = "/code/find")
    public Response<Dictionary> findByCode(@RequestParam("code") String code) {
        try {
            com.yizhi.system.application.domain.Dictionary result = dictionaryService.findByCode(code);
            Dictionary dictionary1 = new Dictionary();
            BeanUtils.copyProperties(result,dictionary1);
            return Response.ok(dictionary1);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 分页查询数据字典
     *
     * @param parentId      父字典Id  （精确匹配）
     * @param code          字典Code （精确匹配）
     * @param cnName        字典中文名（模糊匹配）
     * @param enName        字典英文名（模糊匹配）
     * @param pageNo        要跳转的页数
     * @param pageSize      每页条数
     * @param pageSortFiled 排序字段名
     * @param pageSortType  顺序：asc,倒叙：desc
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "分页查询字典", response = Response.class)
    @GetMapping(value = "/list")
    public Response<Page<Dictionary>> listByPage(
            @ApiParam(name = "parentId", value = "父字典Id, 不传则默认查询所有根节点", required = false) @RequestParam(name = "parentId", defaultValue = "0") Long parentId,
            @ApiParam(name = "code", value = "字典code", required = false) @RequestParam(name = "code", required = false) String code,
            @ApiParam(name = "cnName", value = "字典中文名（模糊匹配）", required = false) @RequestParam(name = "cnName", required = false) String cnName,
            @ApiParam(name = "enName", value = "字典英文名（模糊匹配）", required = false) @RequestParam(name = "enName", required = false) String enName,

            @ApiParam(name = "pageNo", value = "要跳转的页数") @RequestParam(name = "pageNo", defaultValue = "1") int pageNo,
            @ApiParam(name = "pageSize", value = "每页条数，默认：10", required = false) @RequestParam(name = "pageSize", defaultValue = "10") int pageSize,
            @ApiParam(name = "pageSortFiled", value = "排序字段名，默认：sort", required = false) @RequestParam(name = "pageSortFiled", defaultValue = "sort") String pageSortFiled,
            @ApiParam(name = "pageSortType", value = "顺序：asc,倒叙：desc，默认：asc", required = false) @RequestParam(name = "pageSortType", defaultValue = "asc") String pageSortType
    ) {
        try {
            Page<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.listByPage(parentId, code, cnName, enName, pageNo, pageSize, pageSortFiled, pageSortType);
            Page<Dictionary> page = new Page<>(pageNo,pageSize);
            List<com.yizhi.system.application.domain.Dictionary> records = result.getRecords();
            BeanUtils.copyProperties(result,page);
            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 查询所有顶层数据字典
     *
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "查询所有顶层数据字典", response = Response.class)
    @GetMapping(value = "/root/list")
    public Response<List<Dictionary>> listRootDictionary() {
        try {
        	List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.listRootDictionary();

            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 查询当前数据字典下的子节点
     *
     * @param id            要查找的对象的父节点
     * @param includeParent 是否包含父节点（id对应的自身节点），true：包含，false：不包含，默认false
     * @param layer         以id对应对象为第0层，查询多少层子节点，默认1层
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "查询当前数据字典下的子节点", response = Response.class)
    @GetMapping(value = "/child/list")
    public Response<List<Dictionary>> listChildDictionary(
            @ApiParam("要查找的对象的父节点") @RequestParam(name = "id", defaultValue = "0") Long id,
            @ApiParam("是否包含父节点（id对应的自身节点），</br>true：包含，</br>false：不包含，默认false") @RequestParam(name = "includeParent", defaultValue = "false") Boolean includeParent,
            @ApiParam("以id对应对象为第0层，查询多少层子节点，默认1层") @RequestParam(name = "layer", defaultValue = "1") Integer layer
    ) {
        try {
        	List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.listChildren(id, includeParent, layer);
            List<Dictionary> list = domainConverter.toDOList(result, Dictionary.class);
            return Response.ok(list);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }

    }

    /**
     * 查询当前数据字典下的子节点
     *
     * @param code            要查找的对象的父节点
     * @param includeParent 是否包含父节点（id对应的自身节点），true：包含，false：不包含，默认false
     * @param layer         以id对应对象为第0层，查询多少层子节点，默认1层
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "通过父类code查询当前数据字典下的子节点", response = Response.class)
    @GetMapping(value = "/child/listByCode")
    public Response<List<Dictionary>> listChildDictionary(
            @ApiParam("要查找的对象的父节点") @RequestParam(name = "code", defaultValue = "0") String code,
            @ApiParam("是否包含父节点（id对应的自身节点），</br>true：包含，</br>false：不包含，默认false") @RequestParam(name = "includeParent", defaultValue = "false") Boolean includeParent,
            @ApiParam("以id对应对象为第0层，查询多少层子节点，默认1层") @RequestParam(name = "layer", defaultValue = "1") Integer layer
    ) {
        try {
            com.yizhi.system.application.domain.Dictionary dictionary = dictionaryService.findByCode(code);
            if(dictionary==null){
                return Response.ok(Lists.newArrayList());
            }
            List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.listChildren(dictionary.getId(), includeParent, layer);
            List<Dictionary> list = domainConverter.toDOList(result, Dictionary.class);
            return Response.ok(list);
        } catch (Exception e) {
            LOG.error("", e);
            return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }

    }


    /**
     * 查询当前数据字典下的子节点
     *
     * @param code            要查找的对象的父节点
     * @param includeParent 是否包含父节点（id对应的自身节点），true：包含，false：不包含，默认false
     * @param layer         以id对应对象为第0层，查询多少层子节点，默认1层
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "通过父类code查询当前数据字典下的子节点", response = Response.class)
    @GetMapping(value = "/child/listByCode1")
    public List<Dictionary> listChildDictionary1(
            @ApiParam("要查找的对象的父节点") @RequestParam(name = "code", defaultValue = "0") String code,
            @ApiParam("是否包含父节点（id对应的自身节点），</br>true：包含，</br>false：不包含，默认false") @RequestParam(name = "includeParent", defaultValue = "false") Boolean includeParent,
            @ApiParam("以id对应对象为第0层，查询多少层子节点，默认1层") @RequestParam(name = "layer", defaultValue = "1") Integer layer
    ) {
        try {
            com.yizhi.system.application.domain.Dictionary dictionary = dictionaryService.findByCode(code);
            if(dictionary==null){
                return Lists.newArrayList();
            }
            List<com.yizhi.system.application.domain.Dictionary> result = dictionaryService.listChildren(dictionary.getId(), includeParent, layer);
            List<Dictionary> list = domainConverter.toDOList(result, Dictionary.class);
            return list;
        } catch (Exception e) {
            LOG.error("", e);
            return Lists.newArrayList();
        }

    }
    /**
     * 检查该字典下是否有子数据
     *
     * @param id
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "查询当前字典下是否有子数据字典 true:有；false:没有", response = Response.class)
    @GetMapping(value = "/child/check")
    public Response<Boolean> chedkHasChild(@RequestParam("id") Long id) {
        try {
        	Boolean result = dictionaryService.chedkHasChild(id);
            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }

    /**
     * 检查是否已经存在该code
     *
     * @param code
     * @return
     */
    @ApiOperation(value = "数据字典", notes = "检查code是否已经存在", response = Response.class)
    @GetMapping(value = "/code/check")
    public Response<Boolean> checkCode(@RequestParam("code") String code) {
        try {
        	Boolean result = dictionaryService.checkCode(code);
            return Response.ok(result);
        } catch (Exception e) {
            LOG.error("", e);
           return Response.fail(ReturnCode.SERVICE_UNAVAILABLE.getCode(),ReturnCode.SERVICE_UNAVAILABLE.getMsg());
        }
    }


}

