package com.yizhi.system.application.controller.remote;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.enums.MemberType;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.system.application.domain.Account;
import com.yizhi.system.application.domain.AuthzGroupMember;
import com.yizhi.system.application.domain.Company;
import com.yizhi.system.application.service.*;
import com.yizhi.system.application.vo.AccountVO;
import com.yizhi.system.application.vo.CompanySimpleVO;
import com.yizhi.system.application.vo.SiteVO;
import com.yizhi.system.application.vo.TrSitePageVO;
import com.yizhi.system.application.vo.domain.Site;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.text.DateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 站点远程调用接口
 *
 * @author yinyuyan
 */
@RestController
@RequestMapping(value = "/remote/site")
@Slf4j
public class RemoteSiteController {

    private static final Logger LOG = LoggerFactory.getLogger(RemoteSiteController.class);

    @Value("${redis.manage.authz.expired}")
    private Long redisManageAuthzExpired;

    @Autowired
    ICompanyService companyService;
    @Autowired
    private DomainConverter converter;
    @Autowired
    ISiteService iSiteService;
    @Autowired
    RedisCache redisCache;
    @Autowired
    IAccountService accountService;
    @Autowired
    private AuthService authService;

    @Autowired
    private IAuthzGroupMemberService iGroupMemberService;

    @Autowired
    IdGenerator idGenerator;

    /**
     * 根据站点code查询站点信息
     *
     * @param code 站点code
     * @param cid  企业Id
     * @return
     */
    @GetMapping(value = "/code/get")
    public SiteVO findByName(@RequestParam(value = "code") String code, @RequestParam(value = "cid") Long cid) {
        com.yizhi.system.application.domain.Company company = companyService.selectById(cid);
        if (company != null) {
            com.yizhi.system.application.domain.Site site = new com.yizhi.system.application.domain.Site();
            site.setCompanyId(cid);
            site.setCode(code);
            site.setEnabled(Boolean.valueOf(true));
            com.yizhi.system.application.domain.Site resultSite = iSiteService.selectOne(QueryUtil.condition(site));
            if (null == resultSite) {
                return null;
            }
            SiteVO siteVO = new SiteVO();
            BeanUtils.copyProperties(resultSite, siteVO);
            redisCache.hset(CacheNamespace.REDIS_SITES,
                    company.getCode() + CacheNamespace.REDIS_HASH_SEPARATOR + code,
                    JSON.toJSONString(siteVO), redisManageAuthzExpired);
            return siteVO;
        }
        return null;
    }

    @GetMapping(value = "/list/companyid")
    public List<SiteVO> findByCompanyId(@RequestParam(value = "companyId", required = false) Long companyId) {
        return converter.toDOList(iSiteService.findByCompanyId(companyId), SiteVO.class);
    }

    @GetMapping(value = "/company/list")
    public List<CompanySimpleVO> list() {
        com.yizhi.system.application.domain.Company company = new com.yizhi.system.application.domain.Company();
        company.setStatus(2);
        return converter.toDOList(companyService.selectList(QueryUtil.condition(company)), CompanySimpleVO.class);
    }

    /**
     * 根据站点id和企业id，查询站点下所有的用户信息
     *
     * @param siteId    站点id
     * @param companyId 企业Id
     * @return
     */
    @GetMapping(value = "/account/list")
    public List<AccountVO> getSitAccount(@RequestParam(value = "siteId") Long siteId,
                                         @RequestParam(value = "companyId") Long companyId) {
        com.yizhi.system.application.domain.Site site = iSiteService.selectById(siteId);
        if (site != null) {
            return converter.toDOList(accountService.selectBySiteId(siteId), AccountVO.class);
        }
        return null;
    }

    /**
     * 根据站点id和企业id，查询站点下所有的用户信息总条数
     *
     * @param siteId    站点id
     * @param companyId 企业Id
     * @return
     */
    @GetMapping(value = "/account/count")
    public int getSitAccountCount(@RequestParam(value = "siteId") Long siteId,
                                  @RequestParam(value = "companyId") Long companyId) {
        int ret = 0;
        List<AccountVO> voList = getSitAccount(siteId, companyId);
        if (!CollectionUtils.isEmpty(voList)) {
            ret = voList.size();
        }
        return ret;
    }


    /**
     * 根据站点id查询站点信息和站点的企业名
     * @param ids
     * @return
     */
    @PostMapping(value = "/siteIds/company/get")
    public List<SiteVO> selectByIds(@RequestBody List<Long> ids) {
        List<SiteVO> stieVos = new ArrayList<>();
        List<com.yizhi.system.application.domain.Site> sties = iSiteService.selectBatchIds(ids);
        if (!CollectionUtils.isEmpty(sties)) {
            List<Long> companyIds = sties.stream().map(S -> S.getCompanyId()).collect(Collectors.toList());
            List<com.yizhi.system.application.domain.Company> companys = companyService.selectBatchIds(companyIds);
            Map<Long, com.yizhi.system.application.domain.Company> companyMaps = companys.stream().collect(Collectors.toMap(Company::getId, Function.identity()));
            BeanCopier copier = BeanCopier.create(com.yizhi.system.application.domain.Site.class, SiteVO.class, false);
            for (com.yizhi.system.application.domain.Site obj : sties) {
                SiteVO siteVO = new SiteVO();
                copier.copy(obj, siteVO, null);
                Long companyId = companyMaps.get(obj.getCompanyId()).getId();
                String CompanyName = companyMaps.get(obj.getCompanyId()).getName();
                String companyCode = companyMaps.get(obj.getCompanyId()).getCode();
                siteVO.setCompanyName(CompanyName);
                siteVO.setCompanyId(companyId);
                siteVO.setCompanyCode(companyCode);
                stieVos.add(siteVO);
            }
        }
        return stieVos;
    }

    @GetMapping(value = "/id/get")
    public SiteVO selectById(@RequestParam("id") Long id) {
        com.yizhi.system.application.domain.Site site = iSiteService.selectById(id);
        if (site == null) {
            return null;
        }
        SiteVO siteVO = new SiteVO();
        BeanUtils.copyProperties(site, siteVO);
        return siteVO;
    }

    @GetMapping(value = "/isAccessToSite")
    private boolean isAccessToSite(@RequestParam(value = "accountId") long accountId,
                                   @RequestParam(value = "siteId") long siteId,
                                   @RequestParam(value = "companyId") long companyId) {
        boolean siteAccount = false;
        Company company = companyService.get(companyId);
        LOG.info("当前的code：{}", ContextHolder.get().getCompanyCode());
        Set<Long> siteIds = authService.AccessSiteIdsFront(companyId, company.getCode(), accountId);
        LOG.info("当前的code：{}", JSON.toJSON(siteIds));
        if (!org.apache.commons.collections.CollectionUtils.isEmpty(siteIds)) {
            siteAccount = siteIds.contains(siteId);
        }
        return siteAccount;
    }

    @GetMapping(value = "/accessToSite")
    public boolean accessToSite(@RequestParam(value = "accountId") long accountId,
                                @RequestParam(value = "siteId") long siteId,
                                @RequestParam(value = "companyId") long companyId,
                                @RequestParam(value = "companyCode") String companyCode) {
        boolean siteAccount = false;
        Set<Long> siteIds = authService.AccessSiteIdsFront(companyId, companyCode, accountId);
        LOG.info("当前的code：{}", JSON.toJSON(siteIds));
        if (!org.apache.commons.collections.CollectionUtils.isEmpty(siteIds)) {
            siteAccount = siteIds.contains(siteId);
        }
        return siteAccount;
    }

    /**
     * 自动生成账号的接口
     *
     * @param siteId    站点id
     * @param accountId 用户的Id
     * @return
     */
    @GetMapping(value = "/relateAccountSite")
    public boolean relateAccountSite(@RequestParam(name = "siteId") Long siteId,
                                     @RequestParam(name = "accountId") Long accountId,
                                     @RequestParam(name = "companyCode") String companyCode) {

        LOG.info("参数site是：" + siteId + ";参数accountId" + accountId);
        com.yizhi.system.application.domain.Site site = iSiteService.selectById(siteId);

        com.yizhi.system.application.domain.Account account = new Account();
        account.setId(accountId);
        account = accountService.selectById(accountId);

        boolean ret = false;
        if (null != account) {
            com.yizhi.system.application.domain.AuthzGroupMember groupMember = new com.yizhi.system.application.domain.AuthzGroupMember();
            groupMember.setGroupId(site.getUserGroupId());
            groupMember.setType(MemberType.USER.getCode());
            groupMember.setMemberId(account.getId());
            groupMember.setDeleted(0);

            List<AuthzGroupMember> groupMembers = groupMember.selectList(new EntityWrapper(groupMember));
            if (!CollectionUtils.isEmpty(groupMembers)) {
                return true;
            } else {
                groupMember.setId(idGenerator.generate());
                ret = iGroupMemberService.insert(groupMember);
            }
        }
        Long[] arr = new Long[1];
        arr[0] = accountId;
        redisCache.hdel(CacheNamespace.REDIS_AUTHZ_FRONT_COMPANY_ACCOUNT_ACCESS_SITE + companyCode, arr);
        return ret;
    }

    /**
     * 是否是主站点
     *
     * @param siteId
     * @return true:是主站点，false:是子站点
     */
    @GetMapping(value = "/isDefaultSite")
    public Boolean isDefaultSite(@RequestParam(name = "siteId") Long siteId) {
        com.yizhi.system.application.domain.Site site = iSiteService.selectById(siteId);
        if (site == null) {
            return null;
        }
        return site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT);
    }

    @GetMapping("/getAllSiteId")
    public List<Long> getAllSiteId() {
        return iSiteService.getAllSiteId();
    }

    @GetMapping("/getSiteVOs")
    public List<SiteVO> getSiteVOs(@RequestParam(value = "companyId") long companyId) {
        return converter.toDOList(iSiteService.getSites(companyId), SiteVO.class);
    }

    @GetMapping("/getSiteIds")
    public List<Long> getSiteIds(@RequestParam(value = "companyId") long companyId) {
        return iSiteService.getSiteIds(companyId);
    }

    /**
     * 根据站点code和企业的id 获取站点详情
     *
     * @param code
     * @param cid
     * @return
     */
    @GetMapping(value = "/detail")
    public SiteVO findSiteByCode(@RequestParam(value = "code") String code, @RequestParam(value = "cid") Long cid) {
        com.yizhi.system.application.domain.Site site = new com.yizhi.system.application.domain.Site();
        site.setCompanyId(cid);
        site.setCode(code);
        site.setEnabled(Boolean.valueOf(true));
        com.yizhi.system.application.domain.Site tempSite = iSiteService.selectOne(QueryUtil.condition(site));
        SiteVO siteVO = new SiteVO();
        if (null != tempSite) {
            BeanUtils.copyProperties(tempSite, siteVO);
        }
        return siteVO;
    }

    /**
     * 分页查询站点列表
     *
     * @param siteVO
     * @return
     */
    @ApiOperation(value = "分页查询站点列表", notes = "分页查询站点列表")
    @PostMapping(value = "/getPage")
    public Page<com.yizhi.system.application.domain.Site> getPage(@RequestBody TrSitePageVO siteVO) {

        Site s = (Site) converter.toDO(siteVO, Site.class);
        com.yizhi.system.application.domain.Site site = new com.yizhi.system.application.domain.Site();
        BeanUtils.copyProperties(s,site);
        site.setName(siteVO.getKeyword());
        Page<com.yizhi.system.application.domain.Site> page = new Page<>(siteVO.getPageNo(), siteVO.getPageSize());
        page = iSiteService.getPage(site, page);
        return page;
    }

    @ApiOperation(value = "学员可访问站点列表", response = SiteVO.class)
    @GetMapping(value = "/front/list")
    public Set<Long> frontList() {
        RequestContext context = ContextHolder.get();
        return authService.AccessSiteIdsFront(context.getCompanyId(), context.getCompanyCode(), context.getAccountId());
    }

    /**
     * 社群site同步
     * @param date
     * @return
     */
    @PostMapping(value="/public/community/syncSite")
    public List<com.yizhi.system.application.domain.Site> syncGroupMember(@RequestParam("indexTime") long indexTime){
        log.info("===================社群site同步==========================");
        EntityWrapper<com.yizhi.system.application.domain.Site> wrapper = new EntityWrapper<>();
        wrapper.gt("index_time", DateUtil.formatDateTime(new Date(indexTime)));
        List<com.yizhi.system.application.domain.Site> sites = iSiteService.selectList(wrapper);
        if (CollectionUtils.isEmpty(sites)){
            return null;
        }
        return sites;
    }
}
