package com.yizhi.system.application.service.impl;

import com.baomidou.mybatisplus.enums.SqlLike;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.mapper.Wrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.system.application.domain.*;
import com.yizhi.system.application.enums.OrderField;
import com.yizhi.system.application.mapper.*;
import com.yizhi.system.application.model.BaseModel;
import com.yizhi.system.application.request.SiteUnselectedResq;
import com.yizhi.system.application.service.AuthorityService;
import com.yizhi.system.application.service.ISiteService;
import com.yizhi.system.application.vo.CompanySiteVO;
import com.yizhi.system.application.vo.SiteAdminVo;
import com.yizhi.system.application.vo.SiteProtectDefinedVO;
import com.yizhi.system.application.vo.SiteVO;
import com.yizhi.util.application.constant.ReturnCode;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 站点 服务实现类
 * </p>
 *
 * @author fulan123
 * @since 2018-03-08
 */
@Service
@Transactional
public class SiteServiceImpl extends ServiceImpl<SiteMapper, Site> implements ISiteService {

    private static final Logger LOGGER = LoggerFactory.getLogger(SiteServiceImpl.class);

    @Autowired
    SiteMapper siteMapper;
    @Autowired
    OrganizationMapper orgMapper;
    @Autowired
    AccountMapper accountMapper;
    @Autowired
    IdGenerator idGenerator;
    @Autowired
    RedisCache redisCache;
    @Autowired
    AuthorityService authorityService;
    @Autowired
    private AuthzAccountRoleMapper accountRoleMapper;
    @Autowired
    private AuthzRoleUserGroupMapper roleGroupMapper;
    @Autowired
    private AuthzRoleMapper roleMapper;
    @Autowired
    CompanyMapper companyMapper;
    @Autowired
    DomainConverter domainConverter;

    //能随机一个有8个字符的字符串，其中4个大写字母  4个数字
    public static String getRandom() {
        String str = "";
        for (int i = 0; i < 8; i++) {
            char c = ' ';
            if (i % 2 == 0) {
                c = (char) (Math.random() * 10 + 48);//随机出0-9个数字
            } else if (i % 2 == 1) {
                c = (char) (Math.random() * 26 + 65);//随机出大写字母
            } else {
                c = (char) (Math.random() * 26 + 97);//随机出小写字母
            }
            str = c + str;
        }
        return str;
    }

    /**
     * 获取站点的唯一code
     *
     * @return
     */
    @Override
    public String getLoginCode() {
        //先判断有无该key,没有则初始化
        if (!redisCache.hasKey(CacheNamespace.SITE_EXIST_LOGIN_CODE)) {
            EntityWrapper<com.yizhi.system.application.domain.Site> wrapper = new EntityWrapper();
            wrapper.eq("has_deal", true);
            List<com.yizhi.system.application.domain.Site> sites = siteMapper.selectList(wrapper);
            if (CollectionUtils.isNotEmpty(sites)) {
                sites.parallelStream().forEach(a -> {
                    //保存缓存
                    redisCache.hset(CacheNamespace.SITE_EXIST_LOGIN_CODE, a.getLoginCode(), a.getId().toString());
                });
            }else {
                LOGGER.info("没有一个站点有loginCode,开始初始化");
                initSiteLoginCode();
            }
        }
        return getCode();
    }

    private String getCode() {
        String finalLoginCode = getRandom();
        if (redisCache.hExisted(CacheNamespace.SITE_EXIST_LOGIN_CODE, finalLoginCode)) {
            finalLoginCode = getCode();
        }
        return finalLoginCode;
    }
    @Override
    public Boolean save(BaseModel<SiteVO> model) {
        RequestContext context = model.getContext();
        SiteVO siteVO = model.getObj();
        // 检查重复
        if (siteMapper.countExistCodeOrg(0L, context.getCompanyId(), siteVO.getCode()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "站点编码已存在");
        }
        if (siteMapper.countExistNameOrg(0L, context.getCompanyId(), siteVO.getName()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "站点名称已存在");
        }
        // 插入站点
        Site site = new Site();
        site.setId(idGenerator.generate());
        site.setCreateById(context.getAccountId());
        site.setCreateByName(context.getAccountName());
        site.setCreateTime(model.getDate());
        site.setCompanyId(context.getCompanyId());
        site.setType(AuthzConstant.SITE_TYPE_SUB);
        site.setEnabled(Boolean.TRUE);
        site.setRecords(siteVO.getRecords());

        site.setName(siteVO.getName());
        site.setCode(siteVO.getCode());
        site.setLogo(siteVO.getLogo());
        site.setUserGroupId(siteVO.getUserGroupId());
        StringBuilder domain = new StringBuilder();
        //查询公司信息
        Company company = companyMapper.selectById(context.getCompanyId());
        if (null != company) {
            domain.append(company.getDomain());
        }
        site.setDomain(company.getCode() + "/" + siteVO.getCode());
        //获取唯一code
        site.setLoginCode(getLoginCode());
        if (site.insert()) {
            //保存缓存
            redisCache.hset(CacheNamespace.SITE_EXIST_LOGIN_CODE, site.getLoginCode(),site.getId().toString());
            site.setHasDeal(true);
            baseMapper.updateById(site);
            LOGGER.info("保存站点成功，生成的loginCode:" + site.getLoginCode() + "===redisKey:" + CacheNamespace.SITE_EXIST_LOGIN_CODE);
            // 插入默认站点管理员角色
            AuthzRole role = new AuthzRole();
            role.setId(idGenerator.generate());
            role.setName(site.getName() + "-站点管理员");
            role.setCode(site.getCode() + AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX);
            role.setDescription(site.getName() + "默认站点管理员");
            role.setCompanyId(context.getCompanyId());
            role.setCreateById(context.getAccountId());
            role.setCreateByName(context.getAccountName());
            role.setCreateTime(model.getDate());
            role.setEnabled(1);
            role.setSiteId(site.getId());
            if (role.insert()) {
                // 插入默认站点管理员角色与管辖区的关联关系，默认站点角色的管辖范围就是该站点的管辖区id（单条关联关系数据）
                AuthzRoleUserGroup roleGroup = new AuthzRoleUserGroup();
                roleGroup.setId(idGenerator.generate());
                roleGroup.setRoleId(role.getId());
                roleGroup.setUserGroupId(site.getUserGroupId());
                roleGroup.setCreateById(context.getAccountId());
                roleGroup.setCreateByName(context.getAccountName());
                roleGroup.setCreateTime(model.getDate());
                if (roleGroup.insert()) {
                    clearAuthCache(company.getCode());
                } else {
                    throw new BizException(ReturnCode.SAVE_FAIL.getCode(), "插入角色-管辖区关联关系失败");
                }

            } else {
                throw new BizException(ReturnCode.SAVE_FAIL.getCode(), "插入站点默认角色失败");
            }
        } else {
            throw new BizException(ReturnCode.SAVE_FAIL.getCode(), "插入站点失败");
        }
        return true;
    }

    @Override
    public Boolean update(BaseModel<SiteVO> model) {
        SiteVO siteVO = model.getObj();
        Site site = new Site();

        if (StringUtils.isNotEmpty(siteVO.getName())) {
            if (siteMapper.countExistNameOrg(siteVO.getId(), model.getContext().getCompanyId(), siteVO.getName()) > 0) {
                throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "站点名称已存在");
            }
        }
        // code不能修改
        site.setId(siteVO.getId());
        site.setName(siteVO.getName());
        site.setCode(null);
        site.setDomain(siteVO.getDomain());
        site.setLogo(siteVO.getLogo());
        site.setUserGroupId(siteVO.getUserGroupId());

        site.setUpdateById(model.getContext().getAccountId());
        site.setUpdateByName(model.getContext().getAccountName());
        site.setUpdateTime(model.getDate());

        if (site.updateById()) {
            clearAuthCache(model.getContext().getCompanyCode());
            return true;
        } else {
            throw new BizException(ReturnCode.UPDATE_FAIL.getCode(), ReturnCode.UPDATE_FAIL.getMsg());
        }
    }

    @Override
    public Boolean down(Long id, String companyCode) {
        Site site = new Site();
        site.setId(id);
        site.setEnabled(false);
        if (this.updateById(site)) {
            clearAuthCache(companyCode);
            return true;
        } else {
            throw new BizException(ReturnCode.DELETE_FAIL.getCode(), ReturnCode.DELETE_FAIL.getMsg());
        }
    }

    @Override
    public Boolean up(Long id, String companyCode) {
        Site site = new Site();
        site.setId(id);
        site.setEnabled(true);
        if (this.updateById(site)) {
            clearAuthCache(companyCode);
            return true;
        } else {
            throw new BizException(ReturnCode.DELETE_FAIL.getCode(), ReturnCode.DELETE_FAIL.getMsg());
        }
    }

    @Override
    public Site findById(Long id) {
        Site site = this.selectById(id);
        if (null == site) {
            throw new BizException(ReturnCode.RESOURCE_NOT_FOUND.getCode(), ReturnCode.RESOURCE_NOT_FOUND.getMsg());
        }
        return site;
    }

    @Override
    public List<Site> findByCompanyId(Long companyId) {
        Site site = new Site();
        if(companyId != null){
            site.setCompanyId(companyId);
        }

        site.setEnabled(true);
        List<Site> sites = this.selectList(QueryUtil.condition(site));
        if (CollectionUtils.isEmpty(sites)) {
            throw new BizException(ReturnCode.RESOURCE_NOT_FOUND.getCode(), ReturnCode.RESOURCE_NOT_FOUND.getMsg());
        }
        return sites;
    }

    @Override
    public Boolean checkCode(Long id, Long companyId, String code) {
        if (siteMapper.countExistCodeOrg(0L, companyId, code) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "站点编码已存在");
        } else {
            return true;
        }
    }

    @Override
    public Boolean checkName(Long id, Long companyId, String name) {
        if (siteMapper.countExistNameOrg(0L, companyId, name) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "站点名称已存在");
        } else {
            return true;
        }
    }

    @Override
    public Page<Site> list(Site site, Page<Site> page,Long accessSiteId) {
        page.setOrderByField(OrderField.CREATETIME.getCode());
        page.setAsc(false);

        Wrapper<Site> wrapper = QueryUtil.condition(new Site());
        wrapper.ne("id",accessSiteId);
        wrapper.where("company_id={0}", site.getCompanyId());
        if (site.getEnabled() != null) {
            if (site.getEnabled()) {
                wrapper.and("enabled={0}", 1);
            } else {
                wrapper.and("enabled={0}", 0);
            }
        }
        String name = site.getName();
        if (StringUtils.isNotBlank(name)) {
            wrapper.andNew().like("name", name, SqlLike.DEFAULT).or().like("code", name, SqlLike.DEFAULT);
        }
        return this.selectPage(page, wrapper);
    }

    //	权限改造##########################################################################################################

    @Override
    public Integer setSiteAdmin(BaseModel<SiteAdminVo> model) {
        SiteAdminVo siteAdminVo = model.getObj();
        RequestContext context = model.getContext();
        Long siteId = model.getObj().getSiteId();
        if (null != siteId) {
            Set<Long> accountIdsForCache = new HashSet<>();
            // 先查出站点管理员角色
            Site site = siteMapper.selectById(siteAdminVo.getSiteId());
            if (null != site) {
                String siteAdminCode = site.getCode() + AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX;
                AuthzRole role = new AuthzRole();
                role.setSiteId(site.getId());
                role.setCode(siteAdminCode);
                role.setEnabled(1);
                role = roleMapper.selectOne(role);
                if (role != null) {
                    int num = -1;
                    //先删除已存在的站点管理员
                    AuthzAccountRole conditon = new AuthzAccountRole();
                    conditon.setRoleId(role.getId());
                    List<AuthzAccountRole> accountRoles = conditon.selectList(new EntityWrapper(conditon));
                    if (!CollectionUtils.isEmpty(accountRoles)) {
                        Set<Long> idsToDel = new HashSet<>(accountRoles.size());
                        for (AuthzAccountRole ar : accountRoles) {
                            idsToDel.add(ar.getId());
                            accountIdsForCache.add(ar.getAccountId());
                        }
                        accountRoleMapper.deleteBatchIds(idsToDel);
                    }
                    //新增站点管理员
                    if (!CollectionUtils.isEmpty(model.getObj().getAccountIds())) {
                        List<AuthzAccountRole> list = new ArrayList<>();
                        for (Long accountId : siteAdminVo.getAccountIds()) {
                            AuthzAccountRole accountRole = new AuthzAccountRole();
                            accountRole.setId(idGenerator.generate());
                            accountRole.setRoleId(role.getId());
                            accountRole.setAccountId(accountId);
                            accountRole.setSiteId(site.getId());
                            accountRole.setCreateById(model.getContext().getAccountId());
                            accountRole.setCreateByName(model.getContext().getAccountName());
                            accountRole.setCreateTime(model.getDate());
                            list.add(accountRole);
                            // 如果以前有，现在也有，则不用重置权限cache
                            if (accountIdsForCache.contains(accountId)) {
                                accountIdsForCache.remove(accountId);
                            } else {
                                accountIdsForCache.add(accountId);
                            }
                        }
                        num = accountRoleMapper.batchInsert(list);
                    }

                    // 清除权限缓存
//                    if (!CollectionUtils.isEmpty(accountIdsForCache)) {
//                        String[] itemKeys = new String[accountIdsForCache.size()];
//                        int index = 0;
//                        for (Long id : accountIdsForCache) {
//                            itemKeys[index] = site.getCode() + CacheNamespace.REDIS_HASH_SEPARATOR + id;
//                            index++;
//                        }
//                        redisCache.hdel(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + context.getCompanyCode(), itemKeys);
//                        redisCache.hdel(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ACCOUNT_ACCESS_SITE + context.getCompanyCode(), itemKeys);
//                    }
                    clearAuthCache(context.getCompanyCode());
                    return num;
                }
            }
        }
        return null;
    }

    @Override
    public List<Account> getSiteAdmin(Long siteId) {
        RequestContext context = ContextHolder.get();
        List<Account> accounts = new ArrayList<>();
        // 先查出站点管理员角色
        if (siteId == null) {
            siteId = context.getSiteId();
        }
        Site site = siteMapper.selectById(siteId);
        if (null != site) {
            String siteAdminCode = site.getCode() + AuthzConstant.DEFAULT_SITE_ADMIN_ROLE_CODE_SUFFIX;
            AuthzRole role = new AuthzRole();
            role.setCompanyId(context.getCompanyId());
            role.setSiteId(site.getId());
            role.setCode(siteAdminCode);
            role.setEnabled(1);
            role = roleMapper.selectOne(role);
            if (role != null) {
                //查询账号列表
                List<Long> accountIds = accountRoleMapper.getAccountIdsByRoleId(role.getId());
                if (!CollectionUtils.isEmpty(accountIds)) {
                    accounts = accountMapper.selectBatchIds(accountIds);
                }
            }
        }
        return accounts;
    }

    @Override
    public void clearSiteOrgIdsCache(String siteCode, String companyCode) {
        try {
            // 删除站点下部门信息
            redisCache.hdel(CacheNamespace.REIDS_COMPANY_SITE_ORGS + companyCode, new String[]{siteCode});
            redisCache.hdel(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + companyCode, new String[]{siteCode});
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_FRONT_COMPANY_ACCOUNT_ACCESS_SITE + companyCode);
        } catch (Exception e) {
            LOGGER.error("清除站点下部门信息redis错误：try-catch处理，不影响返回", e);
        }
    }

    @Override
    public List<Site> courseAuthOtherSite(RequestContext context) {
        // 查出当前站点
        long siteId = context.getSiteId();
        Site site = siteMapper.selectById(siteId);
        if (site != null) {
            Site siteEx = new Site();
            // 如果是主站点，可以授权当下子站点
            if (site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT)) {
                siteEx.setCompanyId(context.getCompanyId());
                siteEx.setType(AuthzConstant.SITE_TYPE_SUB);
            }
            // 只有子站点了，可以授权给主站点
            else if (site.getType().equals(AuthzConstant.SITE_TYPE_SUB)) {
                siteEx.setCompanyId(context.getCompanyId());
                siteEx.setType(AuthzConstant.SITE_TYPE_DEFAULT);
            }
            // 如果是运营站点，可以授权所有站点

            siteEx.setEnabled(Boolean.valueOf(true));
            EntityWrapper<Site> ew = new EntityWrapper<>(siteEx);
            // 自己不用授权给自己
            ew.notIn("id", context.getSiteId());
            List<Site> sites = siteEx.selectList(ew);
//            // 如果是主站点，可以授权当下子站点 和 运营站点
//            if (site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT) && !site.getOperationSite()) {
//                Site operationSite = new Site();
//                operationSite.setOperationSite(true);
//                operationSite.setEnabled(Boolean.TRUE);
//                List<Site> operationSites = operationSite.selectList(new EntityWrapper(operationSite));
//                if (!CollectionUtils.isEmpty(operationSites)) {
//                    if (!CollectionUtils.isEmpty(sites)) {
//                        sites.addAll(operationSites);
//                    } else {
//                        sites = new ArrayList<>();
//                        sites.addAll(operationSites);
//                    }
//                }
//
//            }
            return sites;
        }
        return null;
    }

    @Override
    public List<Long> getAllSiteId() {
        return siteMapper.getAllSiteId();
    }

    @Override
    public List<Site> getAllSiteByCompanyCode(String companyCode) {
        Company company = new Company();
        company.setCode(companyCode);
        company = company.selectOne(new EntityWrapper(company));

        if (company == null) {
            return null;
        }
        Site site = new Site();
        site.setCompanyId(company.getId());
        site.setEnabled(true);
        EntityWrapper<Site> ew = new EntityWrapper<>(site);
        ew.orderBy("type");
        List<Site> sites = siteMapper.selectList(ew);
        return sites;
    }

    @Override
    public List<CompanySiteVO> getSiteList() {
        EntityWrapper<Site> condition = QueryUtil.condition(new Site());
        List<Site> sites = this.selectList(condition);
        List<com.yizhi.system.application.vo.domain.Site> siteVOs = domainConverter.toDOList(sites, com.yizhi.system.application.vo.domain.Site.class);
        Map<Long, List<com.yizhi.system.application.vo.domain.Site>> longListMap = siteVOs.stream().collect(Collectors.groupingBy(com.yizhi.system.application.vo.domain.Site::getCompanyId));
        List<CompanySiteVO> companySiteVOList = new ArrayList<>(longListMap.size());
        for (Long key : longListMap.keySet()) {
            Company company = companyMapper.selectById(key);
            if (company!=null) {
                CompanySiteVO companySiteVO = new CompanySiteVO();

                companySiteVO.setCompanyId(company.getId());
                companySiteVO.setCompanyCode(company.getCode());
                companySiteVO.setName(company.getName());
                companySiteVO.setSiteList(longListMap.get(key));
                companySiteVOList.add(companySiteVO);
            }
        }
        return companySiteVOList;
    }

    /**
     * 清除权限
     *
     * @param companyCode
     */
    public void clearAuthCache(String companyCode) {
        // 清除管理权限
        try {
            redisCache.delete(CacheNamespace.REDIS_SITES);
            redisCache.delete(CacheNamespace.REIDS_COMPANY_SITE_ORGS + companyCode);
            redisCache.delete(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ORGIDS + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ACCOUNT_ACCESS_SITE + companyCode);
            redisCache.delete(CacheNamespace.REDIS_AUTHZ_FRONT_COMPANY_ACCOUNT_ACCESS_SITE + companyCode);
        } catch (Exception e) {
            LOGGER.error("清除站点下部门信息redis错误：try-catch处理，不影响返回", e);
        }
    }

	@Override
	public List<Site> getSites(Long companyId) {
        Site site = new Site();
        if(companyId != null){
            site.setCompanyId(companyId);
        }
        site.setEnabled(true);
        List<Site> sites = this.selectList(QueryUtil.condition(site));
        return sites;
	}

	@Override
	public List<Long> getSiteIds(Long companyId) {
		// TODO Auto-generated method stub
		return siteMapper.getSiteIds(companyId);
	}

    @Override
    public Page<SiteProtectDefinedVO> getUnselectedSiteList(SiteUnselectedResq siteReq) {

        String siteName = siteReq.getSiteName();
        String companyName = siteReq.getCompanyName();
        if (!org.springframework.util.StringUtils.isEmpty(siteName)){
            siteReq.setSiteName(siteName.trim());
        }
        if (!org.springframework.util.StringUtils.isEmpty(companyName)){
            siteReq.setCompanyName(companyName.trim());
        }
        Page<SiteProtectDefinedVO>  page = new Page<>(siteReq.getPageNo(),siteReq.getPageSize());
        List<SiteProtectDefinedVO> list = siteMapper.getSiteListByConditions(page, siteReq);
        page.setRecords(list);
        return page;
    }

    @Override
    public Page<Site> getPage(Site site, Page<Site> page) {
        Site params = new Site();
        params.setEnabled(site.getEnabled());
        params.setCompanyId(ContextHolder.get().getCompanyId());
        EntityWrapper wrapper = new EntityWrapper(params);

        String name = site.getName();
        if (StringUtils.isNotBlank(name)) {
            wrapper.andNew().like("name", name).or().like("code", name);
        }
        page.setOrderByField(OrderField.CREATETIME.getCode());
        page.setAsc(false);
        return this.selectPage(page, wrapper);
    }

    @Override
    public void initSiteLoginCode() {
        EntityWrapper<Site> wrapper = new EntityWrapper();
        wrapper.eq("has_deal",false);
        List<Site> lists = this.baseMapper.selectList(wrapper);
        if (CollectionUtils.isNotEmpty(lists)) {
            lists.forEach(a -> {
                Site params = new Site();
                params.setLoginCode(getCode());
                params.setHasDeal(true);
                params.setId(a.getId());
                baseMapper.updateById(params);
                //保存缓存
                redisCache.hset(CacheNamespace.SITE_EXIST_LOGIN_CODE, params.getLoginCode(), a.getId().toString());
            });
        }
    }
}
