package com.yizhi.util.application.qrcode;

import java.awt.BasicStroke;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Shape;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.OutputStream;
import java.util.Hashtable;

import javax.imageio.ImageIO;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.BinaryBitmap;
import com.google.zxing.DecodeHintType;
import com.google.zxing.EncodeHintType;
import com.google.zxing.MultiFormatReader;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.Result;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import com.google.zxing.qrcode.decoder.ErrorCorrectionLevel;  

/**
 * 
* 
* @ClassName: QRCodeUtil.java
* @Description: 根据输入参数生成二维码工具类
*
* @author: xingliang
* @date: 2018年1月26日 上午10:12:53 
*
* 2018年1月26日     xingliang
 */
public class QRCodeUtil {
	
	/**
	 * logger参数
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(QRCodeUtil.class);
    
	/**
	 * 编码格式
	 */
	private static final String CHARSET = "utf-8";  
    /**
     * 导出二维码图片类型
     */
    private static final String FORMAT_NAME = "JPG";  
    /**
     * 二维码尺寸 
     */
    private static final int QRCODE_SIZE = 300;  
    /**
     * 宽度
     */
    private static final int WIDTH = 100;  
    /**
     * 高度
     */
    private static final int HEIGHT = 100;  
  
    /**
    * 
    * @Function: QRCodeUtil.java
    * @Description: 生成二维码图片
    * @param content 二维码内容
    * @param imgPath 图片路径
    * @param needCompress 是否压缩
    * @return
    * @throws Exception
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:13:13 
     */
    private static BufferedImage createImage(String content, String imgPath,  
            boolean needCompress) throws Exception {  
        Hashtable hints = new Hashtable();  
        hints.put(EncodeHintType.ERROR_CORRECTION, ErrorCorrectionLevel.H);  
        hints.put(EncodeHintType.CHARACTER_SET, CHARSET);  
        hints.put(EncodeHintType.MARGIN, 1);  
        BitMatrix bitMatrix = new MultiFormatWriter().encode(content,  
                BarcodeFormat.QR_CODE, QRCODE_SIZE, QRCODE_SIZE, hints);
        //绘制图片 宽度 和 高度
        int width = bitMatrix.getWidth();  
        int height = bitMatrix.getHeight();  
        BufferedImage image = new BufferedImage(width, height,  
                BufferedImage.TYPE_INT_RGB);  
        for (int x = 0; x < width; x++) {  
            for (int y = 0; y < height; y++) {  
                image.setRGB(x, y, bitMatrix.get(x, y) ? 0xFF000000  
                        : 0xFFFFFFFF);  
            }  
        }  
        if (imgPath == null || "".equals(imgPath)) {  
            return image;  
        }  
        // 绘制二维码图片 
        QRCodeUtil.insertImage(image, imgPath, needCompress);  
        return image;  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 绘制二维码图片
    *
    * @param source 图片流
    * @param imgPath 图片地址
    * @param needCompress 是否压缩
    * @return：无返回值
    * @throws：异常描述
    *
    * @author: xinglinag
    * @date: 2018年1月26日 上午10:01:15 
    */
    private static void insertImage(BufferedImage source, String imgPath,  
            boolean needCompress) throws Exception { 
    	// 图片文件
        File file = new File(imgPath);  
        if (!file.exists()) {  
        	LOGGER.info(""+imgPath+"   该文件不存在！");  
            return;  
        }  
        Image src = ImageIO.read(new File(imgPath));  
        int width = src.getWidth(null);  
        int height = src.getHeight(null);  
        if (needCompress) { // 压缩二维码  
            if (width > WIDTH) {  
                width = WIDTH;  
            }  
            if (height > HEIGHT) {  
                height = HEIGHT;  
            }  
            Image image = src.getScaledInstance(width, height,  
                    Image.SCALE_SMOOTH);  
            BufferedImage tag = new BufferedImage(width, height,  
                    BufferedImage.TYPE_INT_RGB);  
            Graphics g = tag.getGraphics();  
            g.drawImage(image, 0, 0, null); // 绘制缩小后的图  
            g.dispose();  
            src = image;  
        }  
        // 绘制二维码 
        Graphics2D graph = source.createGraphics();  
        int x = (QRCODE_SIZE - width) / 2;  
        int y = (QRCODE_SIZE - height) / 2;  
        graph.drawImage(src, x, y, width, height, null);  
        Shape shape = new RoundRectangle2D.Float(x, y, width, width, 6, 6);  
        graph.setStroke(new BasicStroke(3f));  
        graph.draw(shape);  
        graph.dispose();  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容、二维码生成路径、二维码存放路径、是否压缩图片、二维码名称 生成二维码图片
 	* @param content 二维码内容
 	* @param imgPath 图片路径
 	* @param destPath 二维码下载路径
 	* @param needCompress 是否压缩
 	* @param filname 二维码名称
 	* @throws Exception
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:15:53 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, String imgPath, String destPath,  
            boolean needCompress,String filname) throws Exception {  
        BufferedImage image = QRCodeUtil.createImage(content, imgPath,  
                needCompress);  
        mkdirs(destPath);  
        String file = filname+".jpg";  
        ImageIO.write(image, FORMAT_NAME, new File(destPath+"/"+file));  
    }  
  
     
    /**
     * 校验二维码存放路径是否存在
     * @param destPath
     * 
     * @author: xingliang
     * @date: 2018年1月26日 上午10:15:53 
     */
    public static void mkdirs(String destPath) {  
        File file =new File(destPath);     
        //当文件夹不存在时，mkdirs会自动创建多层目录 
        if (!file.exists() && !file.isDirectory()) {  
            file.mkdirs();  
        }  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容、二维码生成路径、二维码存放路径  生成二维码图片
    *
    * @param content 二维码内容
    * @param imgPath 图片路径
    * @param destPath 二维码下载路径
    * @throws Exception
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:18:21 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, String imgPath, String destPath)  
            throws Exception {  
        QRCodeUtil.encode(content, imgPath, destPath, false,"");  
    }  
  
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容、二维码存放路径、是否压缩图片  生成二维码图片
    *
    * @param content 二维码内容
    * @param destPath 二维码下载路径
    * @param needCompress 是否压缩
    * @throws Exception
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:19:22 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, String destPath,  
            boolean needCompress) throws Exception {  
        QRCodeUtil.encode(content, null, destPath, needCompress,"");  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容、二维码存放路径  生成二维码图片
    *
    * @param content 二维码内容
    * @param destPath 二维码下载路径
    * @throws Exception
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:20:25 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, String destPath) throws Exception {  
        QRCodeUtil.encode(content, null, destPath, false,"");  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容、二维码生成路径、输出流、是否压缩图片 生成二维码图片并以流的形式返回
    *
    * @param content 二维码内容
    * @param imgPath 图片路径
    * @param output 输出流
    * @param needCompress 是否压缩
    * @throws Exception
    * @author: xingliang
    * @date: 2018年1月26日 上午10:21:09 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, String imgPath,  
            OutputStream output, boolean needCompress) throws Exception {  
        BufferedImage image = QRCodeUtil.createImage(content, imgPath,  
                needCompress);  
        ImageIO.write(image, FORMAT_NAME, output);  
    }  
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 根据生成图片的内容 返回 生成的二维码图片流
    *
    * @param content 二维码内容
    * @param output 输出流
    * @return：
    * @throws：
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:24:28 
    *
    * 2018年1月26日     xingliang
     */
    public static void encode(String content, OutputStream output)  
            throws Exception {  
        QRCodeUtil.encode(content, null, output, false);  
    }
    
    
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description: 生成的二维码转换成BASE64编码
    *
    * @param content  二维码内容
    * @param imgPath  二维码生成地址
    * @param destPath 二维码下载地址
    * @param needCompress 是否压缩
    * @param fileName 二维码名称
    * @return：
    * @throws：
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:25:13 
    *
    * 2018年1月26日     xingliang
     */
    public static String encodeToBASE64(String content, String imgPath, String destPath, boolean needCompress, String fileName) throws Exception {
    	BufferedImage image = QRCodeUtil.createImage(content, imgPath,  
                needCompress);  
    	ByteArrayOutputStream os = new ByteArrayOutputStream();//新建流。  
        ImageIO.write(image, FORMAT_NAME, os);//利用ImageIO类提供的write方法，将bi以png图片的数据模式写入流。  
        byte b[] = os.toByteArray();//从流中获取数据数组。  
        byte[] encodeBase64 = Base64.encodeBase64(b);  
        LOGGER.info("Result: ============" + new String(encodeBase64));  
        
        return  new String(encodeBase64);
    }
  
     
    /**
     * 
    * @Function: QRCodeUtil.java
    * @Description:  根据二维码图片 解码图片 返回二维码图片内容
    *
    * @param file 二维码图片
    * @return：
    * @throws：
    *
    * @author: xingliang
    * @date: 2018年1月26日 上午10:27:17 
    *
    * 2018年1月26日     xingliang
     */
    public static String decode(File file) throws Exception {  
        BufferedImage image;  
        image = ImageIO.read(file);  
        if (image == null) {  
            return null;  
        }  
        BufferedImageLuminanceSource source = new BufferedImageLuminanceSource(  
                image);  
        BinaryBitmap bitmap = new BinaryBitmap(new HybridBinarizer(source));  
        Result result;  
        Hashtable hints = new Hashtable();  
        hints.put(DecodeHintType.CHARACTER_SET, CHARSET);  
        result = new MultiFormatReader().decode(bitmap, hints);  
        String resultStr = result.getText();  
        return resultStr;  
    }  
  
     
    /**
     * 根据二维码图片路径  解码图片 返回二维码图片所含的内容
     * @param path
     * @return
     * @throws Exception
     */
    public static String decode(String path) throws Exception {  
        return QRCodeUtil.decode(new File(path));  
    }  
     
}
