package com.yizhi.util.application.regex;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 正则表达式工具类
 * @author 尹玉艳 2018/05/07
 *
 */
public class RegexUtil {
    /**
     * 验证Email
     * @param email email地址，格式：zhangsan@zuidaima.com，zhangsan@xxx.com.cn，xxx代表邮件服务商
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkEmail(String email) { 
        //String regex = "\\w+@\\w+\\.[a-z]+(\\.[a-z]+)?"; 
    	String regex = "[\\w!#$%&'*+/=?^_`{|}~-]+(?:\\.[\\w!#$%&'*+/=?^_`{|}~-]+)*@(?:[\\w](?:[\\w-]*[\\w])?\\.)+[\\w](?:[\\w-]*[\\w])?";
        return Pattern.matches(regex, email); 
    } 
     
    /**
     * 验证身份证号码
     * @param idCard 居民身份证号码15位或18位，最后一位可能是数字或字母
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkIdCard(String idCard) { 
        String regex = "[1-9]\\d{13,16}[a-zA-Z0-9]{1}"; 
        return Pattern.matches(regex,idCard); 
    } 
     
    /**
     * 验证手机号码（支持国际格式，+86135xxxx...（中国内地），+00852137xxxx...（中国香港））
     * @param mobile 移动、联通、电信运营商的号码段
     *<p>移动的号段：134(0-8)、135、136、137、138、139、147（预计用于TD上网卡）
     *、150、151、152、157（TD专用）、158、159、187（未启用）、188（TD专用）</p>
     *<p>联通的号段：130、131、132、155、156（世界风专用）、185（未启用）、186（3g）</p>
     *<p>电信的号段：133、153、180（未启用）、189</p>
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkMobile(String mobile) { 
        String regex = "^((11[0-9])|(12[0-9])|(13[0-9])|(14[0-9])|(15[0-9])|(16[0-9])|(17[0-9])|(18[0-9])|(19[0-9])|)\\d{8}$";
        return Pattern.matches(regex,mobile); 
    } 
     
    /**
     * 验证固定电话号码
     * @param phone 电话号码，格式：国家（地区）电话代码 + 区号（城市代码） + 电话号码，如：+8602085588447
     * <p><b>国家（地区） 代码 ：</b>标识电话号码的国家（地区）的标准国家（地区）代码。它包含从 0 到 9 的一位或多位数字，
     *  数字之后是空格分隔的国家（地区）代码。</p>
     * <p><b>区号（城市代码）：</b>这可能包含一个或多个从 0 到 9 的数字，地区或城市代码放在圆括号——
     * 对不使用地区或城市代码的国家（地区），则省略该组件。</p>
     * <p><b>电话号码：</b>这包含从 0 到 9 的一个或多个数字 </p>
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkPhone(String phone) { 
        String regex = "(\\+\\d+)?(\\d{3,4}\\-?)?\\d{7,8}$"; 
        return Pattern.matches(regex, phone); 
    } 
     
    /**
     * 验证整数（正整数和负整数）
     * @param digit 一位或多位0-9之间的整数
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkDigit(String digit) { 
        String regex = "\\-?[1-9]\\d+"; 
        return Pattern.matches(regex,digit); 
    }

    /**
     * 验证整数和浮点数（正负整数和正负浮点数）
     * @param decimals 一位或多位0-9之间的浮点数，如：1.23，233.30
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkDecimals(String decimals) { 
        String regex = "\\-?[1-9]\\d+(\\.\\d+)?"; 
        return Pattern.matches(regex,decimals); 
    }  
     
    /**
     * 验证空白字符
     * @param blankSpace 空白字符，包括：空格、\t、\n、\r、\f、\x0B
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkBlankSpace(String blankSpace) { 
        String regex = "\\s+"; 
        return Pattern.matches(regex,blankSpace); 
    } 
     
    /**
     * 验证中文
     * @param chinese 中文字符
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkChinese(String chinese) { 
        String regex = "^[\u4E00-\u9FA5]+$"; 
        return Pattern.matches(regex,chinese); 
    } 
     
    /**
     * 验证日期（年月日）
     * @param birthday 日期，格式：1992-09-03，或1992.09.03
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkBirthday(String birthday) { 
        String regex = "[1-9]{4}([-./])\\d{1,2}\\1\\d{1,2}"; 
        return Pattern.matches(regex,birthday); 
    } 
     
    /**
     * 验证URL地址
     * @param url 格式：http://blog.csdn.net:80/xyang81/article/details/7705960? 或 http://www.csdn.net:80
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkURL(String url) { 
        String regex = "(https?://(w{3}\\.)?)?\\w+\\.\\w+(\\.[a-zA-Z]+)*(:\\d{1,5})?(/\\w*)*(\\??(.+=.*)?(&.+=.*)?)?"; 
        return Pattern.matches(regex, url); 
    } 
    
    /**
     * <pre>
     * 获取网址 URL 的一级域
     * </pre>
     * 
     * @param url
     * @return
     */
    public static String getDomain(String url) {
        Pattern p = Pattern.compile("(?<=http://|\\.)[^.]*?\\.(com|cn|net|org|biz|info|cc|tv)", Pattern.CASE_INSENSITIVE);
        // 获取完整的域名
        // Pattern p=Pattern.compile("[^//]*?\\.(com|cn|net|org|biz|info|cc|tv)", Pattern.CASE_INSENSITIVE);
        Matcher matcher = p.matcher(url);
        matcher.find();
        return matcher.group();
    }
    /**
     * 匹配中国邮政编码
     * @param postcode 邮政编码
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkPostcode(String postcode) { 
        String regex = "[1-9]\\d{5}"; 
        return Pattern.matches(regex, postcode); 
    } 
     
    /**
     * 匹配IP地址(简单匹配，格式，如：192.168.1.1，10.10.12.97，没有匹配IP段的大小)
     * @param ipAddress IPv4标准地址
     * @return 验证成功返回true，验证失败返回false
     */ 
    public static boolean checkIpAddress(String ipAddress) { 
        String regex = "[1-9](\\d{1,2})?\\.(0|([1-9](\\d{1,2})?))\\.(0|([1-9](\\d{1,2})?))\\.(0|([1-9](\\d{1,2})?))"; 
        return Pattern.matches(regex, ipAddress); 
    }
    /**
     * 验证部门编码
     * 匹配由数字、26个英文字母或者下划线横杠组成的字符串
     * @param str
     * @return 验证成功返回true，验证失败返回false
     */
    public static boolean checkOrgCode(String str) { 
        String regex = "^[A-Za-z0-9_-]+$"; 
        return Pattern.matches(regex, str); 
    }
    /**
     * 验证用户名
     * 长度不能超过60个字符，不能少于3个字符
     * 只能输入大小写英文字母，数字，下划线(“_”)和”@”
     * 首字母只能为英文字母
     * @param name
     * @return
     */
    public static boolean checkAccountName(String name){
//    	String regex = "^[a-zA-Z][A-Za-z0-9_@]{2,60}$";
        //用户开头（3-50）
        String regex = "^[a-zA-Z0-9][A-Za-z0-9@.#_-]{2,49}$";
        return Pattern.matches(regex, name); 
    }
    /**
     * 验证密码
     * 只能输入大小英文字母、数字和标点符号
     * 密码长度3~20位
     * 标点符号包括：`~!@#$%^&*()_+{}|[]\”’:;<>,.?/
     * @param password
     * @return
     */
    public static boolean checkAccountPassword(String password){
//    	String regex = "^[A-Za-z0-9`~!@#$%^&*()_+{}|\\[\\]\\\\”’:;<>,.?/]+{6,20}$";
//    	String regex = "^([A-Z]|[a-z]|[0-9]|[`~!@#$%^&*()+=|{}':;',\\\\\\\\[\\\\\\\\].<>/?~！@#￥%……&*（）——+|{}【】‘；：”“'。，、？]){3,20}$";
    	String regex = "^([A-Z]|[a-z]|[0-9]|[`~!@#$%^&*()_+-=|{}':;\\[\\]\\\"',\\\\\\\\[\\\\\\\\].<>/?~]){5,20}$";

    	return Pattern.matches(regex, password);
    }
    
    /**
     * 匹配由数字、26个英文字母或者下划线组成的字符串
     * @param str
     * @return 验证成功返回true，验证失败返回false
     */
    public static boolean checkEngNumH(String str) { 
        String regex = "^\\w+$"; 
        return Pattern.matches(regex, str); 
    }
    
    /**
     * 匹配由数字、26个英文字母组成的字符串
     * @param str
     * @return 验证成功返回true，验证失败返回false
     */
    public static boolean checkEngNum(String str) { 
        String regex = "^[A-Za-z0-9]+$"; 
        return Pattern.matches(regex, str); 
    }
    /**
     * 检查微信
     * 只能输入6-20个字母，数字，下划线”_”和减号”-”。
     * @param str
     * @return
     */
    public static boolean checkWechat(String str) {
        String regex = "^[A-Za-z0-9_-]{6,20}$";
        return Pattern.matches(regex, str);
    }
    
    /**
     * 检查部门名称
     * @param name
     * @return
     */
    public static boolean checkOrgName(String name) { 
        String regex = "^[A-Za-z0-9\u4E00-\u9FA5()_@（）*-]{1,60}$";
        return Pattern.matches(regex,name); 
    }
    
    public static void main(String[] args){
    	System.out.println("密码匹配结果=================="+RegexUtil.checkOrgName("（导）_-@(入部)门01*"));
    }

    /**
     * 判断字符串中仅包含汉字、数字、英文
     *
     * @param str
     * @return
     */
    public static boolean isLetterDigitOrChinese(String str) {
        //其他需要，直接修改正则表达式就好
        String regex = "^[a-z0-9A-Z\u4e00-\u9fa5]+$";
        return str.matches(regex);
    }

    public static boolean isSuccessRemark(String str) {
        //其他需要，直接修改正则表达式就好
        String regex = "^[a-z0-9A-Z\u4e00-\u9fa5.:_-]+$";
        return str.matches(regex);
    }


    public static boolean lecturerRegex(String str) {
        //其他需要，直接修改正则表达式就好
        String regex = "^[0-5]\\d*(\\.\\d{1}+)?$";
        return str.matches(regex);
    }

}
