package com.yizhi.core.application.file.util;

import com.alibaba.fastjson.JSONObject;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.util.application.constant.ReturnCode;
import io.minio.MinioClient;
import io.minio.PutObjectOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import cn.hutool.core.codec.Base64Encoder;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;


/**
 * 上传工具类
 */
@Component
public class MinioTools {

    private Logger logger = LoggerFactory.getLogger(MinioTools.class);

    @Autowired
    private MinioClient minioClient;

    @Value("${spring.minio.switchFlag}")
    private String switchFlag;
    /**
     * 创建桶名
     *
     * @param bucketName
     */
    public void makeBucket(String bucketName){
        try {
            if(!minioClient.bucketExists(bucketName)){
                minioClient.makeBucket(bucketName);
            }
        } catch (Exception e) {
            logger.error("makeBucket异常",e);
        }
    }
    /**
     * 功能描述: <br>
     * 〈base64上传〉
     * @Param: [bucketName, fileName, base64]
     * @Return: void
     * @Author: yanglang
     * @Date: 2023/7/24 10:31
     */
    public void uploadImg(String bucketName, String fileName, byte[] base64) {
            InputStream in = new ByteArrayInputStream(base64);
            uplodFile(bucketName,fileName,in);
    }

    /**
     * 上传文件
     *
     * @param bucketName 存储桶名
     * @param fileName  文件名
     * @param in
     */
    public void uplodFile(String bucketName, String fileName, InputStream in){
        makeBucket(bucketName);
        try {
            PutObjectOptions options = new PutObjectOptions(in.available(), -1);
            minioClient.putObject(bucketName,fileName,in,options);
        } catch (Exception e) {
            logger.error("存储文件异常", e);
            throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"文件存储失败");
        }finally {
            try {
                if(in != null){
                    in.close();
                }
            } catch (IOException e) {
                logger.error("IO流关闭异常", e);
                throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"IO流关闭异常");
            }
        }

    }

    /**
     * 删除文件
     *
     * @param bucketName
     * @param fileName
     * @return
     */
    public String deleteFile(String bucketName, String fileName) {
        try {
            minioClient.removeObject(bucketName, fileName);
        } catch (Exception e) {
            logger.error("删除文件失败", JSONObject.toJSONString(e));
            throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"删除文件失败");
        }
        return "删除成功";
    }

    /**
     * 获取文件上传路径
     *
     * @param bucketName
     * @param fileName
     * @return
     */
    public String getFile(String bucketName, String fileName){
        return "/api/v1/buckets/"+bucketName+"/objects/download/?preview=true&prefix="+fileName;
    }

    /**
     * 获取文件
     *
     * @param bucketName
     * @param fileName
     * @return
     */
    public InputStream getFileInputStream(String bucketName, String fileName){
        InputStream in ;
        try {
            // 调用statObject()来判断对象是否存在。
            // 如果不存在, statObject()抛出异常,
            // 否则则代表对象存在。
            minioClient.statObject(bucketName, fileName);

            // 获取"myobject"的输入流。
            in = minioClient.getObject(bucketName, fileName);
        } catch (Exception e) {
            logger.error("获取文件失败", JSONObject.toJSONString(e));
            throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"获取文件失败");
        }
        return in;
    }

    public String getFileBase64Str(String bucketName, String fileName) throws IOException {
        InputStream in = null;
        try {
            // 调用statObject()来判断对象是否存在。
            // 如果不存在, statObject()抛出异常,
            // 否则则代表对象存在。
            minioClient.statObject(bucketName, fileName);

            // 获取"myobject"的输入流。
            in = minioClient.getObject(bucketName, fileName);

            byte[] b = null;
            byte[] buff = new byte[1024];
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            int n = 0;
            while (-1 != (n = in.read(buff)) ){
                output.write(buff, 0, n);
            }
            b = output.toByteArray();
            // 返回 Base64 编码过的字节数组字符串
            return Base64Encoder.encode(b);
        } catch (Exception e) {
            logger.error("获取文件失败", JSONObject.toJSONString(e));
            throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"获取文件失败");
        }finally {
            if(in!=null){
                in.close();
            }
        }
    }

    public String getFileStr(String bucketName, String fileName) throws IOException {
        InputStream in = null;
        try {
            // 调用statObject()来判断对象是否存在。
            // 如果不存在, statObject()抛出异常,
            // 否则则代表对象存在。
            minioClient.statObject(bucketName, fileName);

            // 获取"myobject"的输入流。
            in = minioClient.getObject(bucketName, fileName);

            byte[] b = null;
            byte[] buff = new byte[1024];
            ByteArrayOutputStream output = new ByteArrayOutputStream();
            int n = 0;
            while (-1 != (n = in.read(buff)) ){
                output.write(buff, 0, n);
            }
            b = output.toByteArray();

            return new String(b, "UTF-8");
        } catch (Exception e) {
            logger.error("获取文件失败", JSONObject.toJSONString(e));
            throw new BizException(ReturnCode.BIZ_FAIL.getCode(),"获取文件失败");
        }finally {
            if(in!=null){
                in.close();
            }
        }
    }
}
