package com.yizhi.application.filter;

import com.alibaba.fastjson.JSONObject;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.enums.InternationalEnums;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.netflix.ribbon.RibbonHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletResponse;
import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

/**
 * @ClassName InternationalZuulFilter
 * @Description 管理端国际化返回
 * @Author shengchenglong
 * @DATE 2020/12/7 10:34
 * @Version 1.0
 */
@Component
@Slf4j
public class InternationalZuulFilter extends BaseZuulFilter {

    public static final String INTERNATIONAL_ENUMS = "INTERNATIONAL_ENUMS";

    private static final String DELIMITER_PREFIX = URLDecoder.decode("%7B%7B");
    private static final String DELIMITER_PREFIX_SPLIT = "\\{\\{";

    private static final String DELIMITER_SUFFIX = URLDecoder.decode("%7D%7D");
    private static final String DELIMITER_SUFFIX_SPLIT = "\\}\\}";

    @Autowired
    private RedisCache redisCache;

    @PostConstruct
    public void initRedisI18nKey() {
        InternationalEnums[] values = InternationalEnums.values();
        if (values != null && values.length > 0) {
            for (InternationalEnums value : values) {
                redisCache.hset(INTERNATIONAL_ENUMS, value.getCode(), value.getName());
            }
        }
    }

    /**
     * 非管理端请求，不处理
     *
     * @return
     */
    @Override
    public boolean shouldFilter() {
        RequestContext requestContext = RequestContext.getCurrentContext();
        if (!(requestContext.getResponseStatusCode() == HttpServletResponse.SC_OK)) {
            return false;
        }
        String uri = requestContext.getRequest().getRequestURI();
        //需要单独处理管理端登录接口
        if (uri.contains("loginManage")){
            //bug14909要求
            return true;
        }
        if (!uri.contains("manage") || uri.contains("public")) {
            return false;
        }
        //放行资源扫描接口
        if (uri.contains("/manage/authz/resource/new/scan")){
            return false;
        }
        return true;
    }

    @Override
    public Object run() {
        RequestContext requestContext = RequestContext.getCurrentContext();
        HttpServletResponse response = requestContext.getResponse();
        RibbonHttpResponse zuulResponse = (RibbonHttpResponse) requestContext.get("zuulResponse");
        String res = null;
        JSONObject jsonObject = null;
        try {
            res = StreamUtils.copyToString(zuulResponse.getBody(), Charset.forName("UTF-8"));
            // 返回体为null 直接返回
            if (StringUtils.isEmpty(res)) {
                return null;
            }

            jsonObject = JSONObject.parseObject(res);
            String originSubMsg = String.valueOf(jsonObject.get("subMsg"));
            String result = buildSubMsg(originSubMsg);

            jsonObject.put("subMsg", StringUtils.isEmpty(result) ? originSubMsg : result);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("处理管理端国际化返回 - 错误：responseBody={}", res);
            log.error(e.getLocalizedMessage());
            return null;
        } finally {
            if (jsonObject != null) {
                requestContext.setResponseBody(jsonObject.toJSONString());
            }
        }
        return null;
    }

    private String buildSubMsg(String subMsg) {
        if (StringUtils.isEmpty(subMsg)) {
            return null;
        }
        // 如果不是组合型的
        if (!(subMsg.contains(DELIMITER_PREFIX) && subMsg.contains(DELIMITER_SUFFIX))) {
            return getWordByCode(subMsg);
        }
        // 组合型的
        List<String> list = new ArrayList<>();
        String[] strings = subMsg.split(DELIMITER_PREFIX_SPLIT);
        for (String string : strings) {
            int index = string.indexOf(DELIMITER_SUFFIX);
            if (index > -1) {
                string = string.substring(0, index);
                if (!StringUtils.isEmpty(string)) {
                    list.add(string);
                    System.out.println(string);
                }
            }
        }
        if (!CollectionUtils.isEmpty(list)) {
            String value = null;
            String replace = null;
            for (String s : list) {
                value = getWordByCode(s);
                replace = DELIMITER_PREFIX + s + DELIMITER_SUFFIX;
                subMsg = subMsg.replace(replace, value);
            }
        }
        return subMsg;
    }

    private String getWordByCode(String key) {
        // 返回体为null 直接返回
        if (StringUtils.isEmpty(key)) {
            return null;
        }
        Object value = redisCache.hget(INTERNATIONAL_ENUMS, key);
        // 国际化key为在redis中找到，抛出异常，并返回，不影响业务
        if (value == null) {
            try {
                throw new UnknownInternationalKeyException(key);
            } catch (UnknownInternationalKeyException e) {
                e.printStackTrace();
                log.error("处理管理端国际化返回 - 错误：code={}", key);
                log.error(e.getLocalizedMessage());
                return null;
            }
        }
        return value.toString();
    }


    class UnknownInternationalKeyException extends Exception {

        private String internationalEnumsCode;

        public UnknownInternationalKeyException(String internationalEnumsCode) {
            this.internationalEnumsCode = internationalEnumsCode;
        }

        @Override
        public String getLocalizedMessage() {
            String message = "国际化key未在redis中找到，internationalEnumsCode=" + internationalEnumsCode + "\n";
            message += super.getLocalizedMessage();
            return message;
        }
    }
}
