package com.yizhi.system.application.request;

import com.yizhi.system.application.enums.SiteProtectEnum;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.beans.Transient;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * 站点维护请求体
 */
@Slf4j
@Data
@ApiModel(value = "站点维护请求")
public class SiteProtectReq {

    @ApiModelProperty(value = "站点维护id")
    private String id;

    @ApiModelProperty(value = "站点维护名称")
    private String name;

    @ApiModelProperty(value = "通知数据列表")
    private List<SiteNoticeReq> siteNoticeReqList;

    /**
     * 站点类型， ALL 所有站点， PART 自选站点
     */
    @ApiModelProperty(value = "站点类型")
    private SiteProtectEnum siteType;

    @ApiModelProperty(value = "指定站点信息列表")
    private List<SiteResq> siteList;

    private Long createById;

    private String createByName;

    private Long updateById;

    private String updateByName;


    /**
     * 校验限制条件
     * @return 合法返回ok；否则返回msg
     */
    @Transient
    public String checkFieldLimitCondition(){
        if (StringUtils.isEmpty(this.name)
                || (StringUtils.isEmpty(this.siteType))

        ){
            log.info("站点维护维护名/通知/站点类型包含空值!");
            return "站点维护维护名/通知/站点类型包含空值";
        }
        String trimName = name.trim();
        if (StringUtils.isEmpty(trimName)){
            log.info("站点维护维护名为空!");
            return "站点维护维护名为空!";
        }
        if (trimName.length() > 100){
            log.info("维护名称大于100");
            return "维护名称长度大于100";
        }
        this.name = trimName;
        for (SiteNoticeReq noticeReq: siteNoticeReqList){
            String message = noticeReq.getMessage();
            if (StringUtils.isEmpty(message)){
                log.info("站点维护文案长度为空，维护名= {}",name);
                return "站点维护文案长度为空";
            }
            String trimMessage = message.trim();
            noticeReq.setMessage(trimMessage);
            if (trimMessage.length() == 0 || trimMessage.length() > 100){
                log.info("站点维护文案长度大于100，维护名= {}",name);
                return "站点维护文案长度大于100";
            }
        }
        return "ok";
    }

    /**
     * 判断传入数据是否正确
     * 例如：不能有两个一样的通知类型(站内通知/关站通知 等)
     * @return 合法返回ok；否则返回msg
     */
    @Transient
    public String checkDataValid(){
        if (null == siteNoticeReqList || siteNoticeReqList.isEmpty() || siteNoticeReqList.size() > 2){
            log.info("[站点维护] 维护名= {}的维护通知方式列表为空或列表数据大于两个!",this.name);
            return "通知方式列表为空或列表数据大于两个";
        }
        if (siteNoticeReqList.size() == 2){
            SiteNoticeReq siteNoticeReq1 = siteNoticeReqList.get(0);
            String noticeType1 = siteNoticeReq1.getNoticeType().toString();
            SiteNoticeReq siteNoticeReq2 = siteNoticeReqList.get(1);
            String noticeType2 = siteNoticeReq2.getNoticeType().toString();
            if (StringUtils.isEmpty(noticeType1)
                    || StringUtils.isEmpty(noticeType2)
                    || noticeType1.equals(noticeType2)){
                log.info("[站点维护] 维护名= {},noticeType1 = {},noticeType2 = {} 的维护通知类型重复!",this.name,noticeType1,noticeType2);
                return "站点维护通知类型重复";
            }
        }
        if (SiteProtectEnum.PART.toString().equals(siteType.toString())){
            //自定义选择站点；站点列表不能为空
            if (null == siteList || siteList.isEmpty()){
                log.info("[站点维护] 维护名= {}，siteType = {} 的维护 自定义选择站点列表为空!",this.name,this.siteType.toString());
                return "自定义选择站点列表为空!";
            }
        }
        return "ok";
    }

    /**
     * 校验前端传入通知时间
     * @param req 前端传入通知数据
     * @return 合法返回true；否则返回false
     */
    /*private boolean checkTime(SiteNoticeReq req) {
        Date startTime = req.getStartTime();
        Date endTime = req.getEndTime();
        if (null == startTime || null == endTime){
            log.info("[站点维护] 维护名= {} 维护通知时间存在空值!",this.name);
            return false;
        }
        long currentTimeMillis = System.currentTimeMillis();
        Date currentDate = new Date(currentTimeMillis);
        if (startTime.before(currentDate)){
            log.info("[站点维护] 维护名= {},startTime = {},currentTime = {}, 维护通知开始时间不能小于当前时间!",this.name,startTime,currentDate);
            return false;
        }
        if (endTime.before(startTime)){
            log.info("[站点维护] 维护名= {} endTime = {},currentTime = {} 维护通知结束时间不能小于开始时间!",this.name,endTime,currentDate);
            return false;
        }
        return true;
    }*/

    /**
     * 校验开始时间
     *  @return 合法返回ok；否则返回错误信息
     */
    public String checkStartTimeValid() {
        for (SiteNoticeReq req : siteNoticeReqList){
            Date startTime = req.getStartTime();
            if (null == startTime ){
                log.info("[站点维护] 维护名= {} 维护通知开始时间为空!",this.name);
                return "开始时间为空";
            }
            long currentTimeMillis = System.currentTimeMillis();
            Date currentDate = new Date(currentTimeMillis);
            if (startTime.before(currentDate)){
                log.info("[站点维护] 维护名= {},startTime = {},currentTime = {}, 维护通知开始时间不能小于当前时间!",this.name,startTime,currentDate);
                return "开始时间不能小于当前时间";
            }
        }
        return "ok";
    }

    /**
     * 校验结束时间
     *  @return 合法返回true；否则返回false
     */
    public String checkEndTimeValid() {
        for (SiteNoticeReq req : siteNoticeReqList){
            Date startTime = req.getStartTime();
            Date endTime = req.getEndTime();
            if (null == startTime || null == endTime){
                log.info("[站点维护] 维护名= {} 维护通知结束时间为空!",this.name);
                return "结束时间为空";
            }
            long currentTimeMillis = System.currentTimeMillis();
            Date currentDate = new Date(currentTimeMillis);
            if (endTime.before(startTime)){
                log.info("[站点维护] 维护名= {} endTime = {},startTime = {} 维护通知结束时间不能小于开始时间!",this.name,endTime,startTime);
                return "结束时间不能小于开始时间";
            }
            if (endTime.before(currentDate)){
                log.info("[站点维护] 维护名= {},endTime = {},currentTime = {}, 维护通知结束时间不能小于当前时间!",this.name,endTime,currentDate);
                return "结束时间不能小于当前时间";
            }
        }
        return "ok";
    }

    /**
     * 转换时间格式
     * @return 成功返回ok,否则返回错误信息
     */
    public String formatDate() {
        for (SiteNoticeReq req : siteNoticeReqList){
            Date startTime = req.getStartTime();
            Date endTime = req.getEndTime();
            if (null == startTime || null == endTime){
                log.info("[站点维护] 维护名= {} 维护通知结束时间为空!",this.name);
                return "结束时间为空";
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
            String startTimeStr = sdf.format(startTime);
            String endTimeStr = sdf.format(endTime);
            try {
                Date start = sdf.parse(startTimeStr);
                Date end = sdf.parse(endTimeStr);
                req.setStartTime(start);
                req.setEndTime(end);
            } catch (ParseException e) {
                return e.getMessage();
            }
        }
        return "ok";
    }

    public Long getId() {
        return org.apache.commons.lang3.StringUtils.isBlank(id)?null:Long.parseLong(id);
    }
}