package com.yizhi.system.application.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.page.PageUtil;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.enums.MemberType;
import com.yizhi.core.application.enums.TaskParamsEnums;
import com.yizhi.core.application.exception.BizException;
import com.yizhi.core.application.log.LogQueue;
import com.yizhi.core.application.log.TaskLogDetailEvent;
import com.yizhi.core.application.task.AbstractTaskHandler;
import com.yizhi.core.application.task.TaskExecutor;
import com.yizhi.core.application.vo.DroolsVo;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.system.application.constant.ImpOrgError;
import com.yizhi.system.application.constant.SecurityError;
import com.yizhi.system.application.domain.*;
import com.yizhi.system.application.enums.SystemCommCode;
import com.yizhi.system.application.mapper.*;
import com.yizhi.system.application.model.ImportModel;
import com.yizhi.system.application.model.OrgImpModel;
import com.yizhi.system.application.service.AuthorityService;
import com.yizhi.system.application.service.IAccountService;
import com.yizhi.system.application.service.IAuthzGroupMemberService;
import com.yizhi.system.application.service.IOrganizationService;
import com.yizhi.system.application.service.using.OrganizationCacheService;
import com.yizhi.system.application.service.using.OrganizationIndexService;
import com.yizhi.system.application.util.CollectionUtil;
import com.yizhi.system.application.util.OrgIndexUtil;
import com.yizhi.system.application.util.RightLeftIndex;
import com.yizhi.system.application.vo.ReportOrgRespVO;
import com.yizhi.system.application.vo.RuleVisibleRangeVo;
import com.yizhi.system.application.vo.SystemRuleItemVo;
import com.yizhi.system.application.vo.UpOrgVO;
import com.yizhi.util.application.constant.ReturnCode;
import com.yizhi.util.application.json.JsonUtil;
import com.yizhi.util.application.page.PageInfo;
import com.yizhi.util.application.regex.RegexUtil;
import com.yizhi.util.application.tree.CommonTree;
import com.yizhi.util.application.tree.TreeUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * <p>
 * 组织部门 服务实现类
 * </p>
 *
 * @author fulan123
 * @since 2018-03-08
 */
@Service
@Transactional
public class OrganizationServiceImpl extends ServiceImpl<OrganizationMapper, Organization> implements IOrganizationService {

    private static final Logger logger = LoggerFactory.getLogger(OrganizationServiceImpl.class);

    @Autowired
    private IdGenerator idGenerator;
    @Autowired
    private OrganizationMapper organizationMapper;
    @Autowired
    private AuthorityService authorityService;
    @Autowired
    private AccountMapper accountMapper;
    @Autowired
    private SiteMapper siteMapper;
    @Autowired
    private AuthzGroupMemberMapper groupMemberMapper;
    /*@Autowired
    RabbitTemplate rabbitTemplate;*/
    @Autowired
    private IAuthzGroupMemberService groupMemberService;
    @Autowired
    private CompanyMapper companyMapper;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private OrganizationCacheService organizationCacheService;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private OrganizationIndexService organizationIndexService;

    @Value("${redis.manage.authz.expired}")
    private Long redisManageAuthzExpired;
    private final static String ROOT_ORG_SYMBOL = "root";

    @Autowired
    private IAccountService accountService;

    /**
     * 获取最大的排序+1
     *
     * @param companyId
     * @param parentId
     * @return
     */
    @Override
    public int getMaxSort(long companyId, long parentId) {
        return this.baseMapper.getMaxSort(companyId, parentId);
    }

    @Override
    public List<Organization> fuzzySearchByName(String name, Long companyId) {
        EntityWrapper<Organization> wrapper = QueryUtil.condition(new Organization());
        wrapper.where("company_id={0}", companyId).andNew().
                like("name", name).or().like("code", name);
        return this.selectList(wrapper);
    }

    @Override
    public boolean insertDetail(Organization organization) {
        //1.检查部门code和name是否存在
        if (organizationMapper.countExistCodeOrg(0L, organization.getCompanyId(), organization.getCode()) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "部门编码已存在");
        }
        //2.设置id
        organization.setId(idGenerator.generate());
        organization.setCompanyId(organization.getCompanyId());
        // 设置parentIdLayer
        if (null == organization.getParentId() || organization.getParentId().equals(Long.valueOf(0L))) {
            organization.setParentIdLayer(String.valueOf(organization.getId()));
            // 设置rootId
            organization.setRootId(0L);
        } else {
            Organization parentOrg = organizationMapper.selectById(organization.getParentId());
            StringBuilder parentLayerId = new StringBuilder(parentOrg.getParentIdLayer());
            parentLayerId.append(Organization.PARENT_ID_LAYER_SEPARATOR).append(organization.getId());
            organization.setParentIdLayer(parentLayerId.toString());
            // 设置rootId
            if (parentOrg.getCode() != null && parentOrg.getCode().contains(ROOT_ORG_SYMBOL)) {
                //父级为根部门，则rootId取根部门id
                organization.setRootId(parentOrg.getId());
            } else {
                organization.setRootId(parentOrg.getRootId());
            }
        }
        // 设置左右索引
        if (this.insert(organization)) {
            // 计算左右索引，清除缓存
            organizationIndexService.addOneOrg(organization.getCompanyId(), organization.getId(), organization.getParentId());
            deleteCompanyInfo(organization.getCompanyId());
            organizationIndexService.addInitTask(organization.getCompanyId());
        } else {
            throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
        }
        return true;
    }

    @Override
    public boolean batchDelete(Long[] ids, Long companyId) {
        if (ids.length > 0) {
            for (Long id : ids) {
                delete(id, companyId);
            }
        }
        return true;
    }

    @Override
    public Organization findByCode(String code, Long companyId) {
        Organization organization = new Organization();
        organization.setCode(code);
        organization.setDeleted(false);
        organization.setCompanyId(companyId);
        return organizationMapper.selectOne(organization);
    }

    @Override
    public Page<Organization> list(Organization org, Page<Organization> page) {
        org.setDeleted(false);
        EntityWrapper<Organization> condition = QueryUtil.condition(org);
        if (StringUtils.isNotEmpty(org.getName())) {
            // TODO 这个地方需要优化，如果模糊查询了就不需要按值查询，所以需要从condition里面吧 =name去掉
            condition.like("cn_name", org.getName());
        }
        return this.selectPage(page, condition);
    }

    @Override
    public Boolean delete(Long id, Long companyId) {
        List<Long> ids = authorityService.getChildOrgIds(id, companyId);
        //查询部门下是否存在员工，存在的话不能删除组织
        EntityWrapper condition = new EntityWrapper<Account>();
        condition.in("org_id", ids);
        condition.and("enabled", 1);
        List<Account> accounts = accountMapper.selectList(condition);
        if (CollectionUtils.isNotEmpty(accounts)) {
            throw new BizException(SecurityError.DELETE_ORG_EXIST_ACCOUNT_ERROR, SecurityError.DELETE_ORG_EXIST_ACCOUNT_ERROR_MSG);
        }

        // 查询是否有子组织，否则不能删除
        Organization org = new Organization();
        org.setDeleted(Boolean.FALSE);
        org.setParentId(id);
        EntityWrapper<Organization> ewOrg = new EntityWrapper<>(org);
        if (CollectionUtils.isNotEmpty(organizationMapper.selectList(ewOrg))) {
            throw new BizException(SecurityError.DELETE_ORG_EXIST_ACCOUNT_ERROR, SecurityError.DELETE_ORG_EXIST_ORG_ERROR_MSG);
        }
        Organization organization = new Organization();
        organization.setId(id);
        organization.setDeleted(Boolean.TRUE);
        int num = organizationMapper.updateById(organization);
        // 清除缓存
        if (num == 1) {
            taskExecutor.asynExecute(new AbstractTaskHandler() {
                @Override
                public void handle() {
                    deleteCompanyInfo(companyId);
                }
            });
        }
        return num == 1;
    }

    @Override
    public List<Organization> listChildren(Long parentId) {
        Organization org = organizationMapper.selectById(parentId);
        if (null != org) {
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange((List<Organization>) CollectionUtil.getArrayList(org));
            List<Organization> organizations = organizationMapper.getByIndexAndCompanyId(org.getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices);
            return organizations;
        }
        return null;
    }

    @Override
    public Set<Long> listChildrenId(Collection orgIds) {
        List<Organization> orgs = organizationMapper.selectBatchIds(orgIds);
        if (CollectionUtils.isNotEmpty(orgs)) {
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange(orgs);
            if (CollectionUtils.isEmpty(indices)) {
                return null;
            }
            Set<Long> Ids = organizationMapper.getIdByIndexAndCompanyId(orgs.get(0).getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices);
            return Ids;
        }
        return null;
    }

    @Override
    public Set<Long> listParentIds(Long orgId) {
        Organization organization = new Organization();
        organization.setId(orgId);
        organization = organization.selectById();
        if (null != organization) {
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(CollectionUtil.getArrayList(organization));
            Set<Long> ids = organizationMapper.getIdByIndexAndCompanyId(organization.getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_PARENT, indices);
            return ids;
        }
        return null;
    }

    @Override
    public List<String> listParentNameIds(Long orgId) {
        Organization organization = new Organization();
        organization.setId(orgId);
        organization = organization.selectById();
        if (null != organization) {
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(CollectionUtil.getArrayList(organization));
            List<String> names = organizationMapper.getNamesByIndexAndCompanyId(organization.getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_PARENT, indices);
            return names;
        }
        return null;
    }

    @Override
    public Set<Long> listParentIdsInSite(Long orgId, Long siteId) {
        Set<Long> allParentIds = this.listParentIds(orgId);
        Site site = siteMapper.selectById(siteId);
        // 默认站点直接返回
        if (site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT)) {
            return allParentIds;
        }
        // 多站点，查询范围内的
        Company company = companyMapper.selectById(site.getCompanyId());
        Set<Long> siteOrgIds = this.getOrgIdsInSite(siteId, company.getCode(), site.getCode());
        if (org.springframework.util.CollectionUtils.isEmpty(siteOrgIds)) {
            return null;
        }
        Set<Long> result = new HashSet<>(siteOrgIds.size());
        for (Long id : siteOrgIds) {
            if (allParentIds.contains(id)) {
                result.add(id);
            }
        }
        return result;
    }

    @Override
    public Boolean countExistCodeOrg(Long id, Long companyId, String code) {
        if (organizationMapper.countExistCodeOrg(id, companyId, code) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "部门编码已存在");
        } else {
            return true;
        }
    }

    @Override
    public Boolean countExistNameOrg(Long id, Long companyId, String name) {
        if (organizationMapper.countExistNameOrg(id, companyId, name) > 0) {
            throw new BizException(ReturnCode.RESOURCE_EXIST.getCode(), "部门名称已存在");
        } else {
            return true;
        }
    }

    @Override
    public Long getRootIdByParentId(Long parentId) {
        Organization condition = new Organization();
        condition.setId(parentId);
        condition.setDeleted(false);
        Organization org = organizationMapper.selectOne(condition);
        if (org != null) {
            if (org.getParentId() != 0L) {
                getRootIdByParentId(org.getParentId());
            }
            return org.getId();
        } else {
            throw new BizException(ReturnCode.RESOURCE_NOT_FOUND.getCode(), ReturnCode.RESOURCE_NOT_FOUND.getMsg());
        }
    }

    @Override
    public List<OrgImpModel> orgImport(ImportModel importModel) {
        List<OrgImpModel> impModels = importModel.getData();
        Long companyId = importModel.getCompanyId();
        Long taskId = importModel.getTaskId();
        if (CollectionUtils.isNotEmpty(impModels)) {
            List<Organization> orgs = new ArrayList<>();
            List<TaskLogDetailEvent> failEvents = new ArrayList<>();
            Map<String, Long> orgMap = this.getOrgMap(companyId);
//			List<Long> permitIds=authorityService.getChildOrgIds(ContextHolder.get().getOrgId(), companyId);//987228960986214400L
            for (int j = 0; j < impModels.size(); j++) {
                OrgImpModel model = impModels.get(j);
                String parentCode = model.getParentCode();
                String code = model.getCode();
                String name = model.getName();
                Long orgId = idGenerator.generate();
                int i = j + 2;
                //检查上级部门不能为空
                if (StringUtils.isEmpty(parentCode)) {
                    model.setResult(ImpOrgError.PARENT_CODE_EMPTY);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.PARENT_CODE_EMPTY));
                    continue;
                }
                //检查名称不能为空
                if (StringUtils.isEmpty(name)) {
                    model.setResult(ImpOrgError.NAME_EMPTY);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.NAME_EMPTY));
                    continue;
                }
                //检查部门名称不能为空
                if (StringUtils.isEmpty(code)) {
                    model.setResult(ImpOrgError.CODE_EMPTY);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.CODE_EMPTY));
                    continue;
                }
                //检查部门code长度是否超过30个字符
                if (code.length() > 30) {
                    model.setResult(ImpOrgError.CODE_LENGTH_ERROR);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.CODE_LENGTH_ERROR));
                    continue;
                }
                //检查部门code是否合法
                if (!RegexUtil.checkOrgCode(code)) {
                    model.setResult(ImpOrgError.CODE_FORMAT_ERROR);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.CODE_FORMAT_ERROR));
                    continue;
                }
                //检查部门name长度是否超过30个字符
                if (name.length() > 30) {
                    model.setResult(ImpOrgError.NAME_LENGTH_ERROR);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.NAME_LENGTH_ERROR));
                    continue;
                }
                //检查部门name是否合法
                if (!RegexUtil.checkOrgName(name)) {
                    model.setResult(ImpOrgError.NAME_FORMAT_ERROR);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.NAME_FORMAT_ERROR));
                    continue;
                }
                Long orgParentId = orgMap.get(parentCode.trim());
                //检查上级部门是否存在
                if (null == orgParentId) {
                    model.setResult(ImpOrgError.PARENT_CODE_NOT_EXIST);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.PARENT_CODE_NOT_EXIST));
                    continue;
                }
                //检查部门是否重复
                if (null != orgMap.get(code)) {
                    model.setResult(ImpOrgError.CODE_EXIST);
                    failEvents.add(new TaskLogDetailEvent(taskId, "第" + i + "行:" + ImpOrgError.CODE_EXIST));
                    continue;
                }
                orgMap.put(code, orgId);
                Organization org = new Organization();
                org.setId(orgId);
                org.setCode(code.trim());
                org.setName(name.trim());
                org.setCompanyId(companyId);
                org.setParentId(orgParentId);
                org.setCreateById(importModel.getAccountId());
                org.setCreateByName(importModel.getAccountName());
                org.setCreateTime(new Date());
                orgs.add(org);
            }
            //正确的数据批量插入部门表
            if (CollectionUtils.isNotEmpty(orgs)) {
                if (!this.insertBatch(orgs)) {
                    throw new BizException(ReturnCode.SAVE_FAIL.getCode(), ReturnCode.SAVE_FAIL.getMsg());
                }

                // 计算左右索引，清除缓存
                organizationIndexService.addInitTask(ContextHolder.get().getCompanyId());
                deleteCompanyInfo(ContextHolder.get().getCompanyId());
            }
            //错误的信息发送给任务详情表
            if (CollectionUtils.isNotEmpty(failEvents)) {
                //rabbitTemplate.convertAndSend(LogQueue.BATCH_TASK_LOG_DETAIL_QUEUE, failEvents);
            }
        }
        return impModels;
    }

    @Override
    public Map<String, Long> getOrgMap(Long companyId) {
        Organization condition = new Organization();
        condition.setCompanyId(companyId);
        List<Organization> orgs = this.selectList(QueryUtil.condition(condition));
        Map<String, Long> rtnMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(orgs)) {
            for (Organization org : orgs) {
                rtnMap.put(org.getCode(), org.getId());
            }
        }
        return rtnMap;
    }

    @Override
    public Set<Long> getOrgIdsInSite(Long siteId, String companyCode, String siteCode) {
        List<Long> orgIdList;
        // 先查缓存
        Object object = redisCache.hget(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + companyCode, siteCode);
        if (null != object) {
            orgIdList = JsonUtil.json2List(object.toString(), Long.class);
        } else {
            // 缓存未命中
            Site site = siteMapper.selectById(siteId);
            Set<Long> orgIds = organizationCacheService.cacheSiteOrgId(site, companyCode, site.getCompanyId());
            return orgIds;
        }
        Set<Long> orgIds = new HashSet<>(orgIdList.size());
        orgIds.addAll(orgIdList);
        return orgIds;
    }

    @Cacheable(cacheNames = "auth:manage", key = "'company:'+ #siteId +':site:' + #companyId + ':orgs'")
    @Override
    public List<CommonTree> getBySiteId(Long siteId, Long companyId, String companyCode) {
        List<CommonTree> trees;
        Site site = siteMapper.selectById(siteId);
        // 先查缓存
        Object object = redisCache.hget(CacheNamespace.REIDS_COMPANY_SITE_ORGS + companyCode, site.getCode());
        if (null != object) {
            trees = JsonUtil.json2List(object.toString(), CommonTree.class);
        } else {
            // 缓存未命中
            trees = organizationCacheService.cacheSiteOrg(site, companyCode, companyId);
        }
        return trees;
    }

    @Cacheable(cacheNames = "auth:manage", key = "'role:' + #roleId + ':orgs'")
    @Override
    public Set<Long> getByRoleId4IntersectionSet(List<Long> roleIds, boolean containSubOrg) {
        Set<Long> orgIds = organizationMapper.getByRoleId(roleIds);
        if (containSubOrg && CollectionUtils.isNotEmpty(orgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(orgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange(organizationList);
            orgIds = new HashSet<>();
            orgIds.addAll(organizationMapper.getIdByIndexAndCompanyId(organizationList.get(0).getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices));
        }
        return orgIds;
    }

    @Override
    public Set<Long> getByRoleId4UnionSet(List<Long> roleIds, boolean containSubOrg) {
        Set<Long> orgIds = organizationMapper.getByRoleId(roleIds);
        if (containSubOrg && CollectionUtils.isNotEmpty(orgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(orgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(organizationList);
            orgIds = new HashSet<>();
            orgIds.addAll(organizationMapper.getIdByIndexAndCompanyId(organizationList.get(0).getCompanyId(), OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices));
        }
        return orgIds;
    }

    /**
     * 根据组织列表查询组织人数信息
     *
     * @param orgs
     * @return
     */
    @Override
    public PageInfo<ReportOrgRespVO> getOrgsCountUser(List<Long> orgs, Integer pageNo, Integer pageSize, String orgCodeOrOrgName) {
        if (StringUtils.trimToEmpty(orgCodeOrOrgName).equals(StringUtils.EMPTY)) {
            orgCodeOrOrgName = null;
        } else {
            orgCodeOrOrgName = "%" + orgCodeOrOrgName + "%";
        }
        PageInfo<ReportOrgRespVO> retInfo = new PageInfo<>();
        Page<ReportOrgRespVO> page = null;
        if (pageNo != null) {
            page = new Page<>();
            page.setCurrent(pageNo);
            page.setSize(pageSize);
            List<ReportOrgRespVO> searchRet = organizationMapper.getOrgsCountUserPage(orgs, page, orgCodeOrOrgName);
            retInfo = PageUtil.versa(page.setRecords(searchRet));
        } else {
            List<ReportOrgRespVO> searchRet = organizationMapper.getOrgsCountUser(orgs, orgCodeOrOrgName);
            retInfo.setRecords(searchRet);
        }
        return retInfo;
    }

    /**
     * 查询管辖的组织
     *
     * @param requestContext
     * @return
     */
    @Override
    public List<Long> getMgrOrgs(RequestContext requestContext) {
        // 如果为管理员
        // 获取管理的用户组织信息
        List<Long> orgs = requestContext.getOrgIds();
        if (requestContext.isAdmin()) {
            if (requestContext.getSiteMember().longValue() > 0) {
                AuthzGroupMember gm = new AuthzGroupMember();
                gm.setDeleted(0);
                gm.setGroupId(requestContext.getSiteMember());
                gm.setType(1);
                List<AuthzGroupMember> gms = groupMemberService.selectList(QueryUtil.condition(gm));
                logger.info("管理员非默认站点查询出管辖区关联组织={}", JSON.toJSONString(gms));
                if (!org.springframework.util.CollectionUtils.isEmpty(gms)) {
                    gms.forEach(item -> {
                        orgs.add(item.getMemberId());
                    });
                }
            } else {
                Organization org = new Organization();
                org.setCompanyId(requestContext.getCompanyId());
                org.setDeleted(false);
                List<Organization> orgList = this.selectList(QueryUtil.condition(org));
                logger.info("管理员默认站点查询出企业关联组织={}", JSON.toJSONString(orgList));
                if (CollectionUtils.isNotEmpty(orgList)) {
                    orgList.forEach(item -> {
                        orgs.add(item.getId());
                    });
                }
            }
        }
        logger.info("最后的站点列表={}", JSON.toJSONString(orgs));
        return orgs;
    }

    /**
     * 获取父级组织
     *
     * @param orgId
     * @param companyId
     * @return
     */
    @Override
    public List<String> getOrgNames(Long orgId, Long companyId) {
        String orgNames = "";
        List<Long> ret = authorityService.getChiefOrgIds(orgId, companyId);
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(ret)) {
            EntityWrapper<Organization> where = QueryUtil.condition(new Organization());
            where.where("1 = {0}", 1).and().in("id", ret).orderBy("id");
            List<Organization> selectRet = this.selectList(where);
            if (org.apache.commons.collections.CollectionUtils.isNotEmpty(selectRet)) {
                for (Organization orgItem : selectRet) {
                    orgNames += orgItem.getName() + SystemCommCode.CHIEF_ORG_SPILT.getCode();
                }
                orgNames = orgNames.substring(0, orgNames.length() - 2);
            }
        }
        return Arrays.asList(orgNames.split(SystemCommCode.CHIEF_ORG_SPILT.getCode()));
    }

    /**
     * 根据组织id获取组织下的用户id列表
     *
     * @param orgId
     * @return
     */
    @Override
    public List<Long> getAccountsByOrgId(Long orgId) {
        return accountMapper.getAccountsByOrgId(orgId);
    }

    @Override
    public Boolean update(Organization organization) throws Exception {
        logger.info("传入的对象内容：{}",organization.getName()+":::"+organization.getId()+":::"+organization.getParentId()+":::"+organization.getDescription());
        Organization oldOrg = organizationMapper.selectById(organization.getId());
        if (null == oldOrg) {
            throw new Exception("未知部门id，请检查");
        }

        if (organizationMapper.updateById(organization) == 1) {
            // 是否更改了父部门
            boolean changeParent = organization.getParentId() != null && !oldOrg.getParentId().equals(organization.getParentId());
            // 清除缓存
            deleteCompanyInfo(oldOrg.getCompanyId());
            if (changeParent) {
                Organization parent = organizationMapper.selectById(organization.getParentId());
                if (null == parent) {
                    throw new Exception("未知父部门id，请检查");
                }
                Long companyId = oldOrg.getCompanyId();
                // 计算左右索引
                taskExecutor.asynExecute(new AbstractTaskHandler() {
                    @Override
                    public void handle() {
                        // 更新部门层级信息
                        organization.setParentIdLayer(parent.getParentIdLayer() + Organization.PARENT_ID_LAYER_SEPARATOR + organization.getId());
                        organization.setLayer(parent.getLayer() + 1);
                        // 查询该部门以下的部门
                        List<Organization> childList = organizationMapper.listChildOrg(oldOrg.getLeftIndex(), oldOrg.getRightIndex(), oldOrg.getCompanyId());
                        if (CollectionUtils.isNotEmpty(childList)) {
                            for (Organization child : childList) {
                                if (child.getId().equals(organization.getId())) {
                                    child.setParentId(organization.getParentId());
                                }
                                child.setParentIdLayer(child.getParentIdLayer().replace(oldOrg.getParentIdLayer(), organization.getParentIdLayer()));
                                child.updateById();
                            }
                        }
                        organizationIndexService.addInitTask(companyId);
                    }
                });
            }
        }
        return true;
    }

    @Override
    public List<Organization> getByGroupId(Long companyId, Long... groupId) {
        Set<Long> relatedOrgIds;
        if (groupId.length == 1) {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId[0], MemberType.ORG.getCode());
        } else {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdsAndType(Arrays.asList(groupId), MemberType.ORG.getCode());
        }
        if (CollectionUtils.isNotEmpty(relatedOrgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(relatedOrgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange(organizationList);
            return organizationMapper.getByIndexAndCompanyId(companyId, OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices);
        }
        return null;
    }

    @Override
    public List<RightLeftIndex> getOrgRangeByGroupId(Long... groupId) {
        Set<Long> relatedOrgIds;
        if (groupId.length == 1) {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId[0], MemberType.ORG.getCode());
        } else {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdsAndType(Arrays.asList(groupId), MemberType.ORG.getCode());
        }
        if (CollectionUtils.isNotEmpty(relatedOrgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(relatedOrgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange(organizationList);
            return indices;
        }
        return null;
    }

    @Override
    public List<RightLeftIndex> getOrgRangeByGroupIdNew(Long... groupId) {
        Set<Long> relatedOrgIds;
        if (groupId.length == 1) {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId[0], MemberType.ORG.getCode());
        } else {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdsAndType(Arrays.asList(groupId), MemberType.ORG.getCode());
        }
        if (CollectionUtils.isNotEmpty(relatedOrgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(relatedOrgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getParentsRange(organizationList);
            return indices;
        }
        return null;
    }

    @Override
    public Set<Long> getIdsByGroupId(Long companyId, Long... groupId) {
        Set<Long> relatedOrgIds;
        if (groupId.length == 1) {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdAndType(groupId[0], MemberType.ORG.getCode());
        } else {
            relatedOrgIds = groupMemberMapper.getMemberIdByGroupIdsAndType(Arrays.asList(groupId), MemberType.ORG.getCode());
        }
        if (CollectionUtils.isNotEmpty(relatedOrgIds)) {
            List<Organization> organizationList = organizationMapper.selectBatchIds(relatedOrgIds);
            List<RightLeftIndex> indices = OrgIndexUtil.getChildrenRange(organizationList);
            return organizationMapper.getIdByIndexAndCompanyId(companyId, OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices);
        }
        return null;
    }

    @Override
    public List<UpOrgVO> findupOrgsByOrgId(List<Long> orgIds) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return null;
        }
        return this.findupOrgsByOrgId(orgIds);
    }

    /**
     * 获取指定范围（站点，组织列表）的包含的组织ids
     *
     * @param siteId
     * @param orgIds
     * @param site
     * @return
     */
    @Override
    public Set<Long> getDestOrgIds(Long siteId, List<Long> orgIds, Site site) {
        List<CommonTree> trees = this.getBySiteId(siteId, site.getCompanyId(), ContextHolder.get().getCompanyCode());
        return (Set<Long>) TreeUtil.getCheckedWithSubId(TreeUtil.traverseTree(trees), orgIds);
    }

    @Override
    public Map<Long, String> getParentNameWithCurrentName(List<Long> orgIds) {
        if (CollectionUtils.isNotEmpty(orgIds)) {
            List<Organization> organizations = organizationMapper.selectBatchIds(orgIds);
            if (CollectionUtils.isNotEmpty(organizations)) {

                Set<Long> parentIds = new HashSet<>();
                Map<Long, String> map = new HashMap<>();

                organizations.forEach(item -> {
                    parentIds.add(item.getParentId());
                    map.put(item.getId(), item.getName());
                });

                parentIds.remove(0L);
                if (CollectionUtils.isNotEmpty(parentIds)) {
                    List<Organization> parentOrganizations = organizationMapper.selectBatchIds(parentIds);
                    if (CollectionUtils.isNotEmpty(parentOrganizations)) {
                        Map<Long, String> parentIdName = new HashMap<>();
                        parentOrganizations.forEach(item -> parentIdName.put(item.getId(), item.getName()));

                        organizations.forEach(item -> {
                            String name = map.get(item.getId());
                            String parentName = parentIdName.get(item.getParentId());
                            if (StringUtils.isNotEmpty(name)) {
                                if (StringUtils.isNotEmpty(parentName)) {
                                    map.put(item.getId(), name.concat("/").concat(parentName));
                                }
                            }
                        });
                    }
                }
                return map;
            }
        }
        return null;
    }

    /**
     * 删除企业信息（一般该方法都是因为权限信息需要重置：部门数据修改等）
     */
    @Override
    public void deleteCompanyInfo(Long companyId) {
        try {
            logger.info("开始删除部门数据：{}",companyId);
            Company company = new Company();
            company.setId(companyId);
            company = company.selectById();
            if (company == null){
                logger.info("开始删除部门数据，查询到公司为空，公司id:,：{}",companyId);
                return;
            }
            // 删除企业信息cache
            String[] companyCode = new String[]{company.getCode()};
            redisCache.hdel(CacheNamespace.REDIS_COMPANIES, companyCode);
            // 删除组织信息cache
            redisCache.hdel(CacheNamespace.REDIS_COMPANY_ORGS, companyCode);

            Set<String> siteCodes = siteMapper.getSiteCode(companyId);
            if (CollectionUtils.isNotEmpty(siteCodes)) {
                logger.info("开始删除部门数据：{}",companyId);
                String[] codes = new String[siteCodes.size()];
                int index = 0;
                for (String s : siteCodes) {
                    codes[index] = company.getCode() + CacheNamespace.REDIS_HASH_SEPARATOR + s;
                    index++;
                }
                // 删除站点信息cache
                redisCache.hdel(CacheNamespace.REDIS_SITES, codes);
                // 删除企业下站点:部门 信息
                redisCache.delete(CacheNamespace.REIDS_COMPANY_SITE_ORGS + company.getCode());
                redisCache.delete(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + company.getCode());
                redisCache.delete(CacheNamespace.REDIS_AUTHZ_FRONT_COMPANY_ACCOUNT_ACCESS_SITE + company.getCode());
                // 删除管理员权限
                redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_MANAGERIDS + company.getCode());
                redisCache.delete(CacheNamespace.REDIS_AUTHZ_MANAGE_COMPANY_ORGIDS + company.getCode());
            }else {
                logger.info("开始删除部门数据，查询到公司下站点为空，公司id:,：{}",companyId);
            }
        } catch (Exception e) {
            logger.error("删除企业redis错误：try-catch处理，不影响返回", e);
        }
    }

    @Override
    public Map<Long, String> getOrgNamesByOrgIds(List<Long> orgIds) {
        List<Organization> list = organizationMapper.selectByIds(orgIds);
        Map<Long, String> map = new HashMap<>(16);
        if (CollectionUtils.isNotEmpty(list)) {
            for (Organization organization : list) {
                map.put(organization.getId(), organization.getName());
            }
        }
        return map;
    }

    @Override
    public Page getClassifyNameByDrools(String field, String value, Page page) {
        if (StringUtils.isEmpty(field)) {
            logger.info("列名不能为空！");
            return page;
        }
        RequestContext requestContext = ContextHolder.get();
        Long companyId = requestContext.getCompanyId();
        List<DroolsVo> voList = null;

        Organization organization = new Organization();
        organization.setCompanyId(companyId);
        organization.setDeleted(false);
        EntityWrapper wrapper = new EntityWrapper(organization);
        wrapper.setSqlSelect("distinct(" + field + ")," + "id ")
                .isNotNull(field)
                .like(field, value)
                .addFilter(field + "!=''")
                .orderBy("create_time", false);

        List<Organization> list = this.baseMapper.selectPage(page, wrapper);
        if (!CollectionUtils.isEmpty(list)) {
            voList = new ArrayList<>(list.size());
            for (Organization a : list) {
                DroolsVo vo = new DroolsVo();
                vo.setTaskId(a.getId());
                vo.setTaskFieldValue(a.getName());
                vo.setTaskParamsType(field);
                voList.add(vo);
            }
        }
        page.setRecords(voList);
        return page;
    }

    @Override
    public List<Long> getOrgIdsByName(RuleVisibleRangeVo rangeVo, Long companyId, Long ruleId) {
        List<Long> orgIds = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(rangeVo.getRuleItemVos())) {

            EntityWrapper wrapper = null;
            Organization org = new Organization();
            org.setCompanyId(companyId);
            org.setDeleted(false);
            wrapper = new EntityWrapper(org);
            wrapper.setSqlSelect("id,left_index as leftIndex,right_index as rightIndex");

            for (SystemRuleItemVo a : rangeVo.getRuleItemVos()) {
                if (TaskParamsEnums.ORG.getCode().equalsIgnoreCase(a.getPropertyName())) {
                    if (a.getRuleId().equals(ruleId)) {
                        a.setPropertyName("name");
                        AccountServiceImpl.buildWrapper(wrapper, a);
                        //还原，为后面org判断起作用
                        a.setPropertyName("org");
                    }
                }
            }
            List<Organization> orgList = this.selectList(wrapper);
            if (CollectionUtils.isNotEmpty(orgList)) {
                List<RightLeftIndex> indices = new ArrayList<>(orgList.size());
                orgList.forEach(o -> {
                    RightLeftIndex index = new RightLeftIndex(o.getLeftIndex(), o.getRightIndex());
                    getMaxIndex(indices, index);
                });

                Set<Long> orgIds1 = this.baseMapper.getIdByIndexAndCompanyId(companyId, OrganizationMapper.SELECT_LAYER_TYPE_CHILD, indices);
                orgIds.clear();
                orgIds.addAll(orgIds1);
            }
        }
        return orgIds == null ? new ArrayList<>() : orgIds;
    }

    @Override
    public String getOrganizationNames(Long orgId) {
        String orgNames = "";
        orgNames = getName(orgId, orgNames);
        return orgNames;
    }

    public String getName(Long orgId, String orgNames) {

        Organization organization = organizationMapper.selectById(orgId);
        Long parentId = organization.getParentId();
        System.out.println(JSON.toJSONString(organization));
        if (null != organization) {
            if (StringUtils.isNotBlank(orgNames)) {
                orgNames = organization.getName() + "," + orgNames;
            } else {
                orgNames = organization.getName();
            }
        }
        int layer = organization.getLayer();

        while (null != organization && layer > 0 ) {
            Organization organization1 = new Organization();
            organization1 = organizationMapper.selectById(parentId);
            if (organization1 == null) {
                logger.error("查询orgId:" + orgId + "的parentId父对象异常");
                break;
            }
            if (organization1.getParentId() .equals(organization1.getId())){
                break;
            }
            parentId = organization1.getParentId();
            System.out.println(JSON.toJSONString(organization1));
            if (StringUtils.isNotBlank(organization1.getName())) {
                orgNames = organization1.getName() + "," + orgNames;
            } else {
                orgNames = organization1.getName();
            }
            layer = organization1.getLayer();
        }
        return orgNames;
    }

    /**
     * 获取最大的index的集合
     *
     * @param list
     * @param index
     */
    private static void getMaxIndex(List<RightLeftIndex> list, RightLeftIndex index) {
        if (CollectionUtils.isEmpty(list)) {
            list.add(index);
            return;
        }
        /**
         * 1:不相交
         * 2：被包含于其中的某一个范围
         * 3：包含其中的某一个范围
         */
        int operation = -1;
        int j = -1;
        for (int i = 0; i < list.size(); i++) {
            j = i;
            // 1:不相交
            if (index.getLeft() > list.get(i).getRight() || index.getRight() < list.get(i).getLeft()) {
                operation = 1;
            }
            // 2：被包含于其中的某一个范围
            else if (index.getLeft() > list.get(i).getLeft() && index.getRight() < list.get(i).getRight()) {
                operation = 2;
                break;
            }
            // 3：包含其中的某一个范围
            else if (index.getLeft() < list.get(i).getLeft() && index.getRight() > list.get(i).getRight()) {
                operation = 3;
                break;
            }
        }

        if (operation == 1) {
            list.add(index);
        } else if (operation == 2) {

        } else if (operation == 3) {
            list.remove(j);
            list.add(index);
        }
    }
}
