package com.yizhi.system.application.service.using;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.system.application.constant.AuthzConstant;
import com.yizhi.system.application.mapper.OrganizationMapper;
import com.yizhi.system.application.service.IOrganizationService;
import com.yizhi.system.application.domain.Company;
import com.yizhi.system.application.domain.Organization;
import com.yizhi.system.application.domain.Site;
import com.yizhi.util.application.tree.CommonTree;
import com.yizhi.util.application.tree.TreeUtil;
import com.yizhi.system.application.vo.CompanyVO;
import com.yizhi.system.application.vo.SiteVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * @Author: shengchenglong
 * @Date: 2018/8/21 14:14
 */
@Component
public class OrganizationCacheService {

    private static final Logger LOGGER = LoggerFactory.getLogger(OrganizationCacheService.class);


    @Value("${redis.manage.authz.expired}")
    private Long redisManageAuthzExpired;

    @Autowired
    private RedisCache redisCache;
    @Autowired
    private IOrganizationService organizationService;
    @Autowired
    private OrganizationMapper organizationMapper;

    /**
     * 缓存企业信息
     *
     * @param company
     */
    public void cacheCompany(Company company) throws Exception {
        // 企业信息
        CompanyVO companyVO = new CompanyVO();
        BeanUtils.copyProperties(company, companyVO);
        // 查询出默认组织code
        Organization rootOrg = new Organization();
        rootOrg.setDeleted(false);
        rootOrg.setCompanyId(companyVO.getId());
        EntityWrapper<Organization> ew = new EntityWrapper<>(rootOrg);
        ew.like("code", AuthzConstant.DEFAULT_COMPANY_ORG_ROOT_CODE_SUFFIX);
        rootOrg = rootOrg.selectOne(ew);
        if (rootOrg == null) {
            LOGGER.error("！！！！！！查找企业：" + company.getName() + " 根组织失败！！！！！！");
            throw new Exception("！！！！！！查找企业：" + company.getName() + " 根组织失败！！！！！！");
        }
        companyVO.setRootOrgId(rootOrg.getId());
        LOGGER.info("------Redis操作：重置企业：{} 信息 开始------", company.getName());
        redisCache.hset(CacheNamespace.REDIS_COMPANIES, company.getCode(), JSON.toJSONString(companyVO), redisManageAuthzExpired);
        LOGGER.info("------Redis操作：重置企业：{} 信息 完成------", company.getName());
    }

    /**
     * 缓存组织信息
     *
     * @param company
     */
    public List<CommonTree> cacheCompanyOrganization(Company company) {
        if (company == null){
            LOGGER.error("company为空！！！");
        }
        Organization orgExample = new Organization();
        orgExample.setCompanyId(company.getId());
        orgExample.setDeleted(Boolean.valueOf(false));
        List<Organization> organizations = orgExample.selectList(new EntityWrapper(orgExample));
        if (CollectionUtils.isNotEmpty(organizations)) {
            List<CommonTree> orgTreeList = new ArrayList<>(organizations.size());
            for (Organization organization : organizations) {
                CommonTree commonTree = new CommonTree();
                commonTree.setPId(organization.getParentId());
                commonTree.setId(organization.getId());
                commonTree.setName(organization.getName());
                commonTree.setNode(organization);
                orgTreeList.add(commonTree);
            }
            orgTreeList = TreeUtil.buildTree(orgTreeList, 0L);
            LOGGER.info("------Redis操作：重置企业：{} 的组织信息，组织数目：{} 开始------", company.getName(), organizations.size());
            redisCache.hset(CacheNamespace.REDIS_COMPANY_ORGS, company.getCode(), JSON.toJSONString(orgTreeList), redisManageAuthzExpired);
            LOGGER.info("------Redis操作：重置企业：{} 的组织信息，组织数目：{} 完成------", company.getName(), organizations.size());
            return orgTreeList;
        }
        return null;
    }

    /**
     * 缓存站点信息
     *
     * @param company
     * @param reCacheSiteOrgs 是否跟新站点信息缓存
     */
    public void cacheSite(Company company, boolean reCacheSiteOrgs) {
        Site siteExample = new Site();
        siteExample.setEnabled(Boolean.valueOf(true));
        siteExample.setCompanyId(company.getId());
        List<Site> siteList = siteExample.selectList(new EntityWrapper(siteExample));
        if (CollectionUtils.isNotEmpty(siteList)) {
            SiteVO siteVO;
            for (Site site : siteList) {
                siteVO = new SiteVO();
                BeanUtils.copyProperties(site, siteVO);
                LOGGER.info("------Redis操作：重置企业：{} 的站点信息，站点数目：{} 开始------", company.getName(), siteList.size());
                redisCache.hset(CacheNamespace.REDIS_SITES,
                        company.getCode() + CacheNamespace.REDIS_HASH_SEPARATOR + site.getCode(),
                        JSON.toJSONString(siteVO), redisManageAuthzExpired);
                LOGGER.info("------Redis操作：重置企业：{} 的站点信息，站点数目：{} 完成------", company.getName(), siteList.size());
                if (reCacheSiteOrgs) {
                    // 站点组织信息
                    cacheSiteOrg(site, company.getCode(), company.getId());
                    cacheSiteOrgId(site, company.getCode(), company.getId());
                }
            }
        }
    }

    /**
     * @param site
     * @param companyCode
     * @param companyId
     * @return
     */
    public List<CommonTree> cacheSiteOrg(Site site, String companyCode, Long companyId) {
        List<Organization> organizations = getOrganizations(site, companyId);
        if (CollectionUtils.isNotEmpty(organizations)) {
            List<CommonTree> orgTreeList = new ArrayList<>(organizations.size());
            for (Organization organization : organizations) {
                CommonTree commonTree = new CommonTree();
                commonTree.setPId(organization.getParentId());
                commonTree.setId(organization.getId());
                commonTree.setName(organization.getName());
                commonTree.setNode(organization);
                orgTreeList.add(commonTree);
            }
            orgTreeList = TreeUtil.buildTree(orgTreeList);
            try {
                LOGGER.info("------Redis操作：重置企业站点：{} 的组织信息 开始------", site.getName());
                redisCache.hset(CacheNamespace.REIDS_COMPANY_SITE_ORGS + companyCode, site.getCode(), JSON.toJSONString(orgTreeList), redisManageAuthzExpired);
                LOGGER.info("------Redis操作：重置企业站点：{} 的组织信息 完成------", site.getName());
            } catch (Exception e) {
                LOGGER.error("!!!!!!!! Redis操作：重置企业站点：" + site.getName() + " 的组织信息 error !!!!!!!!", e);
            }
            return orgTreeList;
        }
        return null;
    }

    public Set<Long> cacheSiteOrgId(Site site, String companyCode, Long companyId) {
        Set<Long> orgIds = getOrganizationIds(site, companyId);
        if (CollectionUtils.isNotEmpty(orgIds)) {
            try {
                LOGGER.info("------Redis操作：重置企业站点：{} 的组织信息 开始------", site.getName());
                redisCache.hset(CacheNamespace.REIDS_COMPANY_SITE_ORGIDS + companyCode, site.getCode(), JSON.toJSONString(orgIds), redisManageAuthzExpired);
                LOGGER.info("------Redis操作：重置企业站点：{} 的组织信息 完成------", site.getName());
            } catch (Exception e) {
                LOGGER.error("!!!!!!!! Redis操作：重置企业站点：" + site.getName() + " 的组织信息 error !!!!!!!!", e);
            }
            return orgIds;
        }
        return null;
    }

    /**
     * @param site
     * @param companyId
     * @return
     */
    private List<Organization> getOrganizations(Site site, Long companyId) {
        // 如果是默认站点
        if (site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT)) {
            Organization organization = new Organization();
            organization.setDeleted(false);
            organization.setCompanyId(companyId);
            return organization.selectList(new EntityWrapper(organization));
        } else {
            // 拿出管辖区id
            Long userGroupId = site.getUserGroupId();
            return organizationService.getByGroupId(companyId, userGroupId);
        }
    }

    /**
     * @param site
     * @param companyId
     * @return
     */
    private Set<Long> getOrganizationIds(Site site, Long companyId) {
        // 如果是默认站点
        if (site.getType().equals(AuthzConstant.SITE_TYPE_DEFAULT)) {
            return organizationMapper.getOrgIdsByCompanyId(companyId);
        } else {
            // 拿出管辖区id
            Long userGroupId = site.getUserGroupId();
            return organizationService.getIdsByGroupId(companyId, userGroupId);
        }
    }
}
