package com.yizhi.application.cache.aop;

import com.yizhi.application.cache.RedisCacheManagerEx;
import com.yizhi.application.cache.annotation.ExamCach;
import com.yizhi.application.cache.annotation.ExamCachDel;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.Cache.ValueWrapper;
import org.springframework.context.expression.MethodBasedEvaluationContext;
import org.springframework.core.DefaultParameterNameDiscoverer;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;

/**
 * 考试缓存生成，根据ExamCach注解生成考试缓存
 * 
 * @author mei
 *
 */
@Aspect
@Component
public class ExamCacheAspect {
	
	/**
	 * Redis缓存
	 */
	@Autowired
	private RedisCacheManagerEx redisCacheManagerEx;	
	
	/**
	 * SpEL表达式解析器
	 */
	private static final ExpressionParser expressionParser = new SpelExpressionParser();
	
	/**
	 * 获取方法参数信息
	 */
	private static final ParameterNameDiscoverer defaultParameterNameDiscoverer = new DefaultParameterNameDiscoverer();

	/**
	 * 缓存处理
	 * @param point 目标对象
	 * @param examCach 考试缓存注解
	 * @return
	 */
	@Around(value="@annotation(examCach)")
	public Object examAround(ProceedingJoinPoint point, ExamCach examCach) throws Throwable {
		//获取方法中是否添加了ExamCach注解
		if(examCach == null){
			return null;
		}
		
		try {
			/**
			 *  获取目标对象的信息
			 */
			Object target = point.getTarget(); // 目标对象
			Object[] arguments = point.getArgs(); // 目标对象参数
			MethodSignature methodSignature = (MethodSignature) point.getSignature(); // 目标对象方法签名
			Method method = methodSignature.getMethod(); // 目标对象方法

			/**
			 *  解析SpEL表达式
			 */
			Expression examCacheNameExp = getExpression(examCach.cacheName()); 	//考试缓存名称表达式
			Expression examCacheKeyExp = getExpression(examCach.cacheKey()); 	//考试缓存键名称表达式
			Expression expireTimeExp = getExpression(examCach.expireTime()); 	//考试缓存失效时间表达式

			/**
			 *  根据SpEL表达式，从目标对象方法中的参数获取对应的值
			 */
			EvaluationContext evaluationContext = new MethodBasedEvaluationContext(target, method, arguments,
					defaultParameterNameDiscoverer); //获取目标方法的上下文
			//根据表达式获取真实的值
			String cacheName = examCacheNameExp.getValue(evaluationContext, String.class); 	//缓存名称
			String cacheKey = examCacheKeyExp.getValue(evaluationContext, String.class); 	//缓存
			String cacheTime = expireTimeExp.getValue(evaluationContext, String.class); 	//失效时间（秒）
			if(StringUtils.isBlank(cacheTime)){
				cacheTime = "0";
			}
			/**
			 * 缓存处理
			 */
			Cache cache = redisCacheManagerEx.getRedisCache(cacheName);
			if(cache == null){
				cache = redisCacheManagerEx.createRedisCache(cacheName, Long.valueOf(cacheTime));
			}
			
			//防止高并发下缓存穿透
			ValueWrapper valApper = cache.get(cacheKey);
			Object resObj = null;
			if(valApper == null){
				synchronized(cacheKey.intern()){
					valApper = cache.get(cacheKey);
					if(valApper == null){
						resObj = point.proceed(); 	//首次执行目标中的方法
						if(cache != null && resObj != null){
							cache.put(cacheKey, resObj);
						}
					} else {
						resObj = valApper.get();
					}
				}
			} else {
				resObj = valApper.get();
			}
			
			return resObj;
		} catch (Throwable e) {
			//e.printStackTrace();
			throw e;
		}
		
		//return null;
	}
	
	@Around(value="@annotation(examCachDel)")
	public Object examCachDel(ProceedingJoinPoint point, ExamCachDel examCachDel) throws Throwable {
		//获取方法中是否添加了ExamCach注解
		if(examCachDel == null){
			return false;
		}
		
		try {
			/**
			 *  获取目标对象的信息
			 */
			Object target = point.getTarget(); // 目标对象
			Object[] arguments = point.getArgs(); // 目标对象参数
			MethodSignature methodSignature = (MethodSignature) point.getSignature(); // 目标对象方法签名
			Method method = methodSignature.getMethod(); // 目标对象方法

			/**
			 *  解析SpEL表达式
			 */
			Expression examCacheNameExp = getExpression(examCachDel.cacheName()); 	//考试缓存名称表达式
			Expression examCacheKeyExp = getExpression(examCachDel.cacheKey()); 	//考试缓存键名称表达式

			/**
			 *  根据SpEL表达式，从目标对象方法中的参数获取对应的值
			 */
			EvaluationContext evaluationContext = new MethodBasedEvaluationContext(target, method, arguments,
					defaultParameterNameDiscoverer); //获取目标方法的上下文
			//根据表达式获取真实的值
			String cacheName = examCacheNameExp.getValue(evaluationContext, String.class); 	//缓存名称
			String cacheKey = examCacheKeyExp.getValue(evaluationContext, String.class); 	//缓存
			/**
			 * 缓存处理
			 */
			Cache cache = redisCacheManagerEx.getRedisCache(cacheName);
			if(cache == null){
				cache = redisCacheManagerEx.createRedisCache(cacheName, Long.valueOf(1L));
			}
			point.proceed();
			cache.evict(cacheKey);
			
			return true;
		} catch (Throwable e) {
			//e.printStackTrace();
			throw e;
		}

		//return false;
	}

	/**
	 * SpEL表达式解析
	 * @param expression SpEL表达式
	 * @return
	 */
	private Expression getExpression(String expression) {
		Expression expressionObject = null;
		synchronized (ExamCacheAspect.class) {
			expressionObject = expressionParser.parseExpression(expression);
		}
		
		return expressionObject;
	}
}