package com.yizhi.application.service.impl;

import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.baomidou.mybatisplus.plugins.Page;
import com.baomidou.mybatisplus.service.impl.ServiceImpl;
import com.yizhi.application.constant.PointTypeEnum;
import com.yizhi.application.domain.Point;
import com.yizhi.application.domain.PointDetails;
import com.yizhi.application.domain.PointTypeStrategy;
import com.yizhi.application.domain.PointUser;
import com.yizhi.application.mapper.PointDetailsMapper;
import com.yizhi.application.mapper.PointMapper;
import com.yizhi.application.mapper.PointProductMapper;
import com.yizhi.application.mapper.PointUserMapper;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.service.PointDetailsService;
import com.yizhi.application.service.PointService;
import com.yizhi.application.service.PointUserService;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.point.application.vo.PointDetailListVO;
import com.yizhi.point.application.vo.PointUserExchangeVO;
import com.yizhi.util.application.date.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;

/**
 * <p>
 * 积分设置 服务实现类
 * </p>
 *
 * @author bob123
 * @since 2018-04-20
 */
@Slf4j
@Service
public class PointServiceImpl extends ServiceImpl<PointMapper, Point> implements PointService {
	
	@Autowired
	PointMapper pointMapper;
	@Autowired
	IdGenerator idGenerator;
	@Autowired
	PointDetailsService pointDetailsService;
	@Autowired
	PointUserService pointUserService;
    @Autowired
    private PointUserMapper pointUserMapper;
	@Autowired
	private PointProductMapper pointProductMapper;
	@Autowired
	private PointDetailsMapper pointDetailsMapper;

	@Override
	public Point pointList(Long companyId, Long siteId) {
		Point point = pointMapper.pointList(companyId, siteId);
		if(point == null){
			point = new Point();
			point.setState(0);
			point.setUnit("积分");
		}
		return point;
	}

	@Override
	public boolean updateList(Point point) {
		Integer state = point.getState(); 
		String unit = point.getUnit();
		Long companyId = point.getCompanyId();
		Long siteId = point.getSiteId();
		Long updateAccountId = point.getUpdateById();
		String updateAccountName = point.getUpdateByName();
		Integer count = pointMapper.updateList(state, unit, companyId, siteId, updateAccountId, updateAccountName);
		if(count > 0){
			return true;
		} else {
			point.setId(idGenerator.generate());
			point.setCreateById(updateAccountId);
			point.setCreateByName(updateAccountName);
			point.setCreateTime(new Date());
			count = pointMapper.insert(point);
			if(count > 0){
				return true;
			}
		}
		
		return false;
	}


	@Transactional(rollbackFor = Exception.class)
	public void addPoint(Long accountId, String type, String sourceId){
		if(accountId == null || StringUtils.isBlank(type)){
			log.error("增加积分入参异常 accountId:{}, type:{}, sourceId:{}", accountId, type, sourceId);
			return;
		}
		log.info("增加积分入参: accountId:{}, type:{}, sourceId:{}", accountId, type, sourceId);
		Long companyId = ContextHolder.get().getCompanyId();
		Long orgId = ContextHolder.get().getOrgId();
		Long siteId = ContextHolder.get().getSiteId();
		String accountName = ContextHolder.get().getAccountName();
		//执行积分奖励策略进行相应的业务校验
		PointTypeStrategy strategy = getPointByType(type, accountId, sourceId);
		log.info("获取赠送积分策略:{}", JSONUtil.toJsonStr(strategy));
		if(strategy != null) {
			PointUser pu = new PointUser();
			pu.setUserId(accountId);
			EntityWrapper<PointUser> wrapper = new EntityWrapper<PointUser>(pu);
			pu = pointUserService.selectOne(wrapper);
			Long id = idGenerator.generate();
			if (pu == null) {
				pu = new PointUser();
				pu.setId(id);
				pu.setUserId(accountId);
				pu.setTotalPoint(strategy.getPoint());
				pu.setState(1);
				pu.setCreateById(accountId);
				pu.setCreateByName(accountName);
				pu.setCreateTime(new Date());
			} else {
				pu.setTotalPoint(pu.getTotalPoint() + strategy.getPoint());
				pu.setUpdateById(accountId);
				pu.setUpdateByName(accountName);
				pu.setUpdateTime(new Date());
			}
			pu.setOrgId(orgId);
			pu.setSiteId(siteId);
			pu.setCompanyId(companyId);
			boolean b = pointUserService.insertOrUpdate(pu);
			if(b) {
				PointDetails pd = new PointDetails();
				pd.setId(idGenerator.generate());
				pd.setTime(new Date());
				pd.setChangeBefore(pu.getTotalPoint());
				pd.setPoint(strategy.getPoint());
				pd.setChangeAfter(pu.getTotalPoint() + strategy.getPoint());
				pd.setMultiple(1);
				pd.setAccountId(accountId);
				pd.setLearnName(strategy.getPointTypeEnum().getDesc());
				pd.setLearnSource(strategy.getLearnSource());
				pd.setLearnType(strategy.getPointTypeEnum().getKey());
				pd.setState(1);
				pd.setCreateById(accountId);
				pd.setCreateByName(accountName);
				pd.setCreateTime(new Date());
				pd.setOrgId(orgId);
				pd.setCompanyId(companyId);
				pd.setSiteId(siteId);
				pointDetailsService.insert(pd);
			}
		}
	}

	public PointTypeStrategy getPointByType(String type, Long accountId, String sourceId){
		PointTypeStrategy strategy = null;
		if(type.equals(PointTypeEnum.LOGIN.getKey())){
			//查询当天是否首次登录，奖励积分
			String date = DateUtil.format(new Date(), "yyyy-MM-dd");
			PointDetails pd = new PointDetails();
			pd.setAccountId(accountId);
			pd.setLearnType(PointTypeEnum.LOGIN.getKey());
			pd.setLearnSource(date);
			EntityWrapper<PointDetails> wrapper = new EntityWrapper<PointDetails>(pd);
			int cnt = pointDetailsService.selectCount(wrapper);
			log.info("登录 赠送积分判断是否为首次登录:{}", cnt);
			if(cnt <= 0) {
				return new PointTypeStrategy(PointTypeEnum.LOGIN, date, PointTypeEnum.LOGIN.getPoint());
			}
		}
		if(type.equals(PointTypeEnum.READ.getKey())){
			int cnt = pointDetailsMapper.getCountTodayByLearnType(accountId, type);
			log.info("阅读文章 赠送积分判断当日是否超过次数:{}", cnt);
			if(cnt < 5){
				return new PointTypeStrategy(PointTypeEnum.READ, sourceId, PointTypeEnum.READ.getPoint());
			}
		}
		if(type.equals(PointTypeEnum.PUBLICATION.getKey())){
			PointDetails pd = new PointDetails();
			pd.setAccountId(accountId);
			pd.setLearnType(PointTypeEnum.PUBLICATION.getKey());
			pd.setLearnSource(sourceId);
			EntityWrapper<PointDetails> wrapper = new EntityWrapper<PointDetails>(pd);
			int cnt = pointDetailsService.selectCount(wrapper);
			log.info("投稿发布 赠送积分判断此投稿是不是首次发布:{}", cnt);
			if(cnt <= 0) {
				return new PointTypeStrategy(PointTypeEnum.PUBLICATION, sourceId, PointTypeEnum.PUBLICATION.getPoint());
			}
			return null;
		}
		if(type.equals(PointTypeEnum.COMMENT.getKey())){
			PointDetails pd = new PointDetails();
			pd.setAccountId(accountId);
			pd.setLearnType(PointTypeEnum.COMMENT.getKey());
			pd.setLearnSource(sourceId);
			EntityWrapper<PointDetails> wrapper = new EntityWrapper<PointDetails>(pd);
			int cnt = pointDetailsService.selectCount(wrapper);
			log.info("评论上架 赠送积分判断此活动是不是首次评论上架:{}", cnt);
			if(cnt <= 0) {
				return new PointTypeStrategy(PointTypeEnum.COMMENT, sourceId, PointTypeEnum.COMMENT.getPoint());
			}
			return null;
		}
		if(type.equals(PointTypeEnum.ACITVITY.getKey())){
			PointDetails pd = new PointDetails();
			pd.setAccountId(accountId);
			pd.setLearnType(PointTypeEnum.ACITVITY.getKey());
			pd.setLearnSource(sourceId);
			EntityWrapper<PointDetails> wrapper = new EntityWrapper<PointDetails>(pd);
			int cnt = pointDetailsService.selectCount(wrapper);
			log.info("活动报名 赠送积分判断此活动是不是首次参加:{}", cnt);
			if(cnt <= 0) {
				return new PointTypeStrategy(PointTypeEnum.ACITVITY, sourceId, PointTypeEnum.ACITVITY.getPoint());
			}
			return null;
		}
		if(type.equals(PointTypeEnum.EXAM.getKey())){
			int score = Double.valueOf(sourceId.split("\\|")[1]).intValue()/ 10;
			sourceId = sourceId.split("\\|")[0];
			PointDetails pd = new PointDetails();
			pd.setAccountId(accountId);
			pd.setLearnType(PointTypeEnum.EXAM.getKey());
			pd.setLearnSource(sourceId);
			EntityWrapper<PointDetails> wrapper = new EntityWrapper<PointDetails>(pd);
			int cnt = pointDetailsService.selectCount(wrapper);
			log.info("考试结束 赠送积分判断此考试是否赠送过积分:{}", cnt);
			if(cnt <= 0) {
				return new PointTypeStrategy(PointTypeEnum.EXAM, sourceId, score);
			}
			return null;
		}
		return null;
	}

	public boolean exchange(Long companyId, Long orgId, Long siteId, Long accountId, String accountName, String productIds){
		log.info("积分兑换入参 accountId:{}, productIds:{}", accountId, productIds);
		//查询本次积分兑换需要扣减的积分
		Integer point = pointProductMapper.getSumPointByProductId(Arrays.asList(productIds.split(",")));
		log.info("积分兑换扣减积分 point:{}", point);
		PointUser pu = new PointUser();
		pu.setUserId(accountId);
		EntityWrapper<PointUser> wrapper = new EntityWrapper<PointUser>(pu);
		PointUser pointUser = pointUserService.selectOne(wrapper);
		if(pointUser == null) {
			return false;
		}
		PointDetails pd = new PointDetails();
		pd.setId(idGenerator.generate());
		pd.setTime(new Date());
		pd.setChangeBefore(pointUser.getTotalPoint());
		pd.setPoint(point * -1);
		pd.setChangeAfter(pointUser.getTotalPoint() - point);
		pd.setMultiple(1);
		pd.setAccountId(accountId);
		pd.setLearnName(PointTypeEnum.EXCHANGE.getDesc());
		pd.setLearnSource(productIds);
		pd.setLearnSourceId(0L);
		pd.setLearnType(PointTypeEnum.EXCHANGE.getKey());
		pd.setState(1);
		pd.setCreateById(accountId);
		pd.setCreateByName(accountName);
		pd.setCreateTime(new Date());
		pd.setOrgId(orgId);
		pd.setCompanyId(companyId);
		pd.setSiteId(siteId);
		pointDetailsService.insert(pd);
		PointUserExchangeVO pointUserExchangeVO = new PointUserExchangeVO();
		pointUserExchangeVO.setUserId(accountId);
		pointUserExchangeVO.setPoint(point);
		int ret = pointUserMapper.updateUserPointExchange(pointUserExchangeVO);
		return ret > 0;
	}

	public Page<PointDetailListVO> getPointDetailsList(Long accountId, Integer pageNo, Integer pageSize) {
		Page<PointDetailListVO> page = new Page<>(pageNo, pageSize);
		log.info("积分明细入参 accountId:{}, pageNo:{}, pageSize:{}", accountId, pageNo, pageSize);
		page.setRecords(pointDetailsService.getPointDetailsList(page, accountId));
		for(PointDetailListVO vo : page.getRecords()) {
			if("point_exchange".equals(vo.getLearnType())) {
				String productNames = pointProductMapper.getProductNamesByProductIds(
						Arrays.asList(vo.getLearnSource().split(",")));
				log.info("积分明细查询商品名称:{}", productNames);
				String learnName = vo.getLearnName() + " 兑换商品:" + productNames;
				vo.setLearnName(learnName);
			}
		}
		return page;
	}

}
