package com.yizhi.site.application.controller;


import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.plugins.Page;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.site.application.domain.InternationalKey;
import com.yizhi.site.application.domain.SiteDic;
import com.yizhi.site.application.mapper.InternationalMapper;
import com.yizhi.site.application.service.IInternationalKeyService;
import com.yizhi.site.application.service.IInternationalService;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.application.orm.util.DomainConverter;
import com.yizhi.application.orm.util.QueryUtil;
import com.yizhi.site.application.service.ISiteDicService;
import com.yizhi.core.application.context.TaskContext;
import com.yizhi.core.application.file.constant.FileConstant;
import com.yizhi.core.application.file.domain.FileInfo;
import com.yizhi.core.application.file.domain.Template;
import com.yizhi.core.application.file.domain.TemplateColumn;
import com.yizhi.core.application.file.exp.ExcelExportor;
import com.yizhi.core.application.file.imp.AbstractImporter;
import com.yizhi.core.application.file.imp.ImportCallBack;
import com.yizhi.core.application.file.util.OssUpload;
import com.yizhi.core.application.log.LogEventPublisher;
import com.yizhi.core.application.log.TaskLogEvent;
import com.yizhi.core.application.task.TaskExecutor;
import com.yizhi.core.application.task.TaskHandler;
import com.yizhi.site.application.vo.site.InternationalFileUpVO;
import com.yizhi.site.application.vo.site.InternationalImport;
import com.yizhi.site.application.vo.site.InternationalVO;
import com.yizhi.system.application.model.ImportModel;
import com.yizhi.util.application.clazz.ClassUtil;
import com.yizhi.util.application.date.DateUtil;
import io.swagger.annotations.ApiOperation;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;

/**
 * <p>
 * APP版本控制表 前端控制器
 * </p>
 *
 * @author lingye
 * @since 2020-8-20
 */
@RestController
@RequestMapping("/international")
@Log4j2
public class InternationalController {

    @Autowired
    DomainConverter converter;
    @Autowired
    LogEventPublisher publisher;
    @Autowired
    TaskExecutor taskExecutor;
    @Autowired
    RedisCache redisCache;
    @Autowired
    private IdGenerator idGenerator;
    @Autowired
    private IInternationalService internationalService;
    @Autowired
    private ISiteDicService siteDicService;
    @Autowired
    private InternationalMapper internationalMapper;
    @Autowired
    private IInternationalKeyService internationalKeyService;

    private  final Long ZH_LANGUAGE_ID = 132L;

    @ApiOperation(value = "国际化导入", notes = "国际化导入")
    @PostMapping(value = "/import")
    public String importInternational(@RequestBody InternationalFileUpVO model) {

        log.info("导入文件信息：{}", JSON.toJSONString(model));
        String taskLogSerialNo = "INTERNATIONAL-" + DateUtil.getYMDHMSS();
        try {
            Long taskId = idGenerator.generate();
            RequestContext context = ContextHolder.get();
            Long companyId = context.getCompanyId();
            Long siteId = context.getSiteId();
            Long accountId = context.getAccountId();
            SiteDic siteDic = new SiteDic();
            Long languageId = model.getLanguageId();
            siteDic.setId(languageId);
            siteDic = siteDicService.selectSiteDic(siteDic);

            String taskName = "导入翻译-" + siteDic.getName();

            TaskContext taskContext = new TaskContext(taskId, taskLogSerialNo, taskName, accountId, new Date(), siteId,
                    companyId);
            ImportModel importModel = (ImportModel) converter.toDO(context, ImportModel.class);
            importModel.setTaskId(taskId);
            ImportCallBack callBack = new ImportCallBack() {
                @Override
                public Boolean exec(TaskHandler taskHandler, List dataObjs) {
                    importModel.setData(dataObjs);
                    List<InternationalImport> models = internationalService.importInternationalList(importModel, context, languageId, model.getType());
                    ExcelForAccountImportExporter<InternationalImport> exporter = new ExcelForAccountImportExporter<>(
                            createExpTemplate(), models);
                    exporter.bindContext(taskContext);
                    exporter.bindPublisher(publisher);
                    exporter.handle();
                    return true;
                }
            };
            AbstractImporter importer = AbstractImporter.build(model.getFileUrl(), null, model.getFileName(), callBack,
                    createImpTemplate(), InternationalImport.class);
            importer.bindContext(taskContext);
            importer.bindPublisher(publisher);
            taskExecutor.asynExecute(importer);
        } catch (Exception e) {
            log.error("导入发生异常：{}", e);
        }
        return taskLogSerialNo;
    }

    private Template createExpTemplate() {
        Template template = createImpTemplate();
        List<TemplateColumn> columns = template.getColumns();
        columns.add(new TemplateColumn("result", "结果", 5));
        template.setColumns(columns);
        template.setBrowserExport(false);
        return template;
    }

    private Template createImpTemplate() {
        Template template = new Template();
        List<TemplateColumn> columns = new ArrayList<>();
        columns.add(new TemplateColumn("key", "key值", 1));
        columns.add(new TemplateColumn("preTranslation", "中文字段", 2));
        columns.add(new TemplateColumn("postTranslation", "翻译", 3));
        columns.add(new TemplateColumn("translationRemark", "备注", 4));
        template.setColumns(columns);
        return template;
    }

    /**
     * 查询国际化的管理端的列表
     *
     * @param type
     * @param languageId
     * @param keywords
     * @param state
     * @return
     */
    @GetMapping("/list")
    public List<InternationalVO> findInternationalList(
            @RequestParam(name = "type", required = false) Integer type,
            @RequestParam(name = "languageId", required = false) Long languageId,
            @RequestParam(name = "keywords", required = false) String keywords,
            @RequestParam(name = "state", required = false) Integer state
    ) {

        return internationalService.findInternationalList(type, languageId, keywords, state);

    }

    /**
     * 分页查询页面
     *
     * @param type
     * @param languageId
     * @param keywords
     * @param state
     * @param pageNo
     * @param pageSize
     * @return
     */
    @GetMapping("/page")
    public Page<InternationalVO> findInternationalPage(
            @RequestParam(name = "type", required = false) Integer type,
            @RequestParam(name = "languageId", required = false) Long languageId,
            @RequestParam(name = "keywords", required = false) String keywords,
            @RequestParam(name = "state", required = false) Integer state,
            @RequestParam(name = "pageNo", required = false) Integer pageNo,
            @RequestParam(name = "pageSize", required = false) Integer pageSize
    ) {

        return internationalService.findInternationalPage(type, languageId, keywords, state, pageNo, pageSize);

    }

    /**
     * 修改国际化实体
     *
     * @param internationalVO
     * @return
     */
    @PostMapping("/update")
    public InternationalVO updateInternational(@RequestBody InternationalVO internationalVO) {

        return internationalService.updateInternational(internationalVO);

    }

    /**
     * 初始化全部语言
     */
    @GetMapping(value = "/language/init")
    public void initLanguage() {
        List<SiteDic> list = siteDicService.getSiteDicListByCode("language");
        List<Integer> terminalTypes = Arrays.asList(0, 1, 2);
        for (Integer a : terminalTypes) {
            Map<Long, InternationalKey> keyMap = internationalKeyService.getKeyIdMap(a, null);
            if (keyMap == null) {
                log.info("查询key数据为空:{}", JSON.toJSON(keyMap));
                continue;
            }
            for (SiteDic dic : list) {
                //设置缓存
                setCache(a,dic.getId(),dic.getCode());
            }
        }
    }

    /**
     * @param type         终端类型
     * @param languageCode
     * @return
     */
    @GetMapping("/language/get")
    public HashMap<String, String> getLanguage(@RequestParam(name = "type") Integer type,
                                               @RequestParam(name = "languageCode") String languageCode) {
        HashMap<String, String> map = null;
        if (StringUtils.isBlank(languageCode) || type == null) {
            log.info("获取国际化数据，" + "参数为空！！");
            return map;
        }
        String key = CacheNamespace.SITE_LANGUAGE_INTERNATIONAL_WORD + type;
        Object hGet = redisCache.hget(key, languageCode);
        log.info("返回的的信息：{}", hGet);
        if (hGet != null) {
            map = JSON.parseObject(hGet.toString(), HashMap.class);
        } else {
            map = getDataMap(type, languageCode);
        }
        return map;
    }


    /**
     * 获取数据
     *
     * @return
     */
    private HashMap<String, String> getDataMap(Integer type, String languageCode) {
        HashMap<String, String> getDataMap = null;
        SiteDic siteDic = new SiteDic();
        siteDic.setCode(languageCode.trim());
        siteDic = siteDicService.selectOne(QueryUtil.condition(siteDic));
        log.info("当前的语言：{}", JSON.toJSON(siteDic));
        if (siteDic != null) {
            //设置缓存
            getDataMap = setCache(type,siteDic.getId(),languageCode);
        }
        return getDataMap;
    }
    /**
     * 设置缓存
     * @param type
     * @param languageId
     * @param code  语言code
     * @return
     */
    private HashMap<String, String> setCache(Integer type, Long languageId,String code) {
        HashMap<String, String> map = null;
        List<InternationalVO> internationalVOS = internationalService.findInternationalList(type, languageId, null, null);
        if (CollectionUtils.isNotEmpty(internationalVOS)) {
            String key = CacheNamespace.SITE_LANGUAGE_INTERNATIONAL_WORD + type;
            map = new HashMap<>(internationalVOS.size());
            if (CollectionUtils.isNotEmpty(internationalVOS)) {
                String translated = "";
                for (InternationalVO a : internationalVOS) {
                    if (a.getKey() == null) {
                        continue;
                    }
                    if (StringUtils.isEmpty(a.getPostTranslation())) {
                        translated = a.getPreTranslation();
                    } else {
                        translated = a.getPostTranslation();
                    }
                    map.put(a.getKey(), translated);
                }
                boolean hSet = redisCache.hset(key, code, JSON.toJSONString(map));
                log.info("hset终端：" + type + "语言code:" + code + "存储结果：{}", hSet);
            }
        }
        return map;
    }

    class ExcelForAccountImportExporter<T> extends ExcelExportor<T> {

        public ExcelForAccountImportExporter(Template template, List data) {
            super(template, data);
        }

        @Override
        public FileInfo render() {
            FileInfo fileInfo = new FileInfo();
            fileInfo.setFileType(template.getFileType());
            // 创建HSSFWorkbook对象(excel的文档对象)
            HSSFWorkbook wb = new HSSFWorkbook();
            // 建立新的sheet对象（excel的表单）
            HSSFSheet sheet = wb.createSheet();
            // 在sheet里创建第一行，标题行
            HSSFRow row1 = sheet.createRow((int) 0);
            // 标题目录
            List<TemplateColumn> columns = template.getColumns();
            for (int i = 0; i < columns.size(); i++) {
                Cell cell = row1.createCell(i);
                cell.setCellValue(columns.get(i).getColumnName().toString());
            }
            // 行数据
            HSSFRow row2 = null;
            int passCount = 0;
            int failCount = 0;
            if (CollectionUtils.isNotEmpty(data)) {
                for (int j = 0; j < data.size(); j++) {
                    T t = data.get(j);
                    row2 = sheet.createRow((int) j + 1);
                    for (int m = 0; m < columns.size(); m++) {
                        TemplateColumn templateColumn = columns.get(m);
                        String cellValue = ClassUtil.invokeMethod(t, templateColumn.getColumn()).toString();
                        row2.createCell(m).setCellValue(cellValue);
                    }
                    if (StringUtils.isNotEmpty(
                            ClassUtil.invokeMethod(t, columns.get(columns.size() - 1).getColumn()).toString())) {
                        failCount++;
                    } else {
                        passCount++;
                    }
                }
            }
            String fileName = template.getFileName();
            if (StringUtils.isEmpty(fileName)) {
                fileName = "导出文件".concat(String.valueOf(System.currentTimeMillis())).concat(".xls");
            }
            fileInfo.setFileName(fileName);
            if (template.getBrowserExport()) {
                response.setHeader("Content-Disposition", "attachment; filename=" + fileName);
                response.setContentType("application/vnd.ms-excel");
                OutputStream output = null;
                try {
                    output = response.getOutputStream();
                    wb.write(output);
                    output.flush();
                    publishTaskEvent(TaskLogEvent.success(taskContext, "导入成功", null));
                } catch (IOException e) {
                    publishTaskEvent(TaskLogEvent.fail(taskContext, "文件导入异常:" + e.getMessage()));
                    log.error("文件导入异常:{}", e);
                } finally {
                    try {
                        output.close();
                    } catch (IOException e) {
                        log.error("文件导入关闭流异常:{}", e);
                    }
                }
            } else {
                String path = new StringBuffer().append(FileConstant.SAVE_PATH).append(File.separator).append(fileName)
                        .toString();
                //文件保存位置
                File saveDir = new File(FileConstant.SAVE_PATH);
                if (!saveDir.exists()) {
                    saveDir.mkdir();
                }
                File file = new File(saveDir + File.separator + fileName);
                try {
                    FileOutputStream os = new FileOutputStream(file);
                    wb.write(os);
                    os.close();
                    // 阿里云返回url
                    String upLoadUrl = OssUpload.upload(path, fileName);// path, fileName
                    publishTaskEvent(TaskLogEvent.success(taskContext, "成功导入" + passCount + "条，失败" + failCount + "条",
                            upLoadUrl));
                    file.delete();
                } catch (Exception e) {
                    publishTaskEvent(TaskLogEvent.fail(taskContext, "写入过程中发生错误:" + e.getMessage()));
                    log.error("写入过程中发生错误{}", e);
                }
            }
            return fileInfo;
        }
    }
}
