package com.yizhi.training.application.service.impl;

import com.baomidou.mybatisplus.mapper.EntityWrapper;
import com.yizhi.application.orm.id.IdGenerator;
import com.yizhi.core.application.cache.CacheNamespace;
import com.yizhi.core.application.cache.RedisCache;
import com.yizhi.core.application.context.ContextHolder;
import com.yizhi.core.application.context.RequestContext;
import com.yizhi.core.application.task.AbstractTaskHandler;
import com.yizhi.core.application.task.TaskExecutor;
import com.yizhi.research.application.feign.ResearchClient;
import com.yizhi.research.application.vo.api.CheckResearchStateVo;
import com.yizhi.training.application.constant.ProjectConstant;
import com.yizhi.training.application.domain.*;
import com.yizhi.training.application.mapper.*;
import com.yizhi.training.application.util.ArrayUtil;
import com.yizhi.training.application.vo.api.TrainingProjectContentActivityVo;
import com.yizhi.training.application.vo.api.TrainingProjectContentPlanVo;
import com.yizhi.training.application.vo.api.TrainingProjectContentVo;
import com.yizhi.util.application.constant.TpActivityType;
import com.yizhi.util.application.enums.i18n.Constants;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.NumberFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description 学员查看培训项目内容的组装
 * @ClassName TpContentStudentStatusServiceUsing
 * @Author shengchenglong
 * @Date 2019-04-01 13:49
 * @Version 1.0
 **/
@Service
@Slf4j
public class TpContentStudentStatusServiceUsing {

    @Autowired
    private IdGenerator idGenerator;
    @Autowired
    private TpPlanMapper tpPlanMapper;
    @Autowired
    private TpPlanActivityMapper tpPlanActivityMapper;
    @Autowired
    private TpPlanConditionPostMapper tpPlanConditionPostMapper;
    @Autowired
    private TpPlanConditionPreMapper tpPlanConditionPreMapper;
    @Autowired
    private TpStudentPlanRecordMapper tpStudentPlanRecordMapper;
    @Autowired
    private TpStudentProjectRecordMapper tpStudentProjectRecordMapper;
    @Autowired
    private RedisCache redisCache;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ResearchClient researchClient;
    //不参与活动的计算  类型有 3-直播 6-证书 7-外链 14-帖子
    private List<Integer> NOT_CALCULATE = Arrays.asList(6);

    /**
     * 查询培训项目内容
     *
     * @param project
     * @param accountId
     * @return
     */
    public TrainingProjectContentVo getTpContent(TrainingProject project, Long accountId) {
        Date date = new Date();
        TrainingProjectContentVo trainingProjectVo;
        // 要返回的vo
        trainingProjectVo = new TrainingProjectContentVo();
        List<TrainingProjectContentPlanVo> planVos = new ArrayList<>();

        // 2. 查出培训计划
        List<TpPlan> plans = null;
        TpPlan planExample = new TpPlan();
        planExample.setTrainingProjectId(project.getId());
        planExample.setDeleted(ProjectConstant.DELETED_NO);
        EntityWrapper<TpPlan> planEW = new EntityWrapper<>(planExample);
        planEW.orderBy("sort");
        plans = tpPlanMapper.selectList(planEW);

        // 3. 查出培训活动（首先处理计划状态）
        if (!CollectionUtils.isEmpty(plans)) {
            buildPlanActivity(project, accountId, date, plans, planVos);
        }

        trainingProjectVo.setPlans(planVos);
        return trainingProjectVo;
    }

    /**
     * 组装计划和活动的入口方法
     *
     * @param project
     * @param accountId
     * @param now
     * @param plans
     * @param planVos
     */
    private void buildPlanActivity(TrainingProject project, Long accountId, Date now, List<TpPlan> plans, List<TrainingProjectContentPlanVo> planVos) {
        TpPlanConditionPost conditionPost = new TpPlanConditionPost();
        conditionPost.setDeleted(ProjectConstant.DELETED_NO);
        conditionPost.setTrainingProjectId(project.getId());
        List<TpPlanConditionPost> conditionPosts = tpPlanConditionPostMapper.selectList(new EntityWrapper<>(conditionPost));
        List<TpPlanConditionPost> conditionPostsTemp;

        TpPlanConditionPre conditionPre = new TpPlanConditionPre();
        conditionPre.setDeleted(ProjectConstant.DELETED_NO);
        conditionPre.setTrainingProjectId(project.getId());
        List<TpPlanConditionPre> conditionPres = tpPlanConditionPreMapper.selectList(new EntityWrapper<>(conditionPre));
        List<TpPlanConditionPre> conditionPresTemp;

        String accountIdStr = String.valueOf(accountId);
        String keyForActivityFinished = CacheNamespace.TP_ACTIVITY_FINISHED.concat(accountIdStr);
        String keyForActivityUnFinished = CacheNamespace.TP_ACTIVITY_UNFINISHED.concat(accountIdStr);
        String keyForActivityClicked = CacheNamespace.TP_ACTIVITY_CLICKED.concat(accountIdStr);

        String keyForPlanFinished = CacheNamespace.TP_PLAN_FINISHED.concat(accountIdStr);
        //项目完成
        String keyForProjectFinished = CacheNamespace.TP_TRAININGPROJECT_FINISHED.concat(accountIdStr);


        // 当前时间在计划时间范围内
        boolean inplanTime;
        // 当前时间在计划时间开始时间后
        boolean inPlanStartTime;
        // 当前时间在计划结束时间之前
        boolean inPlanEndTime;

        // 计划 id ：计划 vo map
        RequestContext context= ContextHolder.get();
        Map<Long, TrainingProjectContentPlanVo> map = new TreeMap<>();
        for (TpPlan plan : plans) {

            TpPlanActivity activityExample = new TpPlanActivity();
            activityExample.setTrainingProjectId(project.getId());
            activityExample.setTpPlanId(plan.getId());
            TpPlanActivity planActivity = tpPlanActivityMapper.selectOne(activityExample);
            if(planActivity != null) {
                CheckResearchStateVo checkResearchStateVo = new CheckResearchStateVo();
                checkResearchStateVo.setContext(context);
                checkResearchStateVo.setResearchId(planActivity.getRelationId());
                Integer count = researchClient.checkTPlanResearchState(checkResearchStateVo);
                log.info("查询活动id:{}中,投票id:{}中的投票状态count:{}", planActivity.getTrainingProjectId(), planActivity.getRelationId(), count);
                if(count.intValue() <= 0) {
                    continue;
                }
            }

            conditionPost.setTpPlanId(plan.getId());
            conditionPre.setPlanId(plan.getId());

            inPlanStartTime = DateUtils.truncatedCompareTo(now, plan.getStartTime(), Calendar.DATE) >= 0;
            inPlanEndTime = DateUtils.truncatedCompareTo(plan.getEndTime(), now, Calendar.DATE) >= 0;
            inplanTime = inPlanStartTime && inPlanEndTime;

            TrainingProjectContentPlanVo planVo = new TrainingProjectContentPlanVo();
            planVo.setId(plan.getId());
            planVo.setName(plan.getName());
            planVo.setStartTime(plan.getStartTime());
            planVo.setEndTime(plan.getEndTime());
            planVo.setInPlanStartTime(inPlanStartTime);
            planVo.setInPlanEndTime(inPlanEndTime);
            planVo.setInTime(inplanTime);

            // 组装计划完成条件
            conditionPostsTemp = conditionPosts.stream().filter(item -> item.getTpPlanId().equals(plan.getId())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(conditionPostsTemp)) {
                for (TpPlanConditionPost post : conditionPostsTemp) {
                    // 指定完成数
                    if (post.getType().equals(0)) {
                        planVo.setToFinishedActivityNum(post.getNum());
                        break;
                    }
                    // 指定完成活动
                    else {
                        planVo.getToFinishedActivityIds().add(post.getTpPlanActivityId());
                    }
                }
            }
            // 组装计划前置计划
            conditionPresTemp = conditionPres.stream().filter(item -> item.getPlanId().equals(plan.getId())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(conditionPresTemp)) {
                for (TpPlanConditionPre pre : conditionPresTemp) {
                    planVo.getConditionPrePlanIds().add(pre.getPrePlanId());
                }
            }
            planVos.add(planVo);
            map.put(plan.getId(), planVo);
        }

        // 2. 查出所有活动，组装到相应的计划 vo 中
        TpPlanActivity activityExample = new TpPlanActivity();
        activityExample.setCompanyId(project.getCompanyId());
        activityExample.setSiteId(project.getSiteId());
        activityExample.setDeleted(ProjectConstant.DELETED_NO);
        activityExample.setTrainingProjectId(project.getId());
        EntityWrapper<TpPlanActivity> activityEW = new EntityWrapper<>(activityExample);
        activityEW.orderBy("sort");
        List<TpPlanActivity> allActivities = tpPlanActivityMapper.selectList(activityEW);

        if (!CollectionUtils.isEmpty(allActivities)) {
            for (TpPlanActivity activity : allActivities) {

                CheckResearchStateVo checkResearchStateVo = new CheckResearchStateVo();
                checkResearchStateVo.setContext(context);
                checkResearchStateVo.setResearchId(activity.getRelationId());
                Integer count = researchClient.checkTPlanResearchState(checkResearchStateVo);
                log.info("查询活动id:{}中,投票id:{}中的投票状态count:{}", activity.getTrainingProjectId(), activity.getRelationId(), count);
                if(count.intValue() <= 0) {
                    continue;
                }

                TrainingProjectContentActivityVo activityVo = new TrainingProjectContentActivityVo();
                activityVo.setType(activity.getType());
                activityVo.setRelationId(activity.getRelationId());
                activityVo.setFinished(redisCache.hExisted(keyForActivityFinished, String.valueOf(activity.getRelationId())));
                // 如果已经完成
                if (activityVo.getFinished()) {
                    map.get(activity.getTpPlanId()).getFinishedActivityIds().add(activity.getId());
                }
                // 否则，未完成：未通过、未导入
                else {
                    // 考试、线下课程 有未通过状态
                    if (activityVo.getType().equals(TpActivityType.TYPE_EXAM) || activityVo.getType().equals(TpActivityType.TYPE_OFFLINE_COURSE)) {
                        activityVo.setUnPassed(redisCache.hExisted(keyForActivityUnFinished, String.valueOf(activity.getRelationId())));
                    }
                    // 线下课程有 未导入状态
                    if (activityVo.getType().equals(TpActivityType.TYPE_OFFLINE_COURSE)) {
                        // 既没有 已完成，也没有 未完成
                        if (!activityVo.getFinished() && !activityVo.getUnPassed()) {
                            activityVo.setUnImported(true);
                        }
                    }
                }

                activityVo.setClicked(redisCache.hExisted(keyForActivityClicked, String.valueOf(activity.getId()).concat("_").concat(String.valueOf(activity.getTrainingProjectId()))));
                activityVo.setId(activity.getId());
                activityVo.setName(activity.getName());
                activityVo.setUrl(activity.getAddress());
                activityVo.setSort(activity.getSort());
                map.get(activity.getTpPlanId()).getActivities().add(activityVo);
                map.get(activity.getTpPlanId()).getAllActivityIds().add(activityVo.getId());
                if (NOT_CALCULATE.contains(activityVo.getType())) {
                    //有些活动类型不参与计算，比如证书
                    continue;
                }
                map.get(activity.getTpPlanId()).getNeedCalculateActivityIds().add(activityVo.getId());
            }
        }

        // 查询计划完成记录
        Set<Long> finishPlanIds = new HashSet<>();
        for (Map.Entry<Long, TrainingProjectContentPlanVo> entry : map.entrySet()) {
            buildPlanStatuss(project, accountId, now, entry.getValue(), keyForPlanFinished);
            if (entry.getValue().getFinished()){
                finishPlanIds.add(entry.getValue().getId());
            }
        }
        //判断当前项目没完成
        if (!redisCache.hExisted(keyForProjectFinished,project.getId().toString())){
            if (finishPlanIds.size() == plans.size()) {
                doInsertProjectRecord(accountId, new Date(), project, keyForProjectFinished);
            }
        }
    }

    /**
     * 组建计划状态
     *
     * @param project
     * @param accountId
     * @param now
     * @param planVo
     * @param keyForPlanFinished
     */
    private void buildPlanStatuss(TrainingProject project, Long accountId, Date now, TrainingProjectContentPlanVo planVo, String keyForPlanFinished) {
        String status = null;
        String percent = null;
        boolean planFinished = false;
        boolean planStarted = false;

        boolean processPlanFinished = false;
        List<Long> conditionPrePlanIds;

        // 如果当前时间在计划开始时间之前
        if (!planVo.getInPlanStartTime()) {
            status = "未开始";
            percent = "";
            planFinished = false;
            planStarted = false;
        }
        // 当前时间在 计划时间之内
        else if (planVo.getInTime()) {
            // 如果已经完成
            if (redisCache.hExisted(keyForPlanFinished, String.valueOf(planVo.getId()))) {
                status = "已完成";
                percent = "100%";
                planFinished = true;
                planStarted = true;
            }
            // 如果没有完成
            else {
                // 查看是否有前置计划
                conditionPrePlanIds = planVo.getConditionPrePlanIds();
                // 没有前置计划 或者 前置计划已完成
                if (CollectionUtils.isEmpty(conditionPrePlanIds)
                        || (redisCache.hAllExisted(keyForPlanFinished, ArrayUtil.forStringArray(conditionPrePlanIds)))) {
                    // 查看活动是否全部完成
                    // 如果是指定完成活动都已经完成
                    if (!CollectionUtils.isEmpty(planVo.getToFinishedActivityIds())) {
                        // 有已完成的并且完成所有的应完成
                        if (!CollectionUtils.isEmpty(planVo.getFinishedActivityIds())
                                && planVo.getFinishedActivityIds().containsAll(planVo.getToFinishedActivityIds())) {
                            processPlanFinished = true;
                        }
                    }
                    // 如果是指定完成活动数达标
                    else if (planVo.getToFinishedActivityNum() > 0) {
                        if (planVo.getToFinishedActivityNum() <= planVo.getFinishedActivityIds().size()) {
                            processPlanFinished = true;
                        }
                    }
                    // 未指定完成条件，如果总活动数 == 已经完成活动数
                    else {
                        if (planVo.getAllActivityIds().size() == planVo.getFinishedActivityIds().size()) {
                            processPlanFinished = true;
                        }
                    }

                    if (processPlanFinished) {
                        status = "已完成";
                        percent = "100%";
                        planStarted = true;
                        planFinished = true;
                        doInsertPlanRecord(accountId, now, project, planVo, keyForPlanFinished);
                    } else {
                        status = "进行中";
                        percent = buildPercent(planVo);
                        planStarted = true;
                        planFinished = false;
                    }
                }
                // 前置计划未完成
                else {
                    status = "未开始";
                    percent = "";
                    planStarted = false;
                    planFinished = false;
                    Set<Object> finishedPlanIds = redisCache.hKeys(keyForPlanFinished);
                    List<Long> finishedPlanIdList = new ArrayList<>();
                    if (finishedPlanIds == null) {
                        finishedPlanIds.forEach(item -> finishedPlanIdList.add((Long) item));
                    }
                    planVo.setPrePlanNames(tpPlanMapper.getNames(planVo.getConditionPrePlanIds(), finishedPlanIdList));
                }
            }
        }
        // 当前时间在结束时间之后
        else if (!planVo.getInPlanEndTime()) {
            // 如果已经完成
            if (redisCache.hExisted(keyForPlanFinished, String.valueOf(planVo.getId()))) {
                status = "已完成";
                percent = "100%";
                planStarted = true;
                planFinished = true;
            }
            // 如果没有完成
            else {
                status = "已过期";
                percent = buildPercent(planVo);
                planStarted = false;
                planFinished = false;
            }
        }
        planVo.setStatus(status);
        planVo.setFinished(planFinished);
        planVo.setStarted(planStarted);
        planVo.setPercentageOfCompletion(percent);

        buildPlanActivityStatuss(planVo);
    }

    /**
     * 组建活动状态
     *
     * @param planVo
     */
    private void buildPlanActivityStatuss(TrainingProjectContentPlanVo planVo) {
        List<TrainingProjectContentActivityVo> activityVos = planVo.getActivities();
        if (!CollectionUtils.isEmpty(activityVos)) {
            String status = null;
            boolean started = false;
            boolean expired = false;

            for (TrainingProjectContentActivityVo activityVo : activityVos) {
                // 计划可以开始
                if (planVo.getStarted()) {
                    // 活动已完成
                    if (activityVo.getFinished()) {
                        status = getFinishedStatus(activityVo.getType());
                        started = true;
                        expired = false;
                    }
                    // 活动未完成
                    else {
                        status = getUnderwayStatus(activityVo);
                        started = true;
                        expired = false;
                    }
                }
                // 计划未到开始时间 || 计划在有效时间范围内(前置计划未完成)
                else if (!planVo.getInPlanStartTime() || planVo.getInTime()) {
                    status = "";
                    started = false;
                    expired = false;
                }
                // 计划已过期
                else {
                    // 活动已完成
                    if (activityVo.getFinished()) {
                        status = getFinishedStatus(activityVo.getType());
                        started = true;
                        expired = false;
                    }
                    // 活动未完成
                    else {
                        status = getExpiredStatus(activityVo.getType());
                        started = false;
                        expired = true;
                    }
                }

                activityVo.setStarted(started);
                activityVo.setStatus(status);
                activityVo.setExpired(expired);
            }
        }
    }

    /**
     * 插入计划记录
     *
     * @param accountId
     * @param date
     * @param project
     * @param planVo
     */
    private void doInsertPlanRecord(long accountId, Date date, TrainingProject project, TrainingProjectContentPlanVo planVo, String keyForPlanFinished) {
        TpStudentPlanRecord planRecord = new TpStudentPlanRecord();
        planRecord.setId(idGenerator.generate());
        planRecord.setAccountId(accountId);
        planRecord.setFinishDate(date);
        planRecord.setSiteId(project.getSiteId());
        planRecord.setFinished(1);
        planRecord.setTpPlanId(planVo.getId());
        planRecord.setTrainingProjectId(project.getId());
        Integer result = tpStudentPlanRecordMapper.insert(planRecord);
        // 缓存完成记录
        if (result.equals(1)) {
            redisCache.hset(keyForPlanFinished, String.valueOf(planRecord.getTpPlanId()), String.valueOf(1));
        }
    }
    /**
     * 插入项目完成记录
     *
     * @param accountId
     * @param date
     * @param project
     */
    private void doInsertProjectRecord(long accountId, Date date, TrainingProject project, String keyForProjectFinished) {
        taskExecutor.asynExecute(new AbstractTaskHandler() {
            @Override
            public void handle() {
                TpStudentProjectRecord projectRecord = new TpStudentProjectRecord();
                projectRecord.setId(idGenerator.generate());
                projectRecord.setAccountId(accountId);
                projectRecord.setFinishDate(date);
                projectRecord.setSiteId(project.getSiteId());
                projectRecord.setFinished(1);
                projectRecord.setTrainingProjectId(project.getId());
                Integer result = tpStudentProjectRecordMapper.insert(projectRecord);
                // 缓存完成记录
                if (result.equals(1)) {
                    redisCache.hset(keyForProjectFinished, String.valueOf(projectRecord.getTrainingProjectId()), String.valueOf(1));
                }
            }
        });
    }

    private String buildPercent(TrainingProjectContentPlanVo planVo) {
        String percent = null;
        int finishedIdNum = 0;
        // 如果是指定完成活动数
        if (planVo.getToFinishedActivityNum() > 0) {
            //这里不需要对活动类型做判断，因为在保存时。已由前端做了筛选，只要是保存了的数据，默认为有效数据
            percent = getPercent(planVo.getFinishedActivityIds().size(), planVo.getToFinishedActivityNum());
        }
        // 如果是指定完成活动
        else if (!CollectionUtils.isEmpty(planVo.getToFinishedActivityIds())) {
            List<Long> finishedIds = planVo.getFinishedActivityIds();
            if (!CollectionUtils.isEmpty(finishedIds)) {
                for (Long id : finishedIds) {
                    if (planVo.getToFinishedActivityIds().contains(id)) {
                        finishedIdNum++;
                    }
                }
            }
            percent = getPercent(finishedIdNum, planVo.getToFinishedActivityIds().size());
        }
        // 如果没有完成条件
        else {
            percent = getPercent(planVo.getFinishedActivityIds().size(), planVo.getNeedCalculateActivityIds().size());
        }
        return percent;
    }

    private String getPercent(int x, int total) {
        // 创建一个数值格式化对象
        NumberFormat numberFormat = NumberFormat.getInstance();
        // 设置精确到小数点后2位
        numberFormat.setMaximumFractionDigits(0);
        String result = numberFormat.format((float) x / (float) total * 100);
        return result + "%";
    }

    /**
     * 获取已完成状态
     *
     * @param type
     * @return
     */
    private String getFinishedStatus(Integer type) {
        if (TpActivityType.TYPE_CERTIFICATE.equals(type)) {
            return "已获得";
        } else {
            return "已完成";
        }
    }

    /**
     * 获取进行中状态
     *
     * @param activityVo
     * @return
     */
    private String getUnderwayStatus(TrainingProjectContentActivityVo activityVo) {

        Integer type = activityVo.getType();

        if (activityVo.getUnPassed()) {
            if (activityVo.getType().equals(TpActivityType.TYPE_OFFLINE_COURSE)) {
                return "未完成";
            }
            if (activityVo.getType().equals(TpActivityType.TYPE_EXAM)) {
                return "进行中";
            }
        }

        if (activityVo.getUnImported()) {
            if (TpActivityType.TYPE_OFFLINE_COURSE.equals(type)) {
                return "未导入";
            }
        }

        if (TpActivityType.TYPE_CERTIFICATE.equals(type)) {
            return "未获得";
        }
        // 如果没有点击过，显示空白状态
        if (!activityVo.getClicked()) {
            return "";
        }

        if (TpActivityType.TYPE_LINK.equals(type) || TpActivityType.TYPE_LIVE.equals(type)||TpActivityType.TYPE_SELECTED_CASE.equals(type)) {
            return "";
        }
        if (TpActivityType.TYPE_OFFLINE_COURSE.equals(type)) {
            return "未完成";
        }
        return "进行中";
    }

    /**
     * 获取过期状态
     *
     * @param type
     * @return
     */
    private String getExpiredStatus(Integer type) {
        if (TpActivityType.TYPE_CERTIFICATE.equals(type)) {
            return "未获得";
        }
        return "未完成";
    }

}
